#include "daemon.h"

CallerXDaemon::CallerXDaemon(QObject *parent) :
    QObject(parent)
{
    readSettings();
    if (!autoStart && !isRunning){
        emit quit();
        return;
    }
}

CallerXDaemon::~CallerXDaemon()
{
    QDBusConnection::systemBus().disconnect(BUS_NAME, BUS_PATH, BUS_NAME, "Coming", this, SLOT(callScreen(const QDBusObjectPath&, const QString&)));
    delete fileWatcher;
    isRunning = false;
    saveSettings();
}

void CallerXDaemon::readSettings()
{
    QSettings settings("callerx", "callerx");
    autoStart = settings.value("auto-start", true).toBool();
    isRunning = settings.value("active", true).toBool();
    blockUnknown = settings.value("blockunknown", false).toBool();
    whitelistMode = settings.value("whitelistmode", false).toBool();
    whitelistPath = settings.value("whitelistpath", DEFAULT_WHITELIST).toString();
    blacklistPath = settings.value("blacklistpath", DEFAULT_BLACKLIST).toString();
    callHistory = settings.value("recentinboundcalls", QStringList()).toStringList();
    listPath = whitelistMode?whitelistPath:blacklistPath;
    activeTimer = false;
}

void CallerXDaemon::saveSettings()
{
    QSettings settings("callerx", "callerx");
    settings.setValue("app_version", APP_VERSION);
    settings.setValue("active", isRunning);
    settings.setValue("whitelistmode", whitelistMode);
    settings.setValue("whitelistpath", whitelistPath);
    settings.setValue("blacklistpath", blacklistPath);
    settings.setValue("blockunknown", blockUnknown);
}

void CallerXDaemon::Start()
{
    QTimer::singleShot(10000, this, SLOT(initialize()));
}

void CallerXDaemon::initialize()
{
    if (!QDBusConnection::systemBus().connect(BUS_NAME, BUS_PATH, BUS_NAME, "Coming", this, SLOT(callScreen(const QDBusObjectPath&, const QString&)))){
        emit quit();
        return;
    }
    if (!QDBusConnection::sessionBus().isConnected()) {
        emit quit();
        return;
    }
    if (!QDBusConnection::sessionBus().registerService("org.callerx.ifdbus")) {
        emit quit();
        return;
    }
    if (!QDBusConnection::sessionBus().registerObject("/org/callerx/ifdbus", this, QDBusConnection::ExportScriptableSlots)) {
        emit quit();
        return;
    }

    QFile fileList(listPath);
    if (!fileList.exists()) {
        if (!fileList.open(QIODevice::WriteOnly | QIODevice::Text)){
            return;
        }
        fileList.close();
    }

    fileWatcher = new QFileSystemWatcher();
    fileWatcher->addPath(listPath);
    QObject::connect(fileWatcher, SIGNAL(fileChanged(const QString&)), this, SLOT(updateList()));

    if (whitelistMode) {
        loadwhitelist();
    }
    else {
        loadblacklist();
    }
    isRunning = true;
    QTimer::singleShot(0, this, SLOT(saveSettings()));
}

void CallerXDaemon::callScreen(const QDBusObjectPath &call, const QString &number)
{
    callPath = call.path();
    lastInNumber = number;
    if (lastInNumber.startsWith("+")){
        lastInNumber.remove(0,1);
    }
    if(isBlocked(lastInNumber)) {
        callRelease(callPath);
        lastInNumber.prepend("x");
    }
    QTimer::singleShot(0, this, SLOT(logNumber()));
}

bool CallerXDaemon::isBlocked(const QString &number)
{
    n = number;
    bool match = whitelistMode;
    if (!n.isEmpty()){
        for (int i = 0; i < nList.size(); ++i){
            if (n.startsWith(nList.at(i))){
                match = !match;
                break;
            }
        }
    }
    else {
        match = blockUnknown;
    }
    return match;
}

void CallerXDaemon::callRelease(const QString &path)
{
    QDBusInterface iface(BUS_NAME, path, INSTANCE_INTERFACE, QDBusConnection::systemBus(), this);
    iface.call("Release");
}

void CallerXDaemon::stop()
{
    isRunning = false;
    saveSettings();
    emit quit();
    return;
}

void CallerXDaemon::reload()
{
    readSettings();
    setLoadList(listPath, whitelistMode);
}

void CallerXDaemon::addrecentnumber()
{
    callRelease(callPath);
    if (!n.isEmpty()) {
        addToList(n);
    }
    else {
        QSettings settings("callerx", "callerx");
        settings.setValue("blockunknown", blockUnknown);
    }
}

void CallerXDaemon::loadblacklist()
{
    QSettings settings("callerx", "callerx");
    blacklistPath = settings.value("blacklistpath", DEFAULT_BLACKLIST).toString();
    setLoadList(blacklistPath, false);
}

void CallerXDaemon::loadwhitelist()
{
    QSettings settings("callerx", "callerx");
    whitelistPath = settings.value("whitelistpath", DEFAULT_WHITELIST).toString();
    setLoadList(whitelistPath, true);
}

void CallerXDaemon::setLoadList(const QString &f, bool m)
{
    if (QFile::exists(f)) {
        whitelistMode = m;
        QSettings settings("callerx", "callerx");
        settings.setValue("whitelistmode", m);
        fileWatcher->removePath(listPath);
        listPath = f;
        fileWatcher->addPath(listPath);
        loadList();
    }
}

void CallerXDaemon::loadList()
{
    nList.clear();
    QFile fileList(listPath);
    if (fileList.exists()) {
        if (!fileList.open(QIODevice::ReadOnly | QIODevice::Text)){
            return;
        }
        QTextStream in(&fileList);
        QString line = in.readLine();
        while (!line.isNull()) {
            if (!nList.contains(line)) {
               nList.append(line);
            }
            line = in.readLine();
        }
        fileList.close();
        activeTimer = false;
    }
}

void CallerXDaemon::addToList(const QString &number)
{
    QFile fileList(listPath);
    if (fileList.exists()) {
        if (!fileList.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Append)){
            return;
        }
        QTextStream out(&fileList);
        out << number << "\n";
        fileList.close();
    }
}

void CallerXDaemon::updateList()
{
    if (!activeTimer) {
        activeTimer = true;
        QTimer::singleShot(1000, this, SLOT(loadList()));
    }
}

void CallerXDaemon::logNumber()
{
    QString t = QLocale::system().toString(QTime::currentTime(), QLocale::ShortFormat);
    QString d = QLocale::system().toString(QDate::currentDate(), QLocale::ShortFormat);
    QString dt = QString(" [ %1 | %2 ]").arg(d).arg(t.toUpper().remove(QRegExp(" ")));
    lastInNumber.append(dt);
    if (callHistory.size() > 20) {
        callHistory.removeLast();
    }
    callHistory.prepend(lastInNumber);
    QSettings settings("callerx", "callerx");
    settings.setValue("recentinboundcalls", callHistory);
    sendSignal();
}

void CallerXDaemon::sendSignal()
{
    QDBusConnection connection = QDBusConnection::sessionBus();
    QDBusMessage msg = QDBusMessage::createSignal("/org/callerx/ifdbus", "org.callerx.ifdbus", "incoming");
    msg << lastInNumber;
    connection.send(msg);
}

QString CallerXDaemon::ping()
{
    return QString("pong");
}

