#include "daemon.h"

CallerXDaemon::CallerXDaemon(QObject *parent) :
    QObject(parent)
{
    whitelistmode = settings.value("whitelistmode", false).toBool();
    whitelistpath = settings.value("whitelistpath", DEFAULT_WHITELIST).toString();
    blacklistpath = settings.value("blacklistpath", DEFAULT_BLACKLIST).toString();
    if (whitelistmode) {
        listPath = whitelistpath;
    }
    else {
        listPath = blacklistpath;
    }
    settings.setValue("whitelistmode", whitelistmode);
    settings.setValue("whitelistpath", whitelistpath);
    settings.setValue("blacklistpath", blacklistpath);
}

CallerXDaemon::~CallerXDaemon()
{
    QDBusConnection::systemBus().disconnect(BUS_NAME, BUS_PATH, BUS_NAME, "Coming", this, SLOT(callScreen(const QDBusObjectPath&, const QDBusMessage&)));
    delete fileWatcher;
    settings.sync();
}

void CallerXDaemon::Start()
{
    if (!QDBusConnection::systemBus().connect(BUS_NAME, BUS_PATH, BUS_NAME, "Coming", this, SLOT(callScreen(const QDBusObjectPath&, const QDBusMessage&)))){
        emit quit();
        return;
    }

    if (!QDBusConnection::sessionBus().isConnected()) {
        emit quit();
        return;
    }
    if (!QDBusConnection::sessionBus().registerService("org.callerx.ifdbus")) {
        emit quit();
        return;
    }
    if (!QDBusConnection::sessionBus().registerObject("/org/callerx/ifdbus", this, QDBusConnection::ExportScriptableSlots)) {
        emit quit();
        return;
    }

    QFile blocklist(listPath);
    if (!blocklist.exists()) {
        if (!blocklist.open(QIODevice::WriteOnly | QIODevice::Text)){
            return;
        }
        blocklist.close();
    }

    fileWatcher = new QFileSystemWatcher();
    fileWatcher->addPath(listPath);
    QObject::connect(fileWatcher, SIGNAL(fileChanged(const QString&)), this, SLOT(loadlist()));

    if (whitelistmode) {
        loadwhitelist();
    }
    else {
        loadblacklist();
    }
}

void CallerXDaemon::callScreen(const QDBusObjectPath &call, const QDBusMessage &msg)
{
    callPath = call.path();
    if(isBlocked(msg.arguments()[1].toString())) {
        callRelease(callPath);
    }
}

bool CallerXDaemon::isBlocked(const QString &number)
{
    n = number;
    bool match = whitelistmode;
    if (n.isEmpty()){
        n = "unknown";
    }
    for (int i = 0; i < nlist.size(); ++i){
        if (n.startsWith(nlist.at(i))){
            match = !match;
            break;
        }
    }
    return match;
}

void CallerXDaemon::callRelease(const QString &path)
{
    QDBusInterface iface(BUS_NAME, path, INSTANCE_INTERFACE, QDBusConnection::systemBus(), this);
    iface.call("Release");
}

void CallerXDaemon::stop()
{
    emit quit();
    return;
}

void CallerXDaemon::reload()
{
    loadlist();
}

void CallerXDaemon::addrecentnumber()
{
    if (n.isEmpty()) {
        n = "unknown";
    }
    callRelease(callPath);
    addtolist(n);
}

void CallerXDaemon::loadblacklist()
{
    blacklistpath = settings.value("blacklistpath", DEFAULT_BLACKLIST).toString();
    setloadlist(blacklistpath, false);
}

void CallerXDaemon::loadwhitelist()
{
    whitelistpath = settings.value("whitelistpath", DEFAULT_WHITELIST).toString();
    setloadlist(whitelistpath, true);
}

void CallerXDaemon::setloadlist(const QString &f, bool m)
{
    if (QFile::exists(f)) {
        whitelistmode = m;
        settings.setValue("whitelistmode", m);
        fileWatcher->removePath(listPath);
        listPath = f;
        fileWatcher->addPath(listPath);
        loadlist();
    }
}

void CallerXDaemon::loadlist()
{
    nlist.clear();
    QFile blocklist(listPath);
    if (blocklist.exists()) {
        if (!blocklist.open(QIODevice::ReadOnly | QIODevice::Text)){
            return;
        }
        QTextStream in(&blocklist);
        QString line = in.readLine();
        while (!line.isNull()) {
            if (!nlist.contains(line)) {
               nlist.append(line);
            }
            line = in.readLine();
        }
        blocklist.close();
    }
}

void CallerXDaemon::addtolist(const QString &number)
{
    QFile blocklist(listPath);
    if (blocklist.exists()) {
        if (!blocklist.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Append)){
            return;
        }
        QTextStream out(&blocklist);
        out << number << endl;
        blocklist.close();
    }
}
