/* 
   Copyright (C) 2011 Ove Kaaven

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

#include "settings.h"
#include <stdio.h>
#include <string.h>
#include <libxml/parser.h>
#include <glib.h>

typedef struct {
  xmlNodePtr region;
  xmlDocPtr conf;
} region_cache;

#define CACHE_ENTRIES 2

static xmlDocPtr system_conf;
static xmlDocPtr region_conf;

static region_cache cache[CACHE_ENTRIES];
static unsigned cache_next;

int exists(const char *fn)
{
  return g_file_test(fn, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_REGULAR);
}

void load_settings(const char *homedir)
{
  unload_settings();
  /* conf = xmlReadFile(fname, NULL, 0); */
  region_conf = xmlReadFile(DATADIR "/regions.xml", NULL, 0);
  if (!region_conf) {
    /* fallback for development... */
    if (exists("regions.xml")) {
      region_conf = xmlReadFile("regions.xml", NULL, 0);
    }
  }
}

void unload_settings(void)
{
  unsigned c;
  for (c=0; c<CACHE_ENTRIES; c++) {
    if (cache[c].conf) {
      xmlFreeDoc(cache[c].conf);
      cache[c].conf = NULL;
    }
    cache[c].region = NULL;
  }
  cache_next = 0;
  if (region_conf) {
    xmlFreeDoc(region_conf);
    region_conf = NULL;
  }
  if (system_conf) {
    xmlFreeDoc(system_conf);
    system_conf = NULL;
  }
}

static int is_region_match(xmlNodePtr region, const char *number, const char *mcc)
{
  xmlNodePtr cur = region ? region->children : NULL;
  while (cur) {
    if (cur->type == XML_ELEMENT_NODE) {
      if (!strcmp((char *)cur->name, "prefix")) {
        if (number[0] == '+') {
          xmlChar *data = xmlNodeGetContent(cur);
          int c = strncmp(number, (char *)data, strlen((char *)data));
          xmlFree(data);
          if (!c) {
            /* prefix match */
            return 1;
          }
        }
      }
      if (!strcmp((char *)cur->name, "mcc")) {
        if (number[0] != '+' && mcc) {
          xmlChar *data = xmlNodeGetContent(cur);
          int c = strcmp(mcc, (char *)data);
          xmlFree(data);
          if (!c) {
            /* MCC match */
            return 1;
          }
        }
      }
    }
    cur = cur->next;
  }
  return 0;
}

static xmlNodePtr search_for_region(const char *number, const char *mcc)
{
  xmlNodePtr root = region_conf ? xmlDocGetRootElement(region_conf) : NULL;
  xmlNodePtr cur = root ? root->children : NULL;
  while (cur) {
    if (is_region_match(cur, number, mcc)) {
      return cur;
    }
    cur = cur->next;
  }
  return NULL;
}

xmlNodePtr get_region(const char *number, const char *mcc)
{
  xmlNodePtr region;
  unsigned c;

  /* search cache */
  for (c=0; c<CACHE_ENTRIES; c++) {
    if (is_region_match(cache[c].region, number, mcc)) {
      return cache[c].region;
    }
  }

  /* search regions.xml */
  region = search_for_region(number, mcc);
  if (!region) return NULL;

  return region;
}

xmlChar *get_region_code(xmlNodePtr region)
{
  xmlNodePtr cur = region ? region->children : NULL;
  while (cur) {
    if (cur->type == XML_ELEMENT_NODE &&
        !strcmp((char *)cur->name, "code")) {
      xmlChar *data = xmlNodeGetContent(cur);
      return data;
    }
    cur = cur->next;
  }
  return NULL;
}

xmlChar *get_region_name(xmlNodePtr region)
{
  xmlNodePtr cur = region ? region->children : NULL;
  while (cur) {
    if (cur->type == XML_ELEMENT_NODE &&
        !strcmp((char *)cur->name, "name")) {
      xmlChar *data = xmlNodeGetContent(cur);
      return data;
    }
    cur = cur->next;
  }
  return NULL;
}

static xmlDocPtr get_region_config(xmlNodePtr region)
{
  unsigned c;
  xmlChar *code;
  char *fn;

  /* search cache */
  for (c=0; c<CACHE_ENTRIES; c++) {
    if (cache[c].region == region) {
      return cache[c].conf;
    }
  }

  /* load into cache */
  c = cache_next;
  cache_next = (cache_next + 1) % CACHE_ENTRIES;
  if (cache[c].conf) {
    xmlFreeDoc(cache[c].conf);
    cache[c].conf = NULL;
  }

  cache[c].region = region;

  code = get_region_code(region);
  fn = g_strconcat(DATADIR, "/", (char *)code, ".xml", NULL);
  cache[c].conf = xmlReadFile(fn, NULL, 0);
  if (!cache[c].conf) {
    /* fallback for development... */
    g_free(fn);
    fn = g_strconcat((char *)code, ".xml", NULL);
    if (exists(fn)) {
      cache[c].conf = xmlReadFile(fn, NULL, 0);
    }
  }
  xmlFree(code);
  g_free(fn);

  return cache[c].conf;
}

xmlChar *get_region_prefix(xmlNodePtr region)
{
  xmlNodePtr cur = region ? region->children : NULL;
  while (cur) {
    if (cur->type == XML_ELEMENT_NODE &&
        !strcmp((char *)cur->name, "prefix")) {
      xmlChar *data = xmlNodeGetContent(cur);
      return data;
    }
    cur = cur->next;
  }
  return NULL;
}

const char *skip_prefix(xmlNodePtr region, const char *number)
{
  if (number[0] == '+') {
    xmlChar *prefix = get_region_prefix(region);
    number += strlen((char *)prefix);
    xmlFree(prefix);
  }
  /* For domestic calls, maybe we'll also need to implement a way
   * to skip long-distance prefixes and such? */
  return number;
}

static xmlChar *get_region_directory_id(xmlNodePtr root, const char *number)
{
  xmlNodePtr cur = root ? root->children : NULL;
  while (cur) {
    if (cur->type == XML_ELEMENT_NODE &&
        !strcmp((char*)cur->name, "area")) {
      xmlAttrPtr prop = cur->properties;
      /* check for area code match */
      while (prop) {
        if (!strcmp((char *)prop->name, "code")) {
          xmlChar *val = xmlNodeGetContent(prop->children);
          int c = strncmp(number, (char *)val, strlen((char *)val));
          xmlFree(val);
          if (!c) break;
        }
        prop = prop->next;
      }
      if (prop) {
        /* match found, get ID */
        prop = cur->properties;
        while (prop) {
          if (!strcmp((char *)prop->name, "id")) {
            return xmlNodeGetContent(prop->children);
          }
          prop = prop->next;
        }
        return NULL;
      }
    }
    cur = cur->next;
  }
  return NULL;
}

static xmlNodePtr get_region_directory_from_id(xmlNodePtr root, xmlChar *dir_id)
{
  xmlNodePtr cur = root ? root->children : NULL;
  while (cur) {
    if (cur->type == XML_ELEMENT_NODE &&
        !strcmp((char*)cur->name, "directory")) {
      xmlAttrPtr prop = cur->properties;
      /* if dir_id == NULL, require a directory without an id property */
      /* if dir_id != NULL, require a matching id property
       * (we'll also allow more than one id property to exist) */
      int match = !dir_id;
      while (prop) {
        if (!strcmp((char *)prop->name, "id")) {
          xmlChar *val = xmlNodeGetContent(prop->children);
          if (dir_id) {
            match = !strcmp((char *)dir_id, (char *)val);
          }
          xmlFree(val);
          if (match) break;
        }
        prop = prop->next;
      }
      if (match) {
        return cur;
      }
    }
    cur = cur->next;
  }
  return NULL;
}

xmlNodePtr get_region_directory(xmlNodePtr region, const char *number)
{
  xmlDocPtr conf = get_region_config(region);
  xmlNodePtr root = conf ? xmlDocGetRootElement(conf) : NULL;
  xmlChar *dir_id = get_region_directory_id(root, number);
  xmlNodePtr dir = get_region_directory_from_id(root, dir_id);
  xmlFree(dir_id);
  return dir;
}

FILE *get_region_text(xmlNodePtr region, const char *number)
{
  xmlChar *code = get_region_code(region);
  char *pf, *fn;
  FILE *f;

  pf = g_strndup(number, 1);

  fn = g_strconcat(DATADIR, "/", (char *)code, ".txt", NULL);
  f = fopen(fn, "r");
  if (f) goto out;

  fn = g_strconcat(DATADIR, "/", (char *)code, "_", pf, ".txt", NULL);
  f = fopen(fn, "r");
  if (f) goto out;

  /* fallback for development */
  fn = g_strconcat((char *)code, ".txt", NULL);
  f = fopen(fn, "r");
  if (f) goto out;

  fn = g_strconcat((char *)code, "_", pf, ".txt", NULL);
  f = fopen(fn, "r");
  if (f) goto out;
      
out:
  xmlFree(code);
  g_free(pf);
  return f;
}
