/* 
   Copyright (C) 2011 Ove Kaaven

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

#include "notify.h"
#include "monitor.h"
#include <stdio.h>
#include <string.h>
#include <time.h>
#ifdef HAVE_OSSO
#include <libosso.h>
#endif
#ifdef HAVE_HILDON
#include <hildon/hildon-banner.h>
#endif

#ifdef HAVE_OSSO
static osso_context_t *osso_ctx;
#endif

void prepare_notify(void)
{
#ifdef HAVE_OSSO
  osso_ctx = osso_initialize("callerid", "1.0", 0, NULL);
  if (!osso_ctx) {
    fprintf(stderr, "Could not initialize OSSO\n");
  }
#endif
}

void cleanup_notify(void)
{
#ifdef HAVE_OSSO
  osso_deinitialize(osso_ctx);
  osso_ctx = NULL;
#endif
}

void print_time(void)
{
  time_t now = time(NULL);
  printf("%02ld:%02ld:%02ld: ",
         (now / 3600) % 24,
         (now / 60) % 60,
         now % 60);
}

void show_id(const char *name)
{
  char *text, *stext;
#ifdef HAVE_HILDON
  GtkWidget *banner;
#endif
#ifdef HAVE_OSSO
  osso_return_t ret;
#endif

  if (!name) name = "unknown";

  text = g_strconcat("Caller ", name, NULL);
  stext = text;

  print_time();
  printf("%s\n", text);

  if (strncmp(name, "in addressbook", 14) == 0) {
    /* skip first line (with name) if name is already in addressbook,
     * since the call dialog is probably already showing it */
    stext = strchr(text, '\n');
    if (!stext) {
      g_free(text);
      return;
    }
    stext++;
  }

#ifdef HAVE_HILDON
  /* We can use Hildon to show the information only if we're running
   * a main loop, which we currently only do in daemon mode.
   * This shows an information banner. */
  if (is_monitor_running()) {
    banner = hildon_banner_show_information_override_dnd(NULL, stext);
    hildon_banner_set_timeout(HILDON_BANNER(banner), 0);
    g_free(text);
    return;
  }
#endif

#ifdef HAVE_OSSO
  /* If we're not running a main loop, we can use OSSO to show the
   * information banner instead, which will work without a main loop.
   * However, we won't do this in daemon mode because an OSSO banner
   * apparently won't get shown at the same time as an incoming call
   * notification... */
  ret = osso_system_note_infoprint(osso_ctx, stext, NULL);
/* ret = osso_system_note_dialog(osso_ctx, text, OSSO_GN_NOTICE, NULL); */
  if (ret) {
    fprintf(stderr, "Could not show system note, OSSO error %d\n", ret);
  }
  else {
    /* success */
    return;
  }
#endif

  g_free(text);
}

void *get_system_dbus(void)
{
#ifdef HAVE_OSSO
  if (!osso_ctx) return NULL;
  return osso_get_sys_dbus_connection(osso_ctx);
#else
  return NULL;
#endif
}
