/* 
   Copyright (C) 2011 Ove Kaaven

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
*/

#include "monitor.h"
#include "lookup.h"
#include "notify.h"
#include <stdio.h>
#include <string.h>
#include <stdint.h>
#include <glib.h>
#include <dbus/dbus.h>

GMainLoop *main_loop;

static const char *dbus_rule = "type='signal',interface='com.nokia.csd.Call',member='Coming',path='/com/nokia/csd/call'";

static DBusHandlerResult filter_cb(DBusConnection *conn, DBusMessage *msg, void *data)
{
  if (dbus_message_is_signal(msg, "com.nokia.csd.Call", "Coming")) {
    DBusError err;
    char *path = NULL, *number = NULL;

    /* Got an incoming call */
    dbus_error_init(&err);
    dbus_message_get_args(msg, &err,
                          DBUS_TYPE_OBJECT_PATH, &path,
                          DBUS_TYPE_STRING, &number,
                          DBUS_TYPE_INVALID);
    if (dbus_error_is_set(&err)) {
      fprintf(stderr, "D-Bus error while parsing message: %s\n", err.message);
    }
    dbus_error_free(&err);
    if (number) {
      char *mcc = get_current_mcc();
      print_time();
      printf("MONITOR: Incoming call from '%s', mcc='%s'\n", number, mcc);
      /* Look up number */
      show_id(lookup(number, mcc));
      g_free(mcc);
    }
    return DBUS_HANDLER_RESULT_HANDLED;
  }
  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

void prepare_monitor(void)
{
  DBusConnection *conn = get_system_dbus();
  DBusError err;
  if (!conn) return;
  if (!dbus_connection_add_filter(conn, filter_cb, NULL, NULL)) {
    fprintf(stderr, "Could not install D-Bus filter\n");
  }
  dbus_error_init(&err);
  dbus_bus_add_match(conn, dbus_rule, &err);
  if (dbus_error_is_set(&err)) {
    fprintf(stderr, "D-Bus error while adding rule: %s\n", err.message);
  }
  dbus_error_free(&err);
}

void cleanup_monitor(void)
{
  DBusConnection *conn = get_system_dbus();
  DBusError err;
  if (!conn) return;
  dbus_error_init(&err);
  dbus_bus_remove_match(conn, dbus_rule, &err);
  dbus_error_free(&err);
  dbus_connection_remove_filter(conn, filter_cb, NULL);
}

char *get_current_mcc(void)
{
  DBusConnection *conn = get_system_dbus();
  DBusMessage *req, *rep;
  DBusError err;
  uint8_t netstatus, nettype, netservices;
  uint16_t lac;
  uint32_t cid, mnc, mcc;
  int32_t neterror;

  if (!conn) return NULL;

  dbus_error_init(&err);
  req = dbus_message_new_method_call("com.nokia.phone.net",
                                     "/com/nokia/phone/net",
                                     "Phone.Net",
                                     "get_registration_status");
  rep = dbus_connection_send_with_reply_and_block(conn, req, -1, &err);
  dbus_message_unref(req);
  if (dbus_error_is_set(&err)) {
    fprintf(stderr, "Could not get GSM connection status\n");
    if (rep) {
      dbus_message_unref(rep);
    }
    dbus_error_free(&err);
    return NULL;
  }

  dbus_message_get_args(rep, &err,
                        DBUS_TYPE_BYTE, &netstatus,
                        DBUS_TYPE_UINT16, &lac,
                        DBUS_TYPE_UINT32, &cid,
                        DBUS_TYPE_UINT32, &mnc,
                        DBUS_TYPE_UINT32, &mcc,
                        DBUS_TYPE_BYTE, &nettype,
                        DBUS_TYPE_BYTE, &netservices,
                        DBUS_TYPE_INT32, &neterror,
                        DBUS_TYPE_INVALID);

  if (dbus_error_is_set(&err)) {
    fprintf(stderr, "Could not get GSM connection status\n");
    dbus_message_unref(rep);
    dbus_error_free(&err);
    return NULL;
  }

  dbus_message_unref(rep);
  dbus_error_free(&err);

  /* We could check status and such, but if we've got an incoming
   * call, chances are that we'll find that the phone is online... */

  return g_strdup_printf("%u", mcc);
}

void run_monitor(void)
{
  DBusConnection *conn = get_system_dbus();
  if (!conn) return;
  main_loop = g_main_loop_new(NULL, FALSE);
  g_main_loop_run(main_loop);
  g_main_loop_unref(main_loop);
  main_loop = NULL;
}

int is_monitor_running(void)
{
  return main_loop != NULL;
}
