#include <QDebug>

#include <QLabel>
#include <QGroupBox>
#include <math.h>

#include "navigationwindow.h"
#include "compass.h"
#include "swapbox.h"

// a hbox with two strings, one left aligned and one right aligned
class InfoWidget : public QWidget {
public:
  InfoWidget(const QString &leftStr = "",
	     const QString &rightStr = "",
	     QWidget *parent = 0) : QWidget(parent) {

    QHBoxLayout *layout = new QHBoxLayout;
    layout->setContentsMargins(0,0,0,0);
    layout->setSpacing(0);

    m_left = new QLabel(leftStr + ":");
    layout->addWidget(m_left);
    layout->addStretch();
    m_right = new QLabel("<b>" + rightStr + "</b>");
    layout->addWidget(m_right);

    setLayout(layout);
  }

  void setInfo(const QString &info) {
    m_right->setText("<b>" + info + "</b>");
  }

private:
  QLabel *m_left, *m_right;
};

#if 0
class SizeCompass: public Compass {
public:
  SizeCompass(const Cache &cache, QWidget *parent): 
    Compass(cache, parent), m_other(NULL) {
  }

  void resizeEvent(QResizeEvent *) {

    // check if parent is really wider than high
    if(parentWidget()->size().width() > 
       parentWidget()->size().height()) {
      // width should then be at least half the total width
      int minWidth = (width() + m_other->width())/2;
      
      if(minWidth > height())
	minWidth = height();

      setMinimumWidth(minWidth);
    } else
      setMinimumWidth(0);
  }

  void setOther(QWidget *other) {
    m_other = other;
  }
  
private:
  QWidget *m_other;
};
#endif

NavigationWindow::NavigationWindow(const Cache &cache, 
	   LocationProvider *locationProvider, 
	   WaypointDb *wptDb, QWidget *parent) : 
  CustomWindow("NavigationWindow", parent), m_cache(cache),
  m_coordinateTool(NULL), m_coo(cache.coordinate()) {

  setWindowTitle(tr("Navigation"));

  SwapBox *sbox = new SwapBox();

  // try to force the compass to be at least as wide
  // as the infos on the right (bottom)

  // left/top: the compass
  Compass *compass = new Compass(m_cache, this);
  sbox->addWidget(compass);

  // right/bottom: the textual info
  QWidget *vbox = new QWidget(compass);
  //  compass->setOther(vbox);
  QVBoxLayout *layout = new QVBoxLayout;
  layout->setContentsMargins(0,0,0,0);
  layout->setSpacing(0);

  QGroupBox *groupBox = new QGroupBox();
  QVBoxLayout *ivbox = new QVBoxLayout;
  m_latitude = new InfoWidget(tr("Latitude"));
  ivbox->addWidget(m_latitude);
  m_longitude = new InfoWidget(tr("Longitude"));
  ivbox->addWidget(m_longitude);
  groupBox->setLayout(ivbox);
  layout->addWidget(groupBox);
  layout->addStretch();

  m_coordinateTool = new CoordinateTool(m_cache, wptDb, this);
  layout->addWidget(m_coordinateTool->createWidget());

  m_distance = new InfoWidget(tr("Distance"));
  m_coordinateTool->addWidget(m_distance);

  layout->addStretch();


  vbox->setLayout(layout);
  sbox->addWidget(vbox);

  setCentralWidget(sbox);

  connect(m_coordinateTool, SIGNAL(coordinateChanged(const QGeoCoordinate &)),
	  compass, SLOT(setTarget(const QGeoCoordinate &)));
  connect(m_coordinateTool, SIGNAL(typeChanged(const Waypoint &)),
	  compass, SLOT(setType(const Waypoint &)));
  connect(m_coordinateTool, SIGNAL(typeChanged()),
	  compass, SLOT(setType()));
  connect(m_coordinateTool, SIGNAL(coordinateChanged(const QGeoCoordinate &)),
	  this, SLOT(setTarget(const QGeoCoordinate &)));

  // send position/azimuth updates to compass widget
  connect(locationProvider, SIGNAL(positionUpdated(const QGeoPositionInfo &)), 
	  compass, SLOT(setPosition(const QGeoPositionInfo &)));
  connect(locationProvider, SIGNAL(positionUpdated(const QCompassReading *)), 
	  compass, SLOT(setPosition(const QCompassReading *)));

  // and process position locally
  connect(locationProvider, SIGNAL(positionUpdated(const QGeoPositionInfo &)), 
	  this, SLOT(setPosition(const QGeoPositionInfo &)));
}

NavigationWindow::~NavigationWindow() {
  qDebug() << __FUNCTION__;

  if(m_coordinateTool) delete m_coordinateTool;
}

void NavigationWindow::setTarget(const QGeoCoordinate &coo) {
  m_coo = coo;
}

void NavigationWindow::setPosition(const QGeoPositionInfo &pos) {
  qreal dist = pos.coordinate().distanceTo(m_coo);

  m_distance->setInfo(distanceString(dist));
  m_latitude->setInfo(CoordinateTool::latitudeString(pos.coordinate()));
  m_longitude->setInfo(CoordinateTool::longitudeString(pos.coordinate()));
}

QString NavigationWindow::distanceString(qreal dist) {
  // current we only support metric units
  
#if 0  
  /* 1 mil = 1760 yd = 5280 ft ... */
  if(dist<0.018)      snprintf(str, len, "%.1f ft", dist*5280.0);
  else if(dist<0.055) snprintf(str, len, "%.1f yd", dist*1760.0);
  else if(dist<0.55)  snprintf(str, len, "%.0f yd", dist*1760.0);
  else if(dist<10.0)  snprintf(str, len, "%.2f mi", dist);
  else if(dist<100.0) snprintf(str, len, "%.1f mi", dist);
  else                snprintf(str, len, "%.0f mi", dist);
#endif

  QString str;
  if(dist<10)          str = QString::number(dist, 'f', 2) + " m";
  else if(dist<100)    str = QString::number(dist, 'f', 1) + " m";
  else if(dist<1000)   str = QString::number(dist, 'f', 0) + " m";
  else if(dist<100000) str = QString::number(dist/1000, 'f', 1) + " km";
  else                 str = QString::number(dist/1000, 'f', 0) + " km";

  return str;
}


