#include "pixmap.h"
#include "filefinder.h"

#include <QDebug>
#include <QPainter>

bool Pixmap::load(QSvgRenderer *renderer, const QString &name) {
  QString filename = FileFinder::find("icons", name);
  if(filename.isNull()) return false;

  renderer->load(filename);

  return true;
}

// load file into renderer and return resulting size
QSize Pixmap::prepare(QSvgRenderer *renderer, const QString &name, int size) {
  if(Pixmap::load(renderer, name + ".svg")) {
    // if no explicit size is given, then use default size of first image
    if(size > 0) 
      return QSize(renderer->defaultSize().width()*size/
		   renderer->defaultSize().height(), size);
    else 
      return(renderer->defaultSize());
  }

  return QSize(size, size);
}

// create empty pixmap for graphic effects
Pixmap::Pixmap(const QStringList &name, int size, State state) :
  QPixmap(size, size) {

  m_offset -= QPoint(width()/2, height()/2);

  // fill with transparency and saved icon name
  fill(Qt::transparent);
  this->m_name = name.join(",");
  this->m_state = state;  
}

// create a Pixmap from svg file
Pixmap::Pixmap(QSvgRenderer *renderer, 
	       const QStringList &name, int size) :
  QPixmap(prepare(renderer, name[0], size)) {

  // try to find hotspot object, otherwise just center
  if(renderer->elementExists("hotspot"))
    m_offset = QPoint(0,0) - renderer->boundsOnElement("hotspot").center().toPoint();
  else
    m_offset -= QPoint(width()/2, height()/2);

  // fill with transparency and saved icon name
  fill(Qt::transparent);
  this->m_name = name.join(",");
  this->m_state = Pixmap::Normal;
  
  QPainter painter(this);
  painter.setRenderHint(QPainter::Antialiasing, true);

  // draw all given images on top of each other
  QStringList::const_iterator i;
  for(i=name.begin();i!=name.end();i++) {
    // check if renderer already successfully loaded a file
    if((i == name.begin()) && renderer->isValid())
      renderer->render(&painter);
    else {
      if(load(renderer, *i + ".svg"))
	renderer->render(&painter);
      else {
	// unable to load image, draw some kind of "broken" icon
	painter.setPen(QPen(QBrush(Qt::red), size/5) );
	painter.drawEllipse(QRect(QPoint(size/8,size/8), this->size() - 
				  QSize(size/4, size/4)));
      }
    }
  }
}

Pixmap::~Pixmap() {
}

bool Pixmap::is(const QString &name, State state) {
  return m_name == name && m_state == state;
}

QPoint Pixmap::offset() {
  return m_offset;
}
