#include <QDebug>
#include <QtPlugin>
#include <QSettings>
#include <math.h>

#include <QDialog>
#include <QGroupBox>
#include <QVBoxLayout>
#include <QLabel>
#include <QNetworkCookie>
#include <QHostInfo>
#include <QNetworkConfigurationManager>
#include <QNetworkSession>
#include <QNetworkInterface>

#include "gcbrowser.h"
#include "gcparser.h"
#include "gchtmlparser.h"

#define PLUGIN_NAME "GcBrowser"

// conversions from and to tile coordinates
int long2tilex(double lon, int z)  { 
  return (int)(floor((lon + 180.0) / 360.0 * pow(2.0, z))); 
}
 
int lat2tiley(double lat, int z) { 
  return (int)(floor((1.0 - log( tan(lat * M_PI/180.0) + 1.0 / cos(lat * M_PI/180.0)) / M_PI) / 2.0 * pow(2.0, z))); 
}
 
double tilex2long(double x, int z)  {
  return x / pow(2.0, z) * 360.0 - 180;
}

double tiley2lat(double y, int z)  {
  double n = M_PI - 2.0 * M_PI * y / pow(2.0, z);
  return 180.0 / M_PI * atan(0.5 * (exp(n) - exp(-n)));
}

// The GcUrl class automatically creates urls for geocaching.com
class GcUrl : public QUrl {
public:
  GcUrl(const QString &url, bool useHttps = false) : 
    QUrl("http" + QString(useHttps?"s":"") + "://www.geocaching.com" + url) {}
};

// the GcNetworkRequest pre-sets some header values
class GcNetworkRequest : public QNetworkRequest {
public:
  GcNetworkRequest() : QNetworkRequest() {
    setRawHeader("User-Agent", "CacheMe " PLUGIN_NAME " Plugin");
  }
};

// return a string containing everything, the string str contains 
// between the first occurances of the strings start and end
QString GcBrowser::subString(const QString &str, const QString &start, const QString &end) {
		
  int is = str.indexOf(start) + start.length();
  int ie = str.indexOf(end, is);

  if (is == start.length()-1 || ie == -1) 
    return NULL;

  return str.mid(is, ie-is);  
}

void GcBrowser::handleViewStateField(const QString &str) {
  QString lvCount = subString(str,  "__VIEWSTATEFIELDCOUNT\" value=\"", "\" />");
  int count = lvCount.isNull()?1:lvCount.toInt();

  QString lvResult;

  for (int i=0; i<count; i++) { 
    QString key = "__VIEWSTATE"+(!i?"":QString::number(i));
    
    QString lvPart = subString(str, key+"\" value=\"", "\" />");
    if(lvPart.isNull()) {
      qDebug() << __FUNCTION__ << "Didn't get a Viewstate!";
      return;
    }

    lvResult.append(i!=0?"&":"").append(key).append("=").append(QUrl::toPercentEncoding(lvPart));
  }

  m_lastViewState = lvResult;
}

// ------------ handling of user token ---------------

bool GcBrowser::decodeUserToken(const QString &str) {
  this->m_userToken = subString(str,  "userToken = '", "';");
  return !this->m_userToken.isNull();
}

void GcBrowser::requestUserToken() {
  GcNetworkRequest request;
  request.setUrl(GcUrl("/map/default.aspx"));
  this->m_posted = QString();
  QNetworkReply *reply = this->m_manager->get(request);
  emit notifyBusy(true);
  this->m_state = RequestedUserToken;

  if(reply->error()) replyFinished(reply);
}

// --------------- log into gc.com --------------------

void GcBrowser::prepareLogin() {
  if(loginIsValid()) {
    GcNetworkRequest request;
    request.setUrl(GcUrl("/login/default.aspx", true));
    this->m_posted = QString();
    QNetworkReply *reply = this->m_manager->get(request);    
    emit notifyBusy(true);
    this->m_state = SentLoginGET;

    if(reply->error()) replyFinished(reply);
  } else {
    error(tr("No valid login specified, cache positions will be inaccurate!"));
    requestUserToken();
  }
}

void GcBrowser::doLogin() {

  GcNetworkRequest request;
  request.setUrl(GcUrl("/login/default.aspx", true));

  QString data = m_lastViewState;
  data += "&ctl00$ContentBody$cbRememberMe=on";
  data += "&ctl00$ContentBody$btnSignIn=Login";
  data += "&ctl00$ContentBody$tbUsername=" + 
    QUrl::toPercentEncoding(m_name);
  data += "&ctl00$ContentBody$tbPassword=" + 
    QUrl::toPercentEncoding(m_password);
 
  this->m_posted = data;

  m_manager->post(request, data.toUtf8());

  emit notifyBusy(true);
  this->m_state = SentLoginPOST;
}

// the following must match the same routine in settingsdialog.cpp
static QString defuscate(const QString &in) {
  if((in.length() < 1) || (in[0] != '!')) 
    return in;

  QByteArray dec = QByteArray::fromHex(QString(in).remove(0,0).toAscii());
  for(int i=0;i<dec.length();i++)
    dec[i] = dec[i] ^ 0x55;

  return dec;
}

GcBrowser::GcBrowser() : m_initialized(false), m_cacheList(PLUGIN_NAME), 
		 m_state(Idle), m_currentRequest(None), m_pendingReload(false) {
  qDebug() << __PRETTY_FUNCTION__;

  // try to get credentials from qsettings
  QSettings settings;
  settings.beginGroup("Account");
  m_name = settings.value("Name", "").toString();
  m_password = defuscate(settings.value("Password", "").toString());
  settings.endGroup();

  // setup network manager and listen for its replies
  this->m_manager = new QNetworkAccessManager(this);

#if 0
  // get current list of ip addresses
  QNetworkConfigurationManager mgr;
  QNetworkSession session (mgr.defaultConfiguration());
  QNetworkInterface ninter = session.interface();
  // the next statement gives you a funny name on windows
  qDebug() << "Ifname" << ninter.name();
  qDebug() << "Ifaddresses" << ninter.allAddresses();
 
  // this provides two ip addresses (1 ipv4 and 1 ipv6) at least on my machine
  QList<QNetworkAddressEntry> laddr = ninter.addressEntries();
  for ( QList<QNetworkAddressEntry> ::const_iterator it = laddr.begin(); it != laddr.end(); ++it ) {
    qDebug() << "Entry" << it->ip();
  }
#endif

  // try to load stored cookies
  QList<QNetworkCookie> cookies;
  settings.beginGroup(PLUGIN_NAME);
  settings.beginGroup("Cookies");
  foreach(QString key, settings.allKeys()) 
    cookies.append(QNetworkCookie(key.toAscii(), settings.value(key).toByteArray()));
  settings.endGroup();
  settings.endGroup();
  this->m_manager->cookieJar()->setCookiesFromUrl(cookies, GcUrl(""));

  connect(this->m_manager, SIGNAL(finished(QNetworkReply*)),
	  this, SLOT(replyFinished(QNetworkReply*)));
}

GcBrowser::~GcBrowser() {
  qDebug() << __PRETTY_FUNCTION__;

  delete this->m_manager;
}

QString GcBrowser::name() {
  return PLUGIN_NAME;
}

bool GcBrowser::canBeDefault() {
  // this plugin doesn't necessarily need configuration to 
  // work, so it can be used as a default
  return true;
}

QString GcBrowser::license() {
  return QObject::tr("GPL version 2\n"
		     "Copyright 2011 by Till Harbaum\n"
		     "QSgml parts Copyright 2010 by Andreas Lehmann");
}

QObject *GcBrowser::object() {
  return this;
}

void GcBrowser::init(QWidget *) {
  prepareLogin();
}

void GcBrowser::login(const QString &name, const QString &password) {
  this->m_name = name;
  this->m_password = password;

  prepareLogin();
}

bool GcBrowser::loginIsValid() {
  return(!m_name.isEmpty() && 
	 !m_password.isEmpty());
}

bool GcBrowser::busy() {
  // initialized and no pending login request
  return(!m_initialized || m_state != Idle);
}

void GcBrowser::replyFinished(QNetworkReply *reply) {

  if(reply->error() == QNetworkReply::UnknownNetworkError) {
    qDebug() << __FUNCTION__ << "bailing out due to unknown error";
    error("");  // clear error to make sure more messages are displayed
    error(tr("!Network connection failed, "
	     "please make sure you are connected to the internet and restart CacheMe!"));
    return;
  }

  QUrl redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();
  if (!redirect.isEmpty()) {
    QString allData = QString::fromUtf8(reply->readAll());

    if(redirect.host().isEmpty())
      redirect.setHost("www.geocaching.com");
    
    if(redirect.scheme().isEmpty())
      redirect.setScheme("https");

    qDebug() << __FUNCTION__ << "redirect to" << redirect;

    GcNetworkRequest request;
    request.setUrl(redirect);
    QNetworkReply *newReply;

    // http 1.1 always builds a get request from a redirected get or post
    newReply = this->m_manager->get(request);
    
    if(newReply->error()) replyFinished(newReply);
    reply->deleteLater();
    return;
  }
  
  if(reply->error()) {
    if(reply->errorString().contains("Error",  Qt::CaseInsensitive))
      error(reply->errorString());
    else
      error(tr("Error") + ": " + reply->errorString());
  }

  // invoke appropriate decoder
  if(reply->isFinished()) {
    QString allData = QString::fromUtf8(reply->readAll());

    handleViewStateField(allData);

    qDebug() << __FUNCTION__ << "reply in state " << m_state;

    if(m_state == RequestedUserToken) {
      if(decodeUserToken(allData))
	qDebug() << __FUNCTION__ << "UserToken is" << this->m_userToken;

      emit reload();

      // now everything is initialized
      m_initialized = true;
      m_state = Idle;

    } else if(this->m_state == SentLoginGET || this->m_state == SentLoginPOST) {

      // check if we are successfully logged in
      if (allData.contains("You are logged in as")) {
	qDebug() << __FUNCTION__ << "login successful";

	QSettings settings;
	settings.beginGroup(PLUGIN_NAME);
	settings.beginGroup("Cookies");
	QList<QNetworkCookie> cookies = 
	  this->m_manager->cookieJar()->cookiesForUrl(GcUrl("", true));

	// save all non-session cookies
	foreach(QNetworkCookie cookie, cookies) 
	  if(!cookie.isSessionCookie()) 
	    settings.setValue(cookie.name(), cookie.value());

	settings.endGroup();
	settings.endGroup();

	requestUserToken();
      } else {
	if(this->m_state == SentLoginGET) {
	  qDebug() << __FUNCTION__ << "we are not already logged in, doing so ...";
	  doLogin();
	} else if(this->m_state == SentLoginPOST) {
	  qDebug() << __FUNCTION__ << "login failed";
	  if(!reply->error()) {
	    error("");  // clear error to make sure more messages are displayed
	    error(tr("Login failed!"));
	  }
	  requestUserToken();
	}
      }
    } else {
      qDebug() << __FUNCTION__ << "reply";

      // initialization is finished, any reply received now should
      // be a reply to a real request
      CurrentRequest type = this->m_currentRequest;

      if(type != Overview) {
	this->m_currentRequest = None;
	emit done();
      }

      if(!reply->error()) {
	GcParser gcParser;
	GcHtmlParser gcHtmlParser;

	switch(type) {
	case Overview: 
	  qDebug() << __FUNCTION__ << "rcvd: " << allData.length();

	  if((!m_tile_is_image) && (allData.length() > 0)) 
	    if(!gcParser.decodeOverview(allData, m_overviewMap, 
					m_tile_x_cur, m_tile_y_cur))
	      error(gcParser.error());

	  if(m_tile_is_image) 
	    m_tile_is_image = false;
	  else {
	    m_tile_is_image = true;
	    m_tile_x_cur++;
	    if(m_tile_x_cur > m_tile_x_max) {
	      m_tile_x_cur = m_tile_x_min;
	      m_tile_y_cur++;
	    }
	  }
	  
	  if(!overviewDone())
	    requestNextTile();
	  else {
	    this->m_currentRequest = None;
	    emit done();

	    m_cacheList.clear();
	    QMap<QString, OverviewEntry>::iterator i = m_overviewMap.begin();
	    while (i != m_overviewMap.end()) {
	      OverviewEntry entry = i.value();

	      // create cache entries from overview map
	      Cache cache;
	      
	      cache.setDescription(entry.n);
	      cache.setType(Cache::TypeTraditional);
	      cache.setName(i.key());
	      cache.setCoordinate(
		 QGeoCoordinate(tiley2lat(entry.y/entry.c, m_tile_zoom),
				tilex2long(entry.x/entry.c, m_tile_zoom)));

	      m_cacheList.append(cache);
	      ++i;
	    }
	    
#if 0
	  // append saved Waypoints if they belong to one of the caches
	  // returned
	  for(QList<Cache>::iterator i = m_cacheList.begin(); 
	      i != m_cacheList.end(); ++i ) 
	    if(i->name() == m_lastWaypointsName)
	      foreach(const Waypoint wpt, m_lastWaypoints)
		i->appendWaypoint(wpt);

#endif

	    // always emit the list, even if it's empty, so existing caches
	    // get cleared
	    emit replyOverview(m_cacheList);
	    m_state = Idle;
	  }
	  
	  break;
	  
	case Info: 
	  if(gcParser.decodeInfo(allData, m_cache))
	    emit replyInfo(m_cache);
	  else
	    error(gcParser.error());
	  break;
	  
	case Detail: 
	  if(gcHtmlParser.decode(allData, m_cache)) {
	    emit replyDetail(m_cache);

	    if(m_lastWaypointsName != m_cache.name()) {
	      // save waypoints for future use
	      m_lastWaypointsName = m_cache.name();
	      m_lastWaypoints = m_cache.waypoints();

	      // request reload of main map to update waypoints
	      m_pendingReload = true;
	    }

	  } else
	    error(gcHtmlParser.error());
	  break;
	  
	default:
	  qDebug() << __FUNCTION__ << "unknown request type" << type;
	  Q_ASSERT(0);
	  break;
	}
      }
    }
  }

  qDebug() << __FUNCTION__ << "next, busy = " << busy();
  if(!busy()) 
    emit next();

  emit notifyBusy(false);
  reply->deleteLater();

  if(m_pendingReload) {
    emit reload();
    m_pendingReload = false;
  }
}

bool GcBrowser::overviewDone() {
  return(m_tile_y_cur > m_tile_y_max);
}

void GcBrowser::requestNextTile() {
  qDebug() << __FUNCTION__;

  if(!overviewDone()) {
    QString flagStr, type;

    if(m_tile_is_image) type = "tile";
    else                type = "info";

#if 0 // not working yet
    if(m_flags & CacheProvider::NoFound) flagStr += "&hf=1";
    if(m_flags & CacheProvider::NoOwned) flagStr += "&hh=1";
#endif

    GcNetworkRequest request;
    request.setUrl(GcUrl("/map/default.aspx/map." + type +
			 "?x=" + QString::number(m_tile_x_cur) + 
			 "&y=" + QString::number(m_tile_y_cur) + 
			 "&z=" + QString::number(m_tile_zoom) + 
			 flagStr));

   qDebug() << request.url();
    this->m_posted = QString();
    QNetworkReply *reply = this->m_manager->get(request);    
    emit notifyBusy(true);
    if(reply->error()) replyFinished(reply);
  }
}

void GcBrowser::processRequestOverview(const QGeoBoundingBox &area, const int zoom, const int flags) {
  qDebug() << __PRETTY_FUNCTION__;

  // processing levels < 13 results in too much data to process on
  // the phone
  if(zoom < 13) {
    m_cacheList.clear();

    error(tr("Too many caches at this zoom level, please zoom in."));

    emit done();
    emit replyOverview(m_cacheList);
    emit next();

    return;
  }

  // check if we can fulfil request from cachelist
  if(m_cacheBox.contains(area) && m_cacheList.size() > 0) {
    qDebug() << __FUNCTION__ << "can fulfil request from cache";

    for(QList<Cache>::iterator i = m_cacheList.begin(); 
	i != m_cacheList.end(); ++i ) 
      if(i->name() == m_lastWaypointsName && i->waypoints().size() == 0)
	foreach(const Waypoint wpt, m_lastWaypoints)
	  i->appendWaypoint(wpt);
    
    // always emit the list, even if it's empty, so existing caches
    // get cleared
    emit done();
    emit replyOverview(m_cacheList);
    emit next();
    
    return;
  }  

  m_cacheBox = area;

  Q_ASSERT(this->m_currentRequest == None);
  this->m_currentRequest = Overview;

  // calculate tile bounds
  m_tile_zoom = zoom;
  m_tile_is_image = true;
  m_flags = flags;

  // setup 
  m_tile_x_min = long2tilex(area.topLeft().longitude(), m_tile_zoom);
  m_tile_x_max = long2tilex(area.bottomRight().longitude(), m_tile_zoom);
  m_tile_y_min = lat2tiley(area.topLeft().latitude(), m_tile_zoom);
  m_tile_y_max = lat2tiley(area.bottomRight().latitude(), m_tile_zoom);
  m_tile_x_cur = m_tile_x_min;
  m_tile_y_cur = m_tile_y_min;

  qDebug() << __FUNCTION__ << "Requesting tile area X:" <<
    m_tile_x_min << m_tile_x_max << " Y:" << m_tile_y_min << m_tile_y_max;

  m_state = ProcessingOverview;
  m_overviewMap.clear();
  requestNextTile();
}

void GcBrowser::processRequestInfo(const QString &name) {
  GcNetworkRequest request;
  
  qDebug() << __PRETTY_FUNCTION__ << name;
  
  Q_ASSERT(this->m_currentRequest == None);
  this->m_currentRequest = Info;
  
  // preset all present knowledge from cachelist (esp. coordinates)
  QList<Cache>::const_iterator i;
  for( i = m_cacheList.begin(); i != m_cacheList.end(); ++i )
    if(i->name() == name)
      m_cache = *i;
  
  request.setUrl(GcUrl("/map/default.aspx/map.details?i=" + name));
  qDebug() << request.url();
  this->m_posted = QString();
  QNetworkReply *reply = this->m_manager->get(request);    
  emit notifyBusy(true);
  if(reply->error()) replyFinished(reply);
}

void GcBrowser::processRequestDetail(const QString &name) {
  qDebug() << __PRETTY_FUNCTION__ << name;

  // assume that an info request has been processed before for
  // exactly this same cache
  Q_ASSERT(name == m_cache.name());

  if(m_cache.guid().isEmpty()) {
    error(tr("Unable to determine cache guid!"));

    // just eat the request for now
    emit done();
    emit next();

    return;
  }

  qDebug() << __FUNCTION__ << m_cache.guid();

  Q_ASSERT(this->m_currentRequest == None);
  this->m_currentRequest = Detail;

  // now request:
  GcNetworkRequest request;
  request.setUrl(GcUrl("/seek/cdpf.aspx?guid=" + m_cache.guid() + "&lc=5"));
  this->m_posted = QString();
  QNetworkReply *reply = this->m_manager->get(request);
  emit notifyBusy(true);

  if(reply->error()) replyFinished(reply);
}

void GcBrowser::applyChanges() {
  qDebug() << PLUGIN_NAME << __PRETTY_FUNCTION__;

  // settings have been saved by settings dialog,
  // so we can already read them again

  QSettings settings;
  settings.beginGroup("Account");
  QString name = settings.value("Name", "").toString();
  QString password = defuscate(settings.value("Password", "").toString());
  settings.endGroup();

  // (re-)login in now if password or username have changed
  if((name != this->m_name) || (password != this->m_password)) 
    login(name, password);
}

void GcBrowser::createConfig(QDialog *parent, QVBoxLayout *) {
  // this plugin doesn't have its own config, but it wants to
  // know when the global settings change
  connect(parent, SIGNAL(accepted()), this, SLOT(applyChanges()));
}

// send an error message to the cache provider
void GcBrowser::error(const QString &msg) {
  if(!msg.isEmpty()) {
    if(msg.at(0) == QChar('!')) 
      emit replyError("!" + tr(PLUGIN_NAME) + ": " + QString(msg).remove(0,1));
    else
      emit replyError(tr(PLUGIN_NAME) + ": " + msg);
  } else
    emit replyError("");
}

Q_EXPORT_PLUGIN2(gcbrowser, GcBrowser);
