#include <QDebug>
#include <QtPlugin>
#include <QSettings>


#include <QDialog>
#include <QGroupBox>
#include <QVBoxLayout>
#include <QLabel>
#include <QNetworkCookie>
#include <QHostInfo>
#include <QNetworkConfigurationManager>
#include <QNetworkSession>
#include <QNetworkInterface>

#include "gcbrowser.h"
#include "gcparser.h"
#include "gchtmlparser.h"

#define PLUGIN_NAME "GcBrowser"


// The GcUrl class automatically creates urls for geocaching.com
class GcUrl : public QUrl {
public:
  GcUrl(const QString &url, bool useHttps = false) : 
    QUrl("http" + QString(useHttps?"s":"") + "://www.geocaching.com" + url) {}
};

// the GcNetworkRequest pre-sets some header values
class GcNetworkRequest : public QNetworkRequest {
public:
  GcNetworkRequest() : QNetworkRequest() {
    setRawHeader("User-Agent", "CacheMe " PLUGIN_NAME " Plugin");
  }
};

// return a string containing everything, the string str contains 
// between the first occurances of the strings start and end
QString GcBrowser::subString(const QString &str, const QString &start, const QString &end) {
		
  int is = str.indexOf(start) + start.length();
  int ie = str.indexOf(end, is);

  if (is == start.length()-1 || ie == -1) 
    return NULL;

  return str.mid(is, ie-is);  
}

void GcBrowser::handleViewStateField(const QString &str) {
  QString lvCount = subString(str,  "__VIEWSTATEFIELDCOUNT\" value=\"", "\" />");
  int count = lvCount.isNull()?1:lvCount.toInt();

  QString lvResult;

  for (int i=0; i<count; i++) { 
    QString key = "__VIEWSTATE"+(!i?"":QString::number(i));
    
    QString lvPart = subString(str, key+"\" value=\"", "\" />");
    if(lvPart.isNull()) {
      qDebug() << __FUNCTION__ << "Didn't get a Viewstate!";
      return;
    }

    lvResult.append(i!=0?"&":"").append(key).append("=").append(QUrl::toPercentEncoding(lvPart));
  }

  m_lastViewState = lvResult;
}

// ------------ handling of user token ---------------

bool GcBrowser::decodeUserToken(const QString &str) {
  this->m_userToken = subString(str,  "userToken = '", "';");
  return !this->m_userToken.isNull();
}

void GcBrowser::requestUserToken() {
  GcNetworkRequest request;
  request.setUrl(GcUrl("/map/default.aspx"));
  this->m_posted = QString();
  QNetworkReply *reply = this->m_manager->get(request);
  emit notifyBusy(true);
  this->m_state = RequestedUserToken;

  if(reply->error()) replyFinished(reply);
}

// --------------- log into gc.com --------------------

void GcBrowser::prepareLogin() {
  if(loginIsValid()) {
    GcNetworkRequest request;
    request.setUrl(GcUrl("/login/default.aspx", true));
    this->m_posted = QString();
    QNetworkReply *reply = this->m_manager->get(request);    
    emit notifyBusy(true);
    this->m_state = SentLoginGET;

    if(reply->error()) replyFinished(reply);
  } else {
    error(tr("No valid login specified, cache positions will be inaccurate!"));
    requestUserToken();
  }
}

void GcBrowser::doLogin() {

  GcNetworkRequest request;
  request.setUrl(GcUrl("/login/default.aspx", true));

  QString data = m_lastViewState;
  data += "&ctl00$ContentBody$cbRememberMe=on";
  data += "&ctl00$ContentBody$btnSignIn=Login";
  data += "&ctl00$ContentBody$tbUsername=" + 
    QUrl::toPercentEncoding(m_name);
  data += "&ctl00$ContentBody$tbPassword=" + 
    QUrl::toPercentEncoding(m_password);
 
  this->m_posted = data;

  m_manager->post(request, data.toUtf8());

  emit notifyBusy(true);
  this->m_state = SentLoginPOST;
}

GcBrowser::GcBrowser() : m_initialized(false), m_cacheList(PLUGIN_NAME), 
		 m_state(Idle), m_currentRequest(None), m_pendingReload(false) {
  qDebug() << __PRETTY_FUNCTION__;

  // try to get credentials from qsettings
  QSettings settings;
  settings.beginGroup("Account");
  m_name = settings.value("Name", "").toString();
  m_password = settings.value("Password", "").toString();
  settings.endGroup();

  // setup network manager and listen for its replies
  this->m_manager = new QNetworkAccessManager(this);

  // get current list of ip addresses
  QNetworkConfigurationManager mgr;
  QNetworkSession session (mgr.defaultConfiguration());
  QNetworkInterface ninter = session.interface();
  // the next statement gives you a funny name on windows
  qDebug() << "Ifname" << ninter.name();
  qDebug() << "Ifaddresses" << ninter.allAddresses();
 
  // this provides two ip addresses (1 ipv4 and 1 ipv6) at least on my machine
  QList<QNetworkAddressEntry> laddr = ninter.addressEntries();
  for ( QList<QNetworkAddressEntry> ::const_iterator it = laddr.begin(); it != laddr.end(); ++it ) {
    qDebug() << "Entry" << it->ip();
  }

  // try to load stored cookies
  QList<QNetworkCookie> cookies;
  settings.beginGroup(PLUGIN_NAME);
  settings.beginGroup("Cookies");
  foreach(QString key, settings.allKeys()) 
    cookies.append(QNetworkCookie(key.toAscii(), settings.value(key).toByteArray()));
  settings.endGroup();
  settings.endGroup();
  this->m_manager->cookieJar()->setCookiesFromUrl(cookies, GcUrl(""));

  connect(this->m_manager, SIGNAL(finished(QNetworkReply*)),
	  this, SLOT(replyFinished(QNetworkReply*)));
}

GcBrowser::~GcBrowser() {
  qDebug() << __PRETTY_FUNCTION__;

  delete this->m_manager;
}

QString GcBrowser::name() {
  return PLUGIN_NAME;
}

bool GcBrowser::canBeDefault() {
  // this plugin doesn't necessarily need configuration to 
  // work, so it can be used as a default
  return true;
}

QString GcBrowser::license() {
  return QObject::tr("GPL version 2\n"
		     "Copyright 2011 by Till Harbaum\n"
		     "QSgml parts Copyright 2010 by Andreas Lehmann");
}

QObject *GcBrowser::object() {
  return this;
}

void GcBrowser::init(QWidget *) {
  prepareLogin();
}

void GcBrowser::login(const QString &name, const QString &password) {
  this->m_name = name;
  this->m_password = password;

  prepareLogin();
}

bool GcBrowser::loginIsValid() {
  return(!m_name.isEmpty() && 
	 !m_password.isEmpty());
}

bool GcBrowser::busy() {
  // initialized and no pending login request
  return(!m_initialized || m_state != Idle);
}

void GcBrowser::replyFinished(QNetworkReply *reply) {

  if(reply->error() == QNetworkReply::UnknownNetworkError) {
    qDebug() << __FUNCTION__ << "bailing out due to unknown error";
    error("");  // clear error to make sure more messages are displayed
    error(tr("!Network connection failed, "
	     "please make sure you are connected to the internet and restart CacheMe!"));
    return;
  }

  QUrl redirect = reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();
  if (!redirect.isEmpty()) {
    QString allData = QString::fromUtf8(reply->readAll());

    if(redirect.host().isEmpty())
      redirect.setHost("www.geocaching.com");
    
    if(redirect.scheme().isEmpty())
      redirect.setScheme("https");

    qDebug() << __FUNCTION__ << "redirect to" << redirect;

    GcNetworkRequest request;
    request.setUrl(redirect);
    QNetworkReply *newReply;

    // http 1.1 always builds a get request from a redirected get or post
#if 0
    if(this->m_posted.isEmpty())
      newReply = this->m_manager->get(request);
    else
      newReply = this->m_manager->post(request, this->m_posted.toUtf8());
#else
    newReply = this->m_manager->get(request);
#endif
    
    if(newReply->error()) replyFinished(newReply);
    reply->deleteLater();
    return;
  }
  
  if(reply->error()) {
    if(reply->errorString().contains("Error",  Qt::CaseInsensitive))
      error(reply->errorString());
    else
      error(tr("Error") + ": " + reply->errorString());
  }

  // invoke appropriate decoder
  if(reply->isFinished()) {
    QString allData = QString::fromUtf8(reply->readAll());

    handleViewStateField(allData);

    qDebug() << __FUNCTION__ << "reply in state " << m_state;

    if(m_state == RequestedUserToken) {
      if(decodeUserToken(allData))
	qDebug() << __FUNCTION__ << "UserToken is" << this->m_userToken;

      emit reload();

      // now everything is initialized
      m_initialized = true;
      m_state = Idle;

    } else if(this->m_state == SentLoginGET || this->m_state == SentLoginPOST) {

      // check if we are successfully logged in
      if (allData.contains("You are logged in as")) {
	qDebug() << __FUNCTION__ << "login successful";

	QSettings settings;
	settings.beginGroup(PLUGIN_NAME);
	settings.beginGroup("Cookies");
	QList<QNetworkCookie> cookies = 
	  this->m_manager->cookieJar()->cookiesForUrl(GcUrl("", true));

	// save all non-session cookies
	foreach(QNetworkCookie cookie, cookies) 
	  if(!cookie.isSessionCookie()) 
	    settings.setValue(cookie.name(), cookie.value());

	settings.endGroup();
	settings.endGroup();

	requestUserToken();
      } else {
	if(this->m_state == SentLoginGET) {
	  qDebug() << __FUNCTION__ << "we are not already logged in, doing so ...";
	  doLogin();
	} else if(this->m_state == SentLoginPOST) {
	  qDebug() << __FUNCTION__ << "login failed";
	  if(!reply->error()) {
	    error("");  // clear error to make sure more messages are displayed
	    error(tr("Login failed!"));
	  }
	  requestUserToken();
	}
      }
    } else {
      qDebug() << __FUNCTION__ << "reply";
      
      // initialization is finished, any reply received now should
      // be a reply to a real request
      CurrentRequest type = this->m_currentRequest;
      this->m_currentRequest = None;

      emit done();

      if(!reply->error()) {
	GcParser gcParser;
	GcHtmlParser gcHtmlParser;
	
	switch(type) {
	case Overview: 
	  // start parser (and tell him whether we tried to load a "big" area)
	  if(!gcParser.decodeOverview(allData, m_cacheList, m_bigArea))
	    error(gcParser.error());

	  // append saved Waypoints if they belong to one of the caches
	  // returned
	  for(QList<Cache>::iterator i = m_cacheList.begin(); 
	      i != m_cacheList.end(); ++i ) 
	    if(i->name() == m_lastWaypointsName)
	      foreach(const Waypoint wpt, m_lastWaypoints)
		i->appendWaypoint(wpt);

	  // always emit the list, even if it's empty, so existing caches
	  // get cleared
	  emit replyOverview(m_cacheList);
	  break;
	  
	case Info: 
	  if(gcParser.decodeInfo(allData, m_cache))
	    emit replyInfo(m_cache);
	  else
	    error(gcParser.error());
	  break;
	  
	case Detail: 
	  if(gcHtmlParser.decode(allData, m_cache)) {
	    emit replyDetail(m_cache);

	    if(m_lastWaypointsName != m_cache.name()) {
	      // save waypoints for future use
	      m_lastWaypointsName = m_cache.name();
	      m_lastWaypoints = m_cache.waypoints();

	      // request reload of main map to update waypoints
	      m_pendingReload = true;
	    }

	  } else
	    error(gcHtmlParser.error());
	  break;
	  
	default:
	  qDebug() << __FUNCTION__ << "unknown request type" << type;
	  Q_ASSERT(0);
	  break;
	}
      }
    }
  }

  qDebug() << __FUNCTION__ << "next, busy = " << busy();
  if(!busy()) 
    emit next();

  emit notifyBusy(false);
  reply->deleteLater();

  if(m_pendingReload) {
    emit reload();
    m_pendingReload = false;
  }
}

void GcBrowser::postJson(const QString &post) {
  // build and send request
  GcNetworkRequest request;
  request.setUrl(GcUrl("/map/default.aspx/MapAction"));
  request.setRawHeader("Accept", "application/json");
  request.setRawHeader("Content-Type", "application/json; charset=utf-8");

  this->m_posted = post;
  m_manager->post(request, post.toUtf8());
  emit notifyBusy(true);
}

void GcBrowser::processRequestOverview(const QGeoBoundingBox &area, const int) {
  qDebug() << __PRETTY_FUNCTION__;

  // check if we can fulfil request from cachelist
  if(m_cacheBox.contains(area) && m_cacheList.size() > 0) {
    qDebug() << __FUNCTION__ << "can fulfil request from cache";

    for(QList<Cache>::iterator i = m_cacheList.begin(); 
	i != m_cacheList.end(); ++i ) 
      if(i->name() == m_lastWaypointsName && i->waypoints().size() == 0)
	foreach(const Waypoint wpt, m_lastWaypoints)
	  i->appendWaypoint(wpt);
    
    // always emit the list, even if it's empty, so existing caches
    // get cleared
    emit done();
    emit replyOverview(m_cacheList);
    emit next();
    
    return;
  }  

  m_cacheBox = area;

  Q_ASSERT(this->m_currentRequest == None);
  this->m_currentRequest = Overview;
  
  // check if bounding box doesn't exceed limits (40km diagonally)
  this->m_bigArea = area.topLeft().distanceTo(area.bottomRight()) > 40000;
  
  // build request string. Isn't there a way to not request "owned" and
  // "found" ones?
  postJson("{\"dto\":{\"data\":{\"c\":1,\"m\":\"\",\"d\":\"" + 
	   QString::number(area.topLeft().latitude()) + "|" + 
	   QString::number(area.bottomRight().latitude()) + "|" + 
	   QString::number(area.bottomRight().longitude()) + "|" + 
	   QString::number(area.topLeft().longitude()) + 
	   "\"},\"ut\":\"" + this->m_userToken + "\"}}");
}

int GcBrowser::getCacheId(const QString &name) {
  int id = -1;

  // try to find matching cache id
  QList<Cache>::const_iterator i;
  for( i = m_cacheList.begin(); i != m_cacheList.end(); ++i ) {
    if(i->name() == name) {
      id = i->id();
      m_cache = *i;
    }
  }

  if(id < 0) {
    // remove current request and continue with the next one
    emit done();
    emit next();

    error(tr("Unable to determine cache id!"));
  }

  return id;
}

void GcBrowser::processRequestInfo(const QString &name) {
  qDebug() << __PRETTY_FUNCTION__ << name;

  Q_ASSERT(this->m_currentRequest == None);
  this->m_currentRequest = Info;
  
  int id = getCacheId(name);
  if(id < 0) return;

  postJson("{\"dto\":{\"data\":{\"c\":2,\"m\":\"\",\"d\":\"" + 
	   QString::number(id) + 
	   "\"},\"ut\":\"" + this->m_userToken + "\"}}");
}

void GcBrowser::processRequestDetail(const QString &name) {
  qDebug() << __PRETTY_FUNCTION__ << name;

  // assume that an info request has been processed before for
  // exactly this same cache
  Q_ASSERT(name == m_cache.name());

  if(m_cache.guid().isEmpty()) {
    error(tr("Unable to determine cache guid!"));

    // just eat the request for now
    emit done();
    emit next();

    return;
  }

  qDebug() << __FUNCTION__ << m_cache.guid();

  Q_ASSERT(this->m_currentRequest == None);
  this->m_currentRequest = Detail;

  // now request:
  GcNetworkRequest request;
  request.setUrl(GcUrl("/seek/cdpf.aspx?guid=" + m_cache.guid() + "&lc=5"));
  this->m_posted = QString();
  QNetworkReply *reply = this->m_manager->get(request);
  emit notifyBusy(true);

  if(reply->error()) replyFinished(reply);
}

void GcBrowser::applyChanges() {
  qDebug() << PLUGIN_NAME << __PRETTY_FUNCTION__;

  // settings have been saved by settings dialog,
  // so we can already read them again

  QSettings settings;
  settings.beginGroup("Account");
  QString name = settings.value("Name", "").toString();
  QString password = settings.value("Password", "").toString();
  settings.endGroup();

  // (re-)login in now if password or username have changed
  if((name != this->m_name) || (password != this->m_password)) 
    login(name, password);
}

void GcBrowser::createConfig(QDialog *parent, QVBoxLayout *) {
  // this plugin doesn't have its own config, but it wants to
  // know when the global settings change
  connect(parent, SIGNAL(accepted()), this, SLOT(applyChanges()));
}

// send an error message to the cache provider
void GcBrowser::error(const QString &msg) {
  if(!msg.isEmpty()) {
    if(msg.at(0) == QChar('!')) 
      emit replyError("!" + tr(PLUGIN_NAME) + ": " + QString(msg).remove(0,1));
    else
      emit replyError(tr(PLUGIN_NAME) + ": " + msg);
  } else
    emit replyError("");
}

Q_EXPORT_PLUGIN2(gcbrowser, GcBrowser);
