// Filename: cachewindow.cpp

#include <QDebug>

#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QLabel>
#include <QPainter>
#include <QKeyEvent>
#include <QPushButton>
#include <QDesktopServices>
#include <QFontInfo>

#include "htmlview.h"
#include "cachewindow.h"
#include "descriptionwindow.h"
#include "navigationwindow.h"
#include "logwindow.h"
#include "attributewindow.h"

#include "swapbox.h"

class HintWindow : public CustomWindow {
public:
  HintWindow(const Cache &cache, QWidget *parent) :
    CustomWindow("HintWindow", parent) {
    setWindowTitle(tr("Hint"));

    setCentralWidget(new HtmlView(cache.hint(), this));
  }
};

void CacheWindow::showHint() {
  HintWindow *hintWindow = new HintWindow(m_cache, this);
  hintWindow->show();
}

// the star widget consists of 5 stars used to display ratings
class StarWidget : public QLabel {
public:
  StarWidget(IconLoader *loader, const qreal rating, 
	     QWidget *parent = 0) : QLabel(parent) {
    int size = 2*font().pointSize();
    int rating2 = (2*rating)+0.5;

    // load the required pixmaps
    QPixmap *starPix = loader->load("star", size);
    QPixmap *starLightPix = loader->load("star_light", size);

    // build the "half" pixmap if required
    QPixmap *starHalfPix = NULL;
    if(rating2 & 1) {
      starHalfPix = new QPixmap(*starPix);
      QPainter halfPainter(starHalfPix);
      const QRectF half(size/2, 0, 
			size/2, loader->size());
      halfPainter.drawPixmap(half, *starLightPix, half);
    }

    // assemble pixmap and draw all five stars
    QPixmap pixmap(5*size, size);
    pixmap.fill(Qt::transparent);

    QPainter painter(&pixmap);
    QPixmap *star;
    int i;
    for(i=0;i<5;i++) {
      if(rating2 < 2*(i+1)-1)    star = starLightPix;
      else if(rating2 < 2*(i+1)) star = starHalfPix;
      else                       star = starPix;
      
      painter.drawPixmap(i*size, 0, *star);
    }

    // and finally place the pixmap in the label
    setPixmap(pixmap);
  }
};

class RatingWidget : public QWidget {
public:
  RatingWidget(IconLoader *loader, 
	       const QString &title, const qreal rating, 
	       QWidget *parent = 0) : QWidget(parent) {
    
    QHBoxLayout *layout = new QHBoxLayout;
    layout->setContentsMargins(0,0,0,0);
    layout->addWidget(new QLabel(title + ":"));
    layout->addWidget(new StarWidget(loader, rating), 0, Qt::AlignRight);
    
    setLayout(layout);
  }
};

class ContainerWidget : public QWidget {
public:
  ContainerWidget(IconLoader *loader, 
		  const QString &title, const Container container, 
		  QWidget *parent = 0) : QWidget(parent) {
    
    QHBoxLayout *layout = new QHBoxLayout;
    layout->setContentsMargins(0,0,0,0);
    layout->addWidget(new QLabel(title + ":"));

    QLabel *label = new QLabel();
    QFont font = label->font();
    label->setPixmap(*loader->load(container, 2*font.pointSize()));
    layout->addWidget(label, 0, Qt::AlignRight);
    
    setLayout(layout);
  }
};

class Label : public QLabel {
public:
  Label(const QString &text, bool bold = false, qreal fontScale = 1.0,
	QWidget *parent = 0) : QLabel(text, parent) {
    QFont font = this->font();
    font.setPointSize(fontScale*font.pointSize());
    if(bold) font.setWeight(QFont::Bold);
    this->setFont(font);
  };
};

#define HACK

// a qpushbutton with a small icon on the right
class Button : public QPushButton {
public:
  Button(const QString &text, QWidget *parent = 0) : 
    QPushButton(text, parent), m_pixmap(NULL) {
  };

  void addIcon(IconLoader *loader, const QString &name) {
#ifndef HACK
    if(!m_pixmap) {
      QMargins margins = contentsMargins();
      qDebug() << __FUNCTION__ << 
	margins.left() << margins.right();
      margins.setRight(margins.right() + iconSize().width());
      margins.setLeft(0);
      setContentsMargins(margins);
      qDebug() << __FUNCTION__ << 
	margins.left() << margins.right();
    }
#endif
    m_pixmap = loader->load("button_" + name, iconSize().width());
#ifdef HACK
    // append some spaces to the text to shift it a little bit
    // to the left
    setText(text()+"           ");
#endif
  }

#ifndef HACK // now done by the additional spaces
  QSize sizeHint() const {
    QSize size = QPushButton::sizeHint();

    // request additional space for icon
    if(m_pixmap)
      size += QSize(m_pixmap->width(),0);

    return size;
  }
#endif

  void paintEvent(QPaintEvent *event) {
    // draw original button
    QPushButton::paintEvent(event);

    if(m_pixmap) {
      // add pixmap on top
      QRect rect = contentsRect();
      QPainter painter(this);
#ifdef HACK
      painter.drawPixmap(rect.x()+rect.width()-1.3*m_pixmap->width(), 
      			 rect.y()+(rect.height()-m_pixmap->height())/2, 
			 *m_pixmap);
#else
      painter.drawPixmap(rect.x(), 
			 rect.y()+(rect.height()-m_pixmap->height())/2, 
			 *m_pixmap);
      painter.drawPixmap(rect.x()+rect.width(), 
			 rect.y()+(rect.height()-m_pixmap->height())/2, 
			 *m_pixmap);
#endif
    }
  }

private:
  QPixmap *m_pixmap;
};

void CacheWindow::openCacheUrl() {
  QDesktopServices::openUrl(m_cache.url());
}

CacheWindow::CacheWindow(const Cache &cache, 
	 LocationProvider *locationProvider, QWidget *parent) : 
  CustomWindow("CacheWindow", parent),  m_cache(cache), m_iconLoader(24),
  m_locationProvider(locationProvider) {

  setWindowTitle(cache.name());

  qDebug() << __FUNCTION__ << cache;

  QWidget *vbox = new QWidget;
  QVBoxLayout *layout = new QVBoxLayout;
  layout->setContentsMargins(0,0,0,0);
  layout->setSpacing(0);
  vbox->setLayout(layout);
  setCentralWidget(vbox);

  // ---------- general info on top -------------
  
  // ------------- title with icon -------------
  QWidget *hbox = new QWidget;
  QHBoxLayout *hlayout = new QHBoxLayout;
  hbox->setLayout(hlayout);

  Label *label = new Label("<h2>"+cache.description()+"</h2>");

  QFontInfo fontInfo(label->font());
  hlayout->addWidget(m_iconLoader.newWidget(cache, 
				 2*fontInfo.pixelSize()));

  label->setWordWrap(true);
  // hcenter sometimes causes broken word wrap
  hlayout->addWidget(label, 1 /*, Qt::AlignHCenter*/ );

  layout->addWidget(hbox);

  // if there's a short description, then use a swapbox to place
  // the description in landscape in the left window half

  // ------------- short description -------------
  SwapBox *sbox = NULL;
  HtmlView *htmlView = NULL;
  if(cache.shortDescription().isSet()) {
    sbox = new SwapBox();

    // html view below
    htmlView = new HtmlView(cache.shortDescription());
    connect(this, SIGNAL(zoomInPressed()), htmlView, SLOT(zoomIn()));
    connect(this, SIGNAL(zoomOutPressed()), htmlView, SLOT(zoomOut()));

    sbox->addWidget(htmlView, 1);
    layout->addWidget(sbox);

    // create a new vbox for the right (button) half
    vbox = new QWidget;
    layout = new QVBoxLayout;
    layout->setContentsMargins(0,0,0,0);
    layout->setSpacing(0);
    vbox->setLayout(layout);
    sbox->addWidget(vbox);
  }

  // ------------- button to open long description -------------
  if(cache.longDescription().isSet()) {
    Button *button = new Button(tr("Description"));
    button->addIcon(&m_iconLoader, "new_window");    

    connect(button, SIGNAL(clicked()), this, SLOT(showDescription()));

    layout->addWidget(button);
  }

  // ------------- button to open navigation window -------------
  if(cache.coordinate().isValid()) {
    Button *button = new Button(tr("Navigation"));
    button->addIcon(&m_iconLoader, "new_window");    
    
    connect(button, SIGNAL(clicked()), this, SLOT(showNavigation()));
    
    layout->addWidget(button);
  }

  // ------------- button to display hint -------------
  if(cache.hint().isSet()) {
    Button *button = new Button(tr("Hint"));
    button->addIcon(&m_iconLoader, "new_window");    

    connect(button, SIGNAL(clicked()), this, SLOT(showHint()));

    layout->addWidget(button);
  }

  // ------------- button to display logs -------------
  if(cache.logs().size()) {
    Button *button = new Button(tr("Logs"));
    button->addIcon(&m_iconLoader, "new_window");    

    connect(button, SIGNAL(clicked()), this, SLOT(showLogs()));

    layout->addWidget(button);
  }

  // ------------- button to display attributes -------------
  if(cache.attributes().size()) {
    Button *button = new Button(tr("Attributes"));
    button->addIcon(&m_iconLoader, "new_window");    

    connect(button, SIGNAL(clicked()), this, SLOT(showAttributes()));

    layout->addWidget(button);
  }

  // ------------- button to open web link -------------
  if(cache.url().isValid()) {
    Button *button = new Button(tr("Open in browser"));
    button->addIcon(&m_iconLoader, "browser");    

    connect(button, SIGNAL(clicked()), this, SLOT(openCacheUrl()));

    layout->addWidget(button);
  }

  QWidget *stretchWidget = new QWidget;
  if(sbox) sbox->setPortraitOnly(stretchWidget);
  layout->addWidget(stretchWidget, 1);
}

CacheWindow::~CacheWindow() {
  qDebug() << __FUNCTION__;
}

void CacheWindow::showDescription() {
  DescriptionWindow *descriptionWindow = 
    new DescriptionWindow(m_cache, this);

  descriptionWindow->show();
}

void CacheWindow::showNavigation() {
  NavigationWindow *navigationWindow = 
    new NavigationWindow(m_cache, m_locationProvider, this);

  navigationWindow->show();
}

void CacheWindow::showLogs() {
  LogWindow *logWindow = 
    new LogWindow(m_cache.logs(), this);

  logWindow->show();
}

void CacheWindow::showAttributes() {
  AttributeWindow *attributeWindow = 
    new AttributeWindow(m_cache.attributes(), this);

  attributeWindow->show();
}
