#include <QDebug>
#include <QtPlugin>
#include <QSettings>


#include <QDialog>
#include <QGroupBox>
#include <QVBoxLayout>
#include <QLabel>
#include <QNetworkCookie>

#include "gclive.h"
#include "gcparser.h"
#include "gchtmlparser.h"

#define PLUGIN_NAME "GcLive"

// The GcUrl class automatically creates urls for geocaching.com
class GcUrl : public QUrl {
public:
  GcUrl(const QString &url) : 
    QUrl("http://www.geocaching.com" + url) {}
};

// the GcNetworkRequest pre-sets some header values
class GcNetworkRequest : public QNetworkRequest {
public:
  GcNetworkRequest() : QNetworkRequest() {
    setRawHeader("User-Agent", "CacheMe " PLUGIN_NAME " Plugin");
  }
};

// return a string containing everything, the string str contains 
// between the first occurances of the strings start and end
QString GcLive::subString(const QString &str, const QString &start, const QString &end) {
		
  int is = str.indexOf(start) + start.length();
  int ie = str.indexOf(end, is);

  if (is == start.length()-1 || ie == -1) 
    return NULL;

  return str.mid(is, ie-is);  
}

void GcLive::handleViewStateField(const QString &str) {
  QString lvCount = subString(str,  "__VIEWSTATEFIELDCOUNT\" value=\"", "\" />");
  int count = lvCount.isNull()?1:lvCount.toInt();

  QString lvResult;

  for (int i=0; i<count; i++) { 
    QString key = "__VIEWSTATE"+(!i?"":QString::number(i));
    
    QString lvPart = subString(str, key+"\" value=\"", "\" />");
    if(lvPart.isNull()) {
      qDebug() << __FUNCTION__ << "Didn't get a Viewstate!";
      return;
    }

    lvResult.append(i!=0?"&":"").append(key).append("=").append(QUrl::toPercentEncoding(lvPart));
  }

  m_lastViewState = lvResult;
}

// ------------ handling of user token ---------------

bool GcLive::decodeUserToken(const QString &str) {
  this->m_userToken = subString(str,  "userToken = '", "';");
  return !this->m_userToken.isNull();
}

void GcLive::requestUserToken() {
  GcNetworkRequest request;
  request.setUrl(GcUrl("/map/default.aspx"));
  QNetworkReply *reply = this->m_manager->get(request);
  emit notifyBusy(true);
  this->m_state = RequestedUserToken;

  if(reply->error()) replyFinished(reply);
}

// --------------- log into gc.com --------------------

void GcLive::prepareLogin() {
  if(loginIsValid()) {
    GcNetworkRequest request;
    request.setUrl(GcUrl("/login/"));
    QNetworkReply *reply = this->m_manager->get(request);    
    emit notifyBusy(true);
    this->m_state = SentLoginGET;

    if(reply->error()) replyFinished(reply);
  } else {
    error(tr("No valid login specified, cache positions will be inaccurate!"));
    requestUserToken();
  }
}

void GcLive::doLogin() {

  GcNetworkRequest request;
  request.setUrl(GcUrl("/login/"));
  
  QString data = m_lastViewState;
  data += "&ctl00$ContentBody$cookie=on";
  data += "&ctl00$ContentBody$Button1=Login";
  data += "&ctl00$ContentBody$myUsername=" + 
    QUrl::toPercentEncoding(m_loginName);
  data += "&ctl00$ContentBody$myPassword=" + 
    QUrl::toPercentEncoding(m_loginPassword);
 
  m_manager->post(request, data.toUtf8());

  emit notifyBusy(true);
  this->m_state = SentLoginPOST;
}

GcLive::GcLive() : m_initialized(false), m_cacheList(PLUGIN_NAME), 
		   m_state(Idle), m_currentRequest(None) {
  qDebug() << __FUNCTION__;

  // try to get credentials from qsettings
  QSettings settings;
  settings.beginGroup(PLUGIN_NAME);
  m_loginName = settings.value("Name", "").toString();
  m_loginPassword = settings.value("Password", "").toString();
  settings.endGroup();

  // setup network manager and listen for its replies
  this->m_manager = new QNetworkAccessManager(this);

  // try to load stored cookies
  QList<QNetworkCookie> cookies;
  settings.beginGroup(PLUGIN_NAME);
  settings.beginGroup("Cookies");
  foreach(QString key, settings.allKeys()) 
    cookies.append(QNetworkCookie(key.toAscii(), settings.value(key).toByteArray()));
  settings.endGroup();
  settings.endGroup();
  this->m_manager->cookieJar()->setCookiesFromUrl(cookies, GcUrl(""));

  connect(this->m_manager, SIGNAL(finished(QNetworkReply*)),
	  this, SLOT(replyFinished(QNetworkReply*)));
}

void GcLive::init() {
  prepareLogin();
}

GcLive::~GcLive() {
  qDebug() << __FUNCTION__;

  delete this->m_manager;
}

QString GcLive::name() {
  return "gclive";
}

void GcLive::login(const QString &name, const QString &password) {
  this->m_loginName = name;
  this->m_loginPassword = password;

  prepareLogin();
}

bool GcLive::loginIsValid() {
  return(!m_loginName.isEmpty() && 
	 !m_loginPassword.isEmpty());
}

bool GcLive::busy() {
  // initialized and no pending login request
  return(!m_initialized || m_state != Idle);
}

void GcLive::replyFinished(QNetworkReply *reply) {
  if(reply->error())
    error(tr("Error") + ": " + reply->errorString());

  // invoke appropriate decoder
  if(reply->isFinished()) {
    QString allData = QString::fromUtf8(reply->readAll());

    handleViewStateField(allData);

    qDebug() << __FUNCTION__ << "reply in state " << m_state;

    if(m_state == RequestedUserToken) {
      if(decodeUserToken(allData))
	qDebug() << __FUNCTION__ << "UserToken is" << this->m_userToken;

      // if m_initialized is already true, then this is a 
      // (re-)login attempt due to user interaction
      if(m_initialized)
	emit settingsChanged();

      // now everything is initialized
      m_initialized = true;
      m_state = Idle;

    } else if(this->m_state == SentLoginGET || this->m_state == SentLoginPOST) {
      // check if we are successfully logged in
      if (allData.contains("You are logged in as")) {
	qDebug() << __FUNCTION__ << "login successful";

	QSettings settings;
	settings.beginGroup(PLUGIN_NAME);
	settings.beginGroup("Cookies");
	QList<QNetworkCookie> cookies = 
	  this->m_manager->cookieJar()->cookiesForUrl(GcUrl(""));

	// save all non-session cookies
	foreach(QNetworkCookie cookie, cookies) 
	  if(!cookie.isSessionCookie()) 
	    settings.setValue(cookie.name(), cookie.value());

	settings.endGroup();
	settings.endGroup();

	requestUserToken();
      } else {
	if(this->m_state == SentLoginGET) {
	  qDebug() << __FUNCTION__ << "we are not already logged in, doing so ...";
	  doLogin();
	} else if(this->m_state == SentLoginPOST) {
	  qDebug() << __FUNCTION__ << "login failed";
	  if(!reply->error()) {
	    error("");  // clear error to make sure more messages are displayed
	    error(tr("Login failed!"));
	  }
	  requestUserToken();
	}
      }
    } else {
      qDebug() << __FUNCTION__ << "reply";
      
      // initialization is finished, any reply received now should
      // be a reply to a real request
      CurrentRequest type = this->m_currentRequest;
      this->m_currentRequest = None;

      emit done();

      if(!reply->error()) {
	GcParser gcParser;
	GcHtmlParser gcHtmlParser;
	
	switch(type) {
	case Overview: 
	  // start parser (and tell him whether we tried to load a "big" area)
	  if(gcParser.decodeOverview(allData, m_cacheList, m_bigArea))
	    emit replyOverview(m_cacheList);
	  else
	    error(gcParser.error());
	  break;
	  
	case Info: 
	  if(gcParser.decodeInfo(allData, m_cache))
	    emit replyInfo(m_cache);
	  else
	    error(gcParser.error());
	  break;
	  
	case Detail: 
	  if(gcHtmlParser.decode(allData, m_cache)) 
	    emit replyDetail(m_cache);
	  else
	    error(gcHtmlParser.error());
	  break;
	  
	default:
	  qDebug() << __FUNCTION__ << "unknown request type" << type;
	  Q_ASSERT(0);
	  break;
	}
      }
    }
  }

  qDebug() << __FUNCTION__ << "next, busy = " << busy();
  if(!busy()) 
    emit next();

  emit notifyBusy(false);
}

void GcLive::postJson(const QString &post) {
  // build and send request
  GcNetworkRequest request;
  request.setUrl(GcUrl("/map/default.aspx/MapAction"));
  request.setRawHeader("Accept", "application/json");
  request.setRawHeader("Content-Type", "application/json; charset=utf-8");

  m_manager->post(request, post.toUtf8());
  emit notifyBusy(true);
}

void GcLive::processRequestOverview(const QGeoBoundingBox &area) {
  qDebug() << __PRETTY_FUNCTION__;

  Q_ASSERT(this->m_currentRequest == None);
  this->m_currentRequest = Overview;
  
  // check if bounding box doesn't exceed limits (40km diagonally)
  this->m_bigArea = area.topLeft().distanceTo(area.bottomRight()) > 40000;
  
  // build request string
  postJson("{\"dto\":{\"data\":{\"c\":1,\"m\":\"\",\"d\":\"" + 
	   QString::number(area.topLeft().latitude()) + "|" + 
	   QString::number(area.bottomRight().latitude()) + "|" + 
	   QString::number(area.bottomRight().longitude()) + "|" + 
	   QString::number(area.topLeft().longitude()) + 
	   "\"},\"ut\":\"" + this->m_userToken + "\"}}");
}

int GcLive::getCacheId(const QString &name) {
  int id = -1;

  // try to find matching cache id
  QList<Cache>::const_iterator i;
  for( i = m_cacheList.begin(); i != m_cacheList.end(); ++i ) {
    if(i->name() == name) {
      id = i->id();
      m_cache = *i;
    }
  }

  if(id < 0) {
    // remove current request and continue with the next one
    emit done();
    emit next();

    error(tr("Unable to determine cache id!"));
  }

  return id;
}

void GcLive::processRequestInfo(const QString &name) {
  qDebug() << __PRETTY_FUNCTION__ << name;

  Q_ASSERT(this->m_currentRequest == None);
  this->m_currentRequest = Info;
  
  int id = getCacheId(name);
  if(id < 0) return;

  postJson("{\"dto\":{\"data\":{\"c\":2,\"m\":\"\",\"d\":\"" + 
	   QString::number(id) + 
	   "\"},\"ut\":\"" + this->m_userToken + "\"}}");
}

void GcLive::processRequestDetail(const QString &name) {
  qDebug() << __PRETTY_FUNCTION__ << name;

  // assume that an info request has been processed before for
  // exactly this same cache
  Q_ASSERT(name == m_cache.name());

  if(m_cache.guid().isEmpty()) {
    error(tr("Unable to determine cache guid!"));

    // just eat the request for now
    emit done();
    emit next();

    return;
  }

  qDebug() << __FUNCTION__ << m_cache.guid();

  Q_ASSERT(this->m_currentRequest == None);
  this->m_currentRequest = Detail;

  // now request:
  GcNetworkRequest request;
  request.setUrl(GcUrl("/seek/cdpf.aspx?guid=" + m_cache.guid() + "&lc=5"));
  QNetworkReply *reply = this->m_manager->get(request);
  emit notifyBusy(true);

  if(reply->error()) replyFinished(reply);
}

void GcLive::applyChanges() {
  qDebug() << __PRETTY_FUNCTION__;

  QSettings settings;
  settings.beginGroup(PLUGIN_NAME);
  settings.setValue("Name", m_name->text());
  settings.setValue("Password", m_password->text());
  settings.endGroup();

  // (re-)login in now
  login(m_name->text(), m_password->text());
}

void GcLive::createConfig(QDialog *parent, QVBoxLayout *box) {
  QGroupBox *groupBox = new QGroupBox(tr("GcLive"));
  QVBoxLayout *vbox = new QVBoxLayout;

  QSettings settings;
  settings.beginGroup(PLUGIN_NAME);

  /* -------------- user name entry --------------- */
  vbox->addWidget(new QLabel(tr("Username") + ":"));
  m_name = new QLineEdit(settings.value("Name", "").toString());
  vbox->addWidget(m_name);

  /* -------------- password entry --------------- */
  vbox->addWidget(new QLabel(tr("Password") + ":"));
  m_password = new QLineEdit(settings.value("Password", "").toString());
  m_password->setEchoMode(QLineEdit::PasswordEchoOnEdit);
  vbox->addWidget(m_password);

  settings.endGroup();

  groupBox->setLayout(vbox);
  box->addWidget(groupBox);

  connect(parent, SIGNAL(accepted()), this, SLOT(applyChanges()));
}

// send an error message to the cache provider
void GcLive::error(const QString &msg) {
  if(!msg.isEmpty())
    emit replyError(tr(PLUGIN_NAME) + ": " + msg);
  else
    emit replyError("");
}

QObject *GcLive::getObject() {
  return this;
}

Q_EXPORT_PLUGIN2(gclive, GcLive);
