#include <QDebug>
#include <QTimer>
#include <QPainter>
#include <QApplication>
#include <QDir>
#include <math.h>

#include "graphicscompass.h"

QString GraphicsCompass::assembleFilename(const QStringList &parts) {
  Q_ASSERT(parts.size() > 1);

  QString fileName;

  QStringList::const_iterator i = parts.begin();
  if(*i == ".") {
    fileName = QApplication::applicationDirPath();
    i++;
  }

  for(;i!=parts.end();i++)
    fileName += QDir::separator() + *i;

  return fileName;
}

bool GraphicsCompass::load(QSvgRenderer *renderer, const QString &name) {
  // try local filename fist
  QString full = assembleFilename(QStringList(".")<<"data"<<"icons"<< name+".svg");
  QFile file(full);

  // local file does not exist, try system wide file
  if(!file.exists()) {
#ifdef DATADIR
    full = assembleFilename(QStringList(QDir::toNativeSeparators(DATADIR))
			    << APPNAME << "icons" << name+".svg");
    QFile file(full);
    if(!file.exists()) {
      qWarning() << __FUNCTION__ << "Unable to find file for " << name;
      return false;
    }
#else
    qWarning() << __FUNCTION__ << "Unable to find file for " << name;
    return false;
#endif
  }

  renderer->load(full);
  return true;
}

GraphicsCompass::GraphicsCompass(const Cache &cache, 
		 QGraphicsItem *parent, Qt::WindowFlags wFlags) : 
  QGraphicsWidget(parent, wFlags), m_timer(NULL), m_cache(cache), 
  m_bgPix(NULL) {

  qDebug() << __FUNCTION__;

  setAutoFillBackground(true);
  
  m_roseRenderer = new QSvgRenderer();
  load(m_roseRenderer, "compass_rose");
  m_needleRenderer = new QSvgRenderer();
  load(m_needleRenderer, "compass_needle");
  m_backgroundRenderer = new QSvgRenderer();
  load(m_backgroundRenderer, "compass_background");
  m_cacheRenderer = new QSvgRenderer();
  load(m_cacheRenderer, cache.typeIconFile());

  m_roseAngle = -200.0;
  m_needleAngle = 0;   // angle between rose and needle
  m_roseSpeed = 0;
}

GraphicsCompass::~GraphicsCompass() {
  qDebug() << __FUNCTION__;
  delete m_roseRenderer;
  delete m_needleRenderer;
  delete m_backgroundRenderer;
  if(m_timer) delete m_timer;
  if(m_bgPix) delete m_bgPix;
}

void GraphicsCompass::paint(QPainter *painter, 
	    const QStyleOptionGraphicsItem *, QWidget *) {

  // determine square size of compass
  int s = size().width()>size().height()?size().height():size().width();

  // check if we have a matching buffered background, delete it if
  // size doesn't match
  if(m_bgPix && (m_bgPix->size() != QSize(s,s))) {
    delete m_bgPix;
    m_bgPix = NULL;
  }

  // create background pixmap if we don't have one
  if(!m_bgPix) {
    m_bgPix = new QPixmap(s, s);
    m_bgPix->fill(Qt::transparent);

    QPainter bgPainter(m_bgPix);
    bgPainter.setRenderHint(QPainter::Antialiasing, true);
    m_backgroundRenderer->render(&bgPainter);
  }
  
  painter->setRenderHint(QPainter::Antialiasing, true);
  painter->drawPixmap(QPoint((size().width()-s)/2,
			     (size().height()-s)/2), *m_bgPix);
  
  if(m_roseAngle < -180.0) return;

  // rotate painter around center
  painter->translate( size().width()/2, size().height()/2 );
  painter->rotate(m_roseVAngle);
  painter->translate( -size().width()/2, -size().height()/2 );
  
  m_roseRenderer->render(painter, QRect(QPoint((size().width()-s)/2,
					       (size().height()-s)/2), 
					QSize(s,s)));

  painter->translate( size().width()/2, size().height()/2 );
  painter->rotate(m_needleAngle);
  painter->translate( -size().width()/2, -size().height()/2 );
  
  m_needleRenderer->render(painter, QRect(QPoint((size().width()-s)/2,
					       (size().height()-s)/2), 
					QSize(s,s)));

  m_cacheRenderer->render(painter, QRect(QPoint((size().width()-s/5)/2,
						(size().height()-10*s/14)/2),
			  QSize(s/5, s/5)));  
}

void GraphicsCompass::timerExpired() {
  // generate rosevangle from roseangle and rosespeed
  qreal diff = (m_roseAngle - m_roseVAngle);
  while(diff >  180.0) diff -= 180.0;
  while(diff < -180.0) diff += 180.0;

  m_roseSpeed += diff/5;
  m_roseSpeed *= 0.3;

  if(m_roseSpeed > 0.1 || m_roseSpeed < -0.1) {
    m_roseVAngle += m_roseSpeed;
    update();
  }
}

void GraphicsCompass::setPosition(const QGeoPositionInfo &pos) {
  if(pos.isValid() && pos.coordinate().isValid()) {
    if(pos.hasAttribute(QGeoPositionInfo::Direction)) {
      bool start = m_roseAngle < -180.0; 
	
      m_roseAngle = -pos.attribute(QGeoPositionInfo::Direction);

      // make sure angle is in -180..180 range
      while(m_roseAngle >  180.0) m_roseAngle -= 180.0;
      while(m_roseAngle < -180.0) m_roseAngle += 180.0;

      if(start) {
	m_roseVAngle = m_roseAngle;

	//  a timer for smooth animations
	m_timer = new QTimer(this);
	connect(m_timer, SIGNAL(timeout()), this, SLOT(timerExpired()));
	m_timer->start(1000/RATE);
      }

    }

    m_needleAngle = pos.coordinate().azimuthTo(m_cache.coordinate());
    update();
  }
}
