#include <QDebug>

#include "cache.h"

//------------------- Description -------------------

Description::Description() {
  this->m_isHtml = false;
}

void Description::set(bool isHtml, const QString &text) {
  this->m_text = text;
  this->m_isHtml = isHtml;
}

QString Description::text() const {
  return this->m_text;
}

bool Description::isHtml() const {
  return this->m_isHtml;
}

bool Description::isSet() const {
  return !this->m_text.isEmpty();
}

QDebug operator<<(QDebug dbg, const Description &d) {
  dbg.nospace() << d.text();
  dbg.nospace() << "(html=" << d.isHtml() << ")";
  return dbg;
}

//------------------- Tribool -------------------
Tribool::Tribool() {
  // no need to set m_value
  this->m_set = false;
}

void Tribool::set(bool val) {
  this->m_set = true;
  this->m_value = val;
}

bool Tribool::value() const {
  Q_ASSERT(this->m_set);
  return this->m_value;
}

bool Tribool::isSet() const {
  return this->m_set;
}

QDebug operator<<(QDebug dbg, const Tribool &t) {
  if(!t.isSet()) dbg.nospace() << "?";
  else           dbg.nospace() << t.value();
  return dbg;
}

//------------------- Rating -------------------
Rating::Rating() {
  this->m_value = 0;
}

void Rating::set(qreal val) {
  this->m_value = val*2;
}

qreal Rating::value() const {
  Q_ASSERT(this->m_value);
  return this->m_value/2.0;
}

bool Rating::isSet() const {
  return(this->m_value != 0);
}

QString Rating::toString() const {
  QString ret;

  if(!isSet()) return("?");

  int i, j = 2*value();
  for(i=2;i<=10;i+=2) {
    if(j < i-1)    ret += "-";
    else if(j < i) ret += "+";
    else           ret += "*";
  }
  return ret;
}

QDebug operator<<(QDebug dbg, const Rating &t) {
  dbg.nospace() << t.toString();
  return dbg;
}

//------------------- Container -------------------

Container::Container() : m_value(ContainerUnknown) {
}

void Container::set(Type type) {
  m_value = type;
}

Container::Type Container::get() const {
  return m_value;
}

QString Container::toString() const {
  int i;

  const struct { Type type; QString str; } tags[] = {
    { ContainerRegular,      "Regular" }, { ContainerSmall,   "Small" },
    { ContainerMicro,          "Micro" }, { ContainerOther,   "Other" },
    { ContainerNotChosen, "Not Chosen" }, { ContainerLarge,   "Large" },
    { ContainerVirtual,      "Virtual" }, { ContainerUnknown,     "?" }
  };

  for(i=0;(tags[i].type!=ContainerUnknown) && (tags[i].type != m_value);i++); 
     
  return tags[i].str;
}

QString Container::iconFile() const {
  int i;

  const struct { Type type; QString str; } tags[] = {
    { ContainerRegular,      "regular" }, { ContainerSmall,   "small" },
    { ContainerMicro,          "micro" }, { ContainerOther,   "other" },
    { ContainerNotChosen, "not_chosen" }, { ContainerLarge,   "large" },
    { ContainerVirtual,      "virtual" }, { ContainerUnknown, "other" }
  };

  for(i=0;(tags[i].type!=ContainerUnknown) && (tags[i].type != m_value);i++); 
     
  return "cache_size_" + tags[i].str;
}

bool Container::isSet() const {
  return m_value != ContainerUnknown;
}

QDebug operator<<(QDebug dbg, const Container &c) {
  dbg.nospace() << c.toString();
  return dbg;
}

//------------------- Cache -------------------

Cache::Cache() :
  m_type(TypeUnknown),
  m_id(0) {
}

Cache::~Cache() {
}

// set "owned" attribute
void Cache::setOwned(bool owned) {
  this->m_owned.set(owned);
}

Tribool Cache::owned() const {
  return this->m_owned;
}

void Cache::setFound(bool found) {
  this->m_found.set(found);
}

Tribool Cache::found() const {
  return this->m_found;
}

void Cache::setAvailable(bool available) {
  this->m_available.set(available);
}

Tribool Cache::available() const {
  return this->m_available;
}

void Cache::setId(int id) {
  this->m_id = id;
}

int Cache::id() const {
  return this->m_id;
}

void Cache::setType(Type type) {
  this->m_type = type;
}

Cache::Type Cache::type() const {
  return this->m_type;
}

QString Cache::typeString() const {
  int i;

  struct { Type type; QString str; } tags[] = {
    { TypeTraditional, "Traditional" },   { TypeMulti,       "Multi" },
    { TypeMystery,     "Mystery" },       { TypeVirtual,     "Virtual" },
    { TypeWebcam,      "Webcam" },        { TypeEvent,       "Event" },
    { TypeLetterbox,   "Letterbox" },     { TypeEarthcache,  "Earthcache" },
    { TypeWherigo,     "Wherigo" },       { TypeMegaEvent,   "Mega-Event" },
    { TypeCito,   "Cache In Trash Out" }, { TypeUnknown,     "?" }
  };

  for(i=0;(tags[i].type!=TypeUnknown) && (tags[i].type != m_type);i++); 
     
  return tags[i].str;
}

QString Cache::typeIconFile() const {
  int i;

  const struct { Type type; QString str; } tags[] = {
    { TypeTraditional, "traditional" }, { TypeMulti,       "multi" },
    { TypeMystery,     "mystery" },     { TypeVirtual,     "virtual" },
    { TypeWebcam,      "webcam" },      { TypeEvent,       "event" },
    { TypeLetterbox,   "letterbox" },   { TypeEarthcache,  "earthcache" },
    { TypeWherigo,     "wherigo" },     { TypeMegaEvent,   "megaevent" },
    { TypeCito,        "cito" },        { TypeUnknown,     "<unknown>" }
  };

  for(i=0;(tags[i].type!=TypeUnknown) && (tags[i].type != m_type);i++); 
     
  return "cache_type_" + tags[i].str;
}

QString Cache::overlayIconFile() const {
  if(this->m_available.isSet() && !this->m_available.value())
    return "cache_type_ovl_unavailable";
  
  if(this->m_owned.isSet() && this->m_owned.value())
    return "cache_type_ovl_mine";

  if(this->m_found.isSet() && this->m_found.value())
    return "cache_type_ovl_found";

  return NULL;
}


void Cache::setDifficulty(qreal val) {
  this->m_difficulty.set(val);
}

Rating Cache::difficulty() const {
  return this->m_difficulty;
}

void Cache::setTerrain(qreal val) {
  this->m_terrain.set(val);
}

Rating Cache::terrain() const {
  return this->m_terrain;
}

void Cache::setContainer(Container::Type val) {
  this->m_container.set(val);
}

Container Cache::container() const {
  Q_ASSERT(this->m_container.isSet());
  return this->m_container;
}

void Cache::setGuid(const QString &guid) {
  this->m_guid = guid;
}

QString Cache::guid() const {
  return this->m_guid;
}

void Cache::setOwner(const QString &owner) {
  m_owner = owner;
}

QString Cache::owner() const {
  if(m_owner.isNull() || m_owner.isEmpty())
    return "?";

  return m_owner;
}

void Cache::setDateOfPlacement(const QDate &date) {
  m_dateOfPlacement = date;
}

QDate Cache::dateOfPlacement() const {
  return m_dateOfPlacement;
}

QString Cache::dateOfPlacementString() const {
  if(m_dateOfPlacement.isNull() || !m_dateOfPlacement.isValid())
    return "?";

  return m_dateOfPlacement.toString();
}

void Cache::setShortDescription(const Description &shortDescription) {
  m_shortDescription = shortDescription;
}

Description Cache::shortDescription() const {
  return m_shortDescription;
}

void Cache::setLongDescription(const Description &longDescription) {
  m_longDescription = longDescription;
}

Description Cache::longDescription() const {
  return m_longDescription;
}

void Cache::setHint(const Description &hint) {
  m_hint = hint;
}

Description Cache::hint() const {
  return m_hint;
}

QDebug operator<<(QDebug debug, const Cache &c) {
  debug.nospace() << "Cache(" << c.name() << 
    ", " << c.coordinate() << ", " << c.description() << ",\n  ";
  debug.nospace() << "placed at " << c.dateOfPlacementString() << " by " << c.owner() << ",\n  ";
  if(c.m_shortDescription.isSet()) debug.nospace() << c.m_shortDescription  << ",\n  ";
  if(c.m_longDescription.isSet()) debug.nospace() << c.m_longDescription  << ",\n  ";
  if(c.url().isValid()) debug.nospace() << "url=" << c.url()  << ",\n  ";
  if(!c.guid().isEmpty()) debug.nospace() << "guid=" << c.guid() << ",\n  ";
  debug.nospace() << "diffculty=" << c.m_difficulty << 
    ", terrain=" << c.m_terrain << ", container=" << c.m_container << ",\n  ";
  debug.nospace() << 
    "type=" << c.typeString() << ", id=" << c.id() << 
    ", found=" << c.m_found << ", owned=" << c.m_owned << 
    ", available=" << c.m_available << ")";
  return debug;
}

CacheList::CacheList(const QString &name) {
  this->m_name = (QString)name;
}

CacheList::~CacheList() {
}

void CacheList::append(Cache cache) {
  QList<Cache>::append(cache);
}

int CacheList::size() const {
  return QList<Cache>::size();
}

Cache CacheList::at(int index) {
  return QList<Cache>::at(index);
}

QString CacheList::name() {
  return m_name;
}

QList<Cache>::const_iterator CacheList::begin() const {
  return QList<Cache>::begin();
}

QList<Cache>::const_iterator CacheList::end() const {
  return QList<Cache>::end();
}

void CacheList::clear() {
  qDebug() << __FUNCTION__;

  QList<Cache>::clear();
}
