#include <QDebug>

#include "gchtmlparser.h"
#include "QSgml.h"

GcHtmlParser::GcHtmlParser() {
  qDebug() << __PRETTY_FUNCTION__;
}

GcHtmlParser::~GcHtmlParser() {
  qDebug() << __PRETTY_FUNCTION__;
}

void GcHtmlParser::dump(const QSgmlTag *tag) {
  qDebug() << __FUNCTION__ << 
    tag->Type << tag->Name << tag->Value;
  
  foreach(QString key, tag->Attributes.keys()) 
    qDebug() << key << tag->Attributes.value(key);

}

void GcHtmlParser::dump(const QSgmlTag::QSgmlTaglist &list, int level) {
  if(!list.size()) return;

  for(int i=0;i<list.size();i++) {
    qDebug() << __FUNCTION__ << level << 
      list[i]->Type << list[i]->Name << list[i]->Value;

    foreach(QString key, list[i]->Attributes.keys()) 
      qDebug() << key << list[i]->Attributes.value(key);

    dump(list[i]->Children, level+1);
  }
}

static bool isDiv(QSgmlTag *tag, const QString &str) {
  return(!tag->Name.compare("div", Qt::CaseInsensitive) && 
	 tag->Type == QSgmlTag::eStartTag &&
	 tag->hasAttribute("class") && 
	 !tag->getArgValue("class").compare(str, Qt::CaseInsensitive));
}

static bool isTag(QSgmlTag *tag, const QString &str) {
  return(!tag->Name.compare(str, Qt::CaseInsensitive) && 
	 tag->Type == QSgmlTag::eStartTag);
}

// return only the pure text below the current tag
QString GcHtmlParser::getText(const QSgmlTag::QSgmlTaglist &list) {
  QString str;
  foreach(QSgmlTag *tag, list) {
    if(tag->Type == QSgmlTag::eCdata) str += tag->Value;
    else                              str += getText(tag->Children);
  }
  return str;
}

// return the entire html code below current tag
QString GcHtmlParser::getHtml(const QSgmlTag::QSgmlTaglist &list) {
  QString str;
  foreach(QSgmlTag *tag, list) {
    if(tag->Type == QSgmlTag::eCdata) str += tag->Value;
    else {
      QString subStr =  getHtml(tag->Children);
      str += "<" + tag->Name;

      foreach(QString key, tag->Attributes.keys()) 
	str += " " + key + "=\"" + tag->Attributes.value(key) + "\"";

      if(!subStr.isEmpty())
	str += ">" + getHtml(tag->Children) + "</" + tag->Name + ">";
      else
	str += "/>";
    }
  }
  return str;
}

QString GcHtmlParser::searchForItemHeaderH2(const QSgmlTag::QSgmlTaglist &list) {
  foreach(QSgmlTag *tag, list) {
    if(isTag(tag, "h2"))
      return getText(tag->Children).trimmed();
    else {
      QString str = searchForItemHeaderH2(tag->Children);
      if(!str.isEmpty()) return str;
    }
  }
  return NULL;
}

QString GcHtmlParser::searchForItemHeader(const QSgmlTag::QSgmlTaglist &list) {
  foreach(QSgmlTag *tag, list) {
    QString header = isDiv(tag, "item-header")? 
      searchForItemHeaderH2(tag->Children):
      searchForItemHeader(tag->Children);

    if(!header.isEmpty()) return header;
  }
  return NULL;
}

QString GcHtmlParser::searchForItemContentText(const QSgmlTag::QSgmlTaglist &list) {
  foreach(QSgmlTag *tag, list) {
    QString content = isDiv(tag, "item-content")?
      getHtml(tag->Children).trimmed():
      searchForItemContentText(tag->Children);
    
    if(!content.isEmpty()) return content;
  }
  return NULL;
}

QString GcHtmlParser::searchForHintEncrypted(const QSgmlTag::QSgmlTaglist &list) {
  foreach(QSgmlTag *tag, list) {
    QString hint = isDiv(tag, "hint-encrypted")?
      getHtml(tag->Children).trimmed():
      searchForHintEncrypted(tag->Children);
    
    if(!hint.isEmpty()) return hint;
  }
  return NULL;
}

void GcHtmlParser::searchForItem(const QSgmlTag::QSgmlTaglist &list, 
				 Cache &cache) {
  foreach(QSgmlTag *tag, list) {
    if(isDiv(tag, "item")) {
      QString header = searchForItemHeader(tag->Children);

      if(!header.isEmpty()) {
	qDebug() << __FUNCTION__ << "found" << header;
	
	/* now search for matching "item-content" */
	if(!header.compare("Short Description", Qt::CaseInsensitive)) {
	  Description description;
	  description.set(true, searchForItemContentText(tag->Children));
	  if(description.isSet()) cache.setShortDescription(description);

	} else if(!header.compare("Long Description", Qt::CaseInsensitive)) {
	  Description description;
	  description.set(true, searchForItemContentText(tag->Children));
	  if(description.isSet()) cache.setLongDescription(description);

	} else if(!header.compare("Additional Hints", Qt::CaseInsensitive)) {
	  Description hint;
	  hint.set(true, searchForHintEncrypted(tag->Children));
	  if(hint.isSet()) cache.setHint(hint);
	}
      }
    } else
      searchForItem(tag->Children, cache);
  }
}

bool GcHtmlParser::decode(const QByteArray &data, Cache &cache) {
  m_error = QString();
  
  QSgml html(QString::fromUtf8(data)); 
  QList<QSgmlTag*> body;
  html.getElementsByName("body", &body);
  if(body.size() > 0)
    searchForItem(body[0]->Children, cache);

  return true;
}

QString GcHtmlParser::error() const {
  return m_error;
}
