/*
 * btc-tmdb-image.c
 *
 * This file is part of butaca
 * Copyright (C) 2010 Simón Pena <spenap@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 */

#include "btc-tmdb-image.h"

G_DEFINE_TYPE (BtcTmdbImage, btc_tmdb_image, G_TYPE_OBJECT)

enum {
        PROP_0,
        PROP_TYPE,
        PROP_URL,
        PROP_SIZE,
        PROP_WIDTH,
        PROP_HEIGHT,
        PROP_ID,
};

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), BTC_TYPE_TMDB_IMAGE, BtcTmdbImagePrivate))

struct _BtcTmdbImagePrivate {
        gchar *type;
        gchar *url;
        gchar *size;
        gint width;
        gint height;
        gchar *id;
};

static const gchar*
image_fields[LAST_FIELD] = { "type", "url", "size", "id" };

static void
btc_tmdb_image_get_property (GObject *object, guint property_id,
                         GValue *value, GParamSpec *pspec)
{
        BtcTmdbImage *self = BTC_TMDB_IMAGE (object);

        switch (property_id) {
        case PROP_TYPE:
                g_value_set_string (value, self->priv->type);
                break;
        case PROP_URL:
                g_value_set_string (value, self->priv->url);
                break;
        case PROP_SIZE:
                g_value_set_string (value, self->priv->size);
                break;
        case PROP_WIDTH:
                g_value_set_int (value, self->priv->width);
                break;
        case PROP_HEIGHT:
                g_value_set_int (value, self->priv->height);
                break;
        case PROP_ID:
                g_value_set_string (value, self->priv->id);
                break;
        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        }
}

static void
btc_tmdb_image_set_property (GObject *object, guint property_id,
                         const GValue *value, GParamSpec *pspec)
{
        BtcTmdbImage *self = BTC_TMDB_IMAGE (object);

        switch (property_id) {
        case PROP_TYPE:
                btc_tmdb_image_set_imagetype (self, g_value_get_string (value));
                break;
        case PROP_URL:
                btc_tmdb_image_set_url (self, g_value_get_string (value));
                break;
        case PROP_SIZE:
                btc_tmdb_image_set_size (self, g_value_get_string (value));
                break;
        case PROP_WIDTH:
                btc_tmdb_image_set_width (self, g_value_get_int (value));
                break;
        case PROP_HEIGHT:
                btc_tmdb_image_set_height (self, g_value_get_int (value));
                break;
        case PROP_ID:
                btc_tmdb_image_set_id (self, g_value_get_string (value));
                break;
        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        }
}

static void
btc_tmdb_image_finalize (GObject *object)
{
        BtcTmdbImage *self = BTC_TMDB_IMAGE (object);

        g_free (self->priv->type);
        g_free (self->priv->url);
        g_free (self->priv->size);
        g_free (self->priv->id);

        G_OBJECT_CLASS (btc_tmdb_image_parent_class)->finalize (object);
}

static void
btc_tmdb_image_class_init (BtcTmdbImageClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (klass);

        g_type_class_add_private (klass, sizeof (BtcTmdbImagePrivate));

        object_class->get_property = btc_tmdb_image_get_property;
        object_class->set_property = btc_tmdb_image_set_property;
        object_class->finalize = btc_tmdb_image_finalize;

        g_object_class_install_property
                (object_class, PROP_TYPE,
                 g_param_spec_string ("type", "Image type",
                                 "The image type",
                                 NULL,
                                 G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

        g_object_class_install_property
                (object_class, PROP_URL,
                 g_param_spec_string ("url", "Image url",
                                 "The image URL",
                                 NULL,
                                 G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

        g_object_class_install_property
                (object_class, PROP_SIZE,
                 g_param_spec_string ("size", "Image size",
                                 "The image size",
                                 NULL,
                                 G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

        g_object_class_install_property
                (object_class, PROP_WIDTH,
                 g_param_spec_int ("width", "Image width",
                                 "The image width",
                                 0,
                                 INT_MAX,
                                 0,
                                 G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

        g_object_class_install_property
                (object_class, PROP_HEIGHT,
                 g_param_spec_int ("height", "Image height",
                                 "The image height",
                                 0,
                                 INT_MAX,
                                 0,
                                 G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

        g_object_class_install_property
                (object_class, PROP_ID,
                 g_param_spec_string ("id", "Image id",
                                 "The image id",
                                 NULL,
                                 G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
}

static void
btc_tmdb_image_init (BtcTmdbImage *self)
{
        self->priv = GET_PRIVATE (self);

        self->priv->type = NULL;
        self->priv->url = NULL;
        self->priv->width = 0;
        self->priv->height = 0;
        self->priv->size = NULL;
        self->priv->id = NULL;
}

BtcTmdbImage*
btc_tmdb_image_new ()
{
        return g_object_new (BTC_TYPE_TMDB_IMAGE, NULL);
}

gboolean
btc_tmdb_image_set_imagetype (BtcTmdbImage *self, const gchar *type)
{
        g_return_val_if_fail (BTC_TMDB_IMAGE (self), FALSE);

        g_free (self->priv->type);

        self->priv->type = g_strdup (type);

        return self->priv->type != NULL;
}

const gchar*
btc_tmdb_image_get_imagetype (BtcTmdbImage *self)
{
        return self->priv->type;
}

gboolean
btc_tmdb_image_set_url (BtcTmdbImage *self, const gchar *url)
{
        g_return_val_if_fail (BTC_TMDB_IMAGE (self), FALSE);

        g_free (self->priv->url);

        self->priv->url = g_strdup (url);

        return self->priv->url != NULL;
}

const gchar*
btc_tmdb_image_get_url (BtcTmdbImage *self)
{
        return self->priv->url;
}

gboolean
btc_tmdb_image_set_size (BtcTmdbImage *self, const gchar *size)
{
        g_return_val_if_fail (BTC_TMDB_IMAGE (self), FALSE);

        g_free (self->priv->size);

        self->priv->size = g_strdup (size);

        return self->priv->size != NULL;
}

const gchar*
btc_tmdb_image_get_size (BtcTmdbImage *self)
{
        return self->priv->size;
}

gboolean
btc_tmdb_image_set_id (BtcTmdbImage *self, const gchar *id)
{
        g_return_val_if_fail (BTC_TMDB_IMAGE (self), FALSE);

        g_free (self->priv->id);

        self->priv->id = g_strdup (id);

        return self->priv->id != NULL;
}

const gchar*
btc_tmdb_image_get_id (BtcTmdbImage *self)
{
        return self->priv->id;
}

gboolean
btc_tmdb_image_set_width (BtcTmdbImage *self, gint width)
{
        g_return_val_if_fail (BTC_TMDB_IMAGE (self), FALSE);

        self->priv->width = width;

        return TRUE;
}

gint
btc_tmdb_image_get_width (BtcTmdbImage *self)
{
        return self->priv->width;
}

gboolean
btc_tmdb_image_set_height (BtcTmdbImage *self, gint height)
{
        g_return_val_if_fail (BTC_TMDB_IMAGE (self), FALSE);

        self->priv->height = height;

        return TRUE;
}

gint
btc_tmdb_image_get_height (BtcTmdbImage *self)
{
        return self->priv->width;
}

void
btc_tmdb_image_print (BtcTmdbImage *self)
{
        int i;
        gchar *value = NULL;

        for (i = 0; i < LAST_FIELD; i ++) {
                g_object_get (self, image_fields[i], &value, NULL);
                g_print ("*[%s]: %s\n", image_fields[i], value);
                g_free (value);
        }
}

const gchar*
btc_tmdb_image_get_field (int field)
{
        g_return_val_if_fail (field >= 0 && field < LAST_FIELD, NULL);

        return image_fields[field];
}
