/*
 * btc-minfo-provider.c
 *
 * This file is part of butaca
 * Copyright (C) 2010 Simón Pena <spenap@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 */

#include "btc-minfo-provider.h"

#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <json-glib/json-glib.h>

#include "btc-tmdb-movie.h"
#include "btc-tmdb-image.h"
#include "btc-watc-movie.h"

#define TMDB_API_KEY "249e1a42df9bee09fac5e92d3a51396b"
#define TMDB_LANGUAGE "en"
#define TMDB_FORMAT "xml"
#define TMDB_METHOD "Movie.search"
#define TMDB_BASE_URL "http://api.themoviedb.org/2.1/%s/%s/%s/%s/%s"
#define TMDB_MOVIE_XPATH "/OpenSearchDescription/movies/movie"

#define WATC_BASE_URL "http://whatsafterthecredits.com/api.php?action=%s&format=%s&search=%s"
#define WATC_ACTION "opensearch"
#define WATC_FORMAT "json"

G_DEFINE_TYPE (BtcMInfoProvider, btc_minfo_provider, G_TYPE_OBJECT)

enum {
        PROP_0,
        PROP_FORMAT,
};

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), BTC_TYPE_MINFO_PROVIDER, BtcMInfoProviderPrivate))

struct _BtcMInfoProviderPrivate {
        gchar *format;
        BtcService service;
};

enum {
        RESPONSE_RECEIVED,
        LAST_SIGNAL
};

static guint
signals[LAST_SIGNAL] = { 0 };


static void
btc_minfo_provider_get_property (GObject *object, guint property_id,
                         GValue *value, GParamSpec *pspec)
{
        BtcMInfoProvider *self = BTC_MINFO_PROVIDER (object);

        switch (property_id) {
        case PROP_FORMAT:
                g_value_set_string (value, self->priv->format);
                break;
        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        }
}

static void
btc_minfo_provider_set_property (GObject *object, guint property_id,
                         const GValue *value, GParamSpec *pspec)
{
        BtcMInfoProvider *self = BTC_MINFO_PROVIDER (object);

        switch (property_id) {
        case PROP_FORMAT:
                btc_minfo_provider_set_format (self,
                                g_value_get_string (value));
                break;
        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        }
}

static void
btc_minfo_provider_finalize (GObject *object)
{
        BtcMInfoProvider *self = BTC_MINFO_PROVIDER (object);

        g_free (self->priv->format);

        G_OBJECT_CLASS (btc_minfo_provider_parent_class)->finalize (object);
}

static void
btc_minfo_provider_class_init (BtcMInfoProviderClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (klass);

        g_type_class_add_private (klass, sizeof (BtcMInfoProviderPrivate));

        object_class->get_property = btc_minfo_provider_get_property;
        object_class->set_property = btc_minfo_provider_set_property;
        object_class->finalize = btc_minfo_provider_finalize;

        g_object_class_install_property
                (object_class, PROP_FORMAT,
                 g_param_spec_string ("format", "The format", "The format",
                                      TMDB_FORMAT,
                                      G_PARAM_READWRITE | G_PARAM_CONSTRUCT));

        signals[RESPONSE_RECEIVED] = g_signal_new ("response-received",
                        BTC_TYPE_MINFO_PROVIDER,
                        G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
                        0,
                        NULL,
                        NULL,
                        g_cclosure_marshal_VOID__UINT_POINTER,
                        G_TYPE_NONE,
                        2,
                        G_TYPE_UINT,
                        G_TYPE_POINTER,
                        NULL);
}

static void
btc_minfo_provider_init (BtcMInfoProvider *self)
{
        self->priv = GET_PRIVATE (self);
        self->priv->format = NULL;
        self->priv->service = BTC_SERVICE_TMDB;
}

BtcMInfoProvider*
btc_minfo_provider_new (void)
{
        return g_object_new (BTC_TYPE_MINFO_PROVIDER, NULL);
}

static BtcTmdbImage*
create_tmdb_image (xmlNodePtr node)
{
        BtcTmdbImage *image = btc_tmdb_image_new ();
        gchar *value = NULL;
        int i;

        /* <image type="poster"
         * url="http://....jpg"
         * size="original"
         * id="4bc91...e007304"/> */

        for (i = 0; i < LAST_FIELD; i ++) {

                const gchar *image_field = btc_tmdb_image_get_field (i);

                value = (gchar *) xmlGetProp (node,
                                              (const xmlChar *) image_field);
                g_object_set (image, image_field, value, NULL);

                g_free (value);
        }

        return image;
}

static BtcTmdbMovie*
create_tmdb_movie (xmlNodePtr node)
{
        xmlNodePtr cur_node = NULL;
        BtcTmdbMovie *movie_info = btc_tmdb_movie_new ();
        GList *image_list = NULL;

        /* We use the loop to append each property to the movie object */
        for (cur_node = node; cur_node; cur_node = cur_node->next) {
                if (cur_node->type == XML_ELEMENT_NODE) {
                        gchar *value = NULL;

                        if (xmlStrcmp (cur_node->name, (const xmlChar *) "images") == 0) {
                                xmlNodePtr cur_image = NULL;
                                for (cur_image = cur_node->children; cur_image;
                                                cur_image = cur_image->next) {

                                        BtcTmdbImage *tmdb_image = create_tmdb_image (cur_image);
                                        image_list = g_list_append (image_list, tmdb_image);
                                }
                        }
                        else {
                                value = (gchar *) xmlNodeGetContent (cur_node);
                                g_object_set (movie_info, (gchar *) cur_node->name, value, NULL);
                                g_free (value);
                        }
                }
        }

        btc_tmdb_movie_set_images (movie_info, image_list);

        return movie_info;
}

static GList*
generate_list (xmlNodeSetPtr node_set)
{
        int i = 0;
        GList *list = NULL;

        for (i = 0; i < node_set->nodeNr; i++) {
                xmlNodePtr node = node_set->nodeTab[i];
                if (node->type == XML_ELEMENT_NODE) {
                        BtcTmdbMovie *movie_info =
                                        create_tmdb_movie (node->children);
                        if (movie_info)
                                list = g_list_prepend (list, movie_info);
                }
        }

        if (list)
                list = g_list_reverse (list);

        return list;
}

static GList*
parse_xml (const char *xml_data, goffset length)
{
        GList *list = NULL;
        xmlDocPtr document = xmlReadMemory (xml_data, length,
                        NULL,
                        NULL,
                        XML_PARSE_NOBLANKS | XML_PARSE_RECOVER);
        g_return_val_if_fail (document, NULL);

        xmlXPathContextPtr context_ptr = xmlXPathNewContext (document);

        xmlXPathObjectPtr xpath_obj =
                        xmlXPathEvalExpression ((const xmlChar *) TMDB_MOVIE_XPATH,
                                                context_ptr);

        xmlNodeSetPtr nodeset = xpath_obj->nodesetval;

        if (nodeset->nodeNr > 0) {
                list = generate_list (nodeset);
        }

        xmlXPathFreeObject (xpath_obj);
        xmlXPathFreeContext (context_ptr);
        xmlFreeDoc (document);

        return list;
}

static GList *
parse_json (const char *json_data, goffset length)
{
        JsonParser *parser = NULL;
        JsonNode *root = NULL;
        GError *error = NULL;
        GList *list = NULL;

        parser = json_parser_new ();

        json_parser_load_from_data (parser, json_data, length, &error);
        if (error)
        {
                g_warning ("Unable to parse data '%s': %s\n",
                                json_data, error->message);
                g_error_free (error);
                g_object_unref (parser);
                return list;
        }

        /* Don't free */
        root = json_parser_get_root (parser);
        JsonArray *response = json_node_get_array (root);

        /* The response is expected with the following format:
         * [ SEARCH_TERM ,[ SEARCH_RESULT_1, SEARCH_RESULT_N]] */

        if (json_array_get_length (response) != 2) {

                g_warning ("Wrong response format: %s\n", json_data);

                g_object_unref (parser);
                return list;
        }

        const gchar *search_term = json_array_get_string_element (response, 0);
        g_message ("Searched for: %s\n", search_term);

        JsonArray *results = json_array_get_array_element (response, 1);
        int i;
        int array_length = json_array_get_length (results);

        for (i = 0; i < array_length; i++) {
                const gchar *result =
                                json_array_get_string_element (results, i);
                BtcWatcMovie *watc_movie = btc_watc_movie_new (result);
                list = g_list_prepend (list, watc_movie);
        }

        g_object_unref (parser);

        if (list)
                list = g_list_reverse (list);

        return list;
}

static void
process_response_cb (SoupSession *session, SoupMessage *message,
                    gpointer user_data)
{
        BtcMInfoProvider *self = BTC_MINFO_PROVIDER (user_data);
        const gchar *mime = NULL;
        GList *list = NULL;
        guint service = BTC_SERVICE_NONE;

        if (!SOUP_STATUS_IS_SUCCESSFUL (message->status_code) ||
                        message->response_body->length <= 0) {

                g_print ("%s\n", message->reason_phrase);
        }
        else {

                mime = soup_message_headers_get_content_type
                                (message->response_headers, NULL);
                g_message ("Mime type: %s\n", mime);

                if (g_strcmp0 (mime, "text/xml") == 0) {
                        list = parse_xml (message->response_body->data,
                                        message->response_body->length);
                        service = BTC_SERVICE_TMDB;
                }
                else if (g_strcmp0 (mime, "application/json") == 0) {
                        list = parse_json (message->response_body->data,
                                        message->response_body->length);
                        service = BTC_SERVICE_WATC;
                }
        }

        g_signal_emit (self, signals[RESPONSE_RECEIVED], 0, service, list);
}

static gchar *
get_query_uri (BtcMInfoProvider *self, const char *query)
{
        gchar *uri = NULL;

        if (self->priv->service == BTC_SERVICE_TMDB) {
                /* METHOD/LANGUAGE/FORMAT/APIKEY/MOVIENAME */
                uri = g_strdup_printf (TMDB_BASE_URL, TMDB_METHOD,
                                TMDB_LANGUAGE,
                                self->priv->format,
                                TMDB_API_KEY,
                                query);

        }
        else if (self->priv->service == BTC_SERVICE_WATC) {
                /* WATCBASE_URL/ACTION/FORMAT/QUERY */
                uri = g_strdup_printf (WATC_BASE_URL,
                                WATC_ACTION,
                                WATC_FORMAT,
                                query);
        }
        else {
                g_warning ("Service unsupported\n");
        }

        g_message ("%s", uri);
        return uri;
}

gboolean
btc_minfo_provider_query (BtcMInfoProvider *self, BtcService service,
                          const gchar *query)
{
        g_return_val_if_fail (BTC_IS_MINFO_PROVIDER (self), FALSE);

        self->priv->service = service;

        SoupSession *session = NULL;
        SoupMessage *message = NULL;
        gboolean message_queued = FALSE;

        gchar *uri = get_query_uri (self, query);

        g_return_val_if_fail (uri, FALSE);

        session = soup_session_async_new ();
        message = soup_message_new ("GET", uri);

        if (message) {
                soup_session_queue_message (session, message,
                                process_response_cb, self);
                message_queued = TRUE;
        }

        g_free (uri);

        return message_queued;
}

gboolean
btc_minfo_provider_set_format (BtcMInfoProvider *self,
                               const gchar *format)
{
        g_return_val_if_fail (BTC_IS_MINFO_PROVIDER (self), FALSE);

        g_free (self->priv->format);

        self->priv->format = g_strdup (format);

        return TRUE;
}
