/*
 * btc-minfo-provider-service.c
 *
 * This file is part of butaca
 * Copyright (C) 2010 Simón Pena <spenap@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 */

#include <dbus/dbus-glib-bindings.h>

#include "btc-minfo-provider-service.h"
#include "btc-minfo-provider.h"
#include "btc-tmdb-movie-service.h"
#include "btc-tmdb-movie.h"
#include "btc-watc-movie-service.h"
#include "btc-watc-movie.h"
#include "btc-marshal.h"

#define MINFO_PROVIDER_SERVICE_OBJECT_PATH "/MInfoProvider"
#define MINFO_PROVIDER_SERVICE_NAME "com.simonpena.butaca.minfoprovider"
#define TMDB_MOVIE_INTERFACE "com.simonpena.butaca.tmdbmovie"
#define WATC_MOVIE_INTERFACE "com.simonpena.butaca.watcmovie"

G_DEFINE_TYPE (BtcMInfoProviderService, btc_minfo_provider_service, G_TYPE_OBJECT)

enum {
        PROP_0,
        PROP_DBUSGCONN,
};

enum {
        RESPONSE_RECEIVED,
        LAST_SIGNAL
};

static guint
btc_minfo_provider_service_signals[LAST_SIGNAL] = { 0 };

#define GET_PRIVATE(o) \
        (G_TYPE_INSTANCE_GET_PRIVATE ((o), BTC_TYPE_MINFO_PROVIDER_SERVICE, BtcMInfoProviderServicePrivate))

struct _BtcMInfoProviderServicePrivate {
        BtcMInfoProvider *minfo_provider;
        DBusGConnection *connection;
        GMainLoop *main_loop;
        GList *previous_results;
        guint search_id;
};

gboolean
btc_minfo_provider_service_query (BtcMInfoProviderService *self,
                                  BtcService service,
                                  const gchar *query,
                                  GError **error)
{
        return btc_minfo_provider_query (self->priv->minfo_provider,
                        service, query);
}

gboolean
btc_minfo_provider_service_exit (BtcMInfoProviderService *self, GError **error)
{
        g_main_loop_quit (self->priv->main_loop);
        return TRUE;
}

#include "btc-minfo-provider-service-glue.h"

static void
clear_previous_results (GList *previous_results)
{
        g_list_foreach (previous_results, (GFunc) g_object_unref, NULL);
        g_list_free (previous_results);
}

static void
response_received_cb (BtcMInfoProvider *provider, guint service, GList *list,
                      gpointer user_data)
{
        BtcMInfoProviderService *self = BTC_MINFO_PROVIDER_SERVICE (user_data);
        GList *iter = NULL;
        gchar **object_paths= g_new0 (gchar*, g_list_length (list) + 1);
        gchar *movie_interface = NULL;
        guint i = 0;

        movie_interface = service == BTC_SERVICE_TMDB ?
                        g_strdup (TMDB_MOVIE_INTERFACE):
                        g_strdup (WATC_MOVIE_INTERFACE);

        clear_previous_results (self->priv->previous_results);
        self->priv->previous_results = NULL;

        for (iter = list; iter; iter = iter->next) {
                if (BTC_IS_TMDB_MOVIE (iter->data)) {
                        BtcTmdbMovie *tmdb_movie = BTC_TMDB_MOVIE (iter->data);
                        gchar *uid_suffix = g_strdup_printf ("%d_%s",
                                        self->priv->search_id,
                                        btc_tmdb_movie_get_id (tmdb_movie));

                        BtcTmdbMovieService *movie = btc_tmdb_movie_service_new (
                                        self->priv->connection,
                                        tmdb_movie, uid_suffix);
                        object_paths[i] = g_strdup_printf ("/TMDBMovie/%s",
                                        uid_suffix);
                        g_free (uid_suffix);

                        self->priv->previous_results =
                                        g_list_prepend (self->priv->previous_results,
                                                        movie);
                }
                else if (BTC_IS_WATC_MOVIE (iter->data)) {
                        BtcWatcMovie *watc_movie = BTC_WATC_MOVIE (iter->data);
                        gchar *uid_suffix = g_strdup_printf ("%d_%d",
                                        self->priv->search_id,
                                        i);
                        BtcWatcMovieService *movie = btc_watc_movie_service_new (
                                        self->priv->connection,
                                        watc_movie, uid_suffix);
                        object_paths[i] = g_strdup_printf ("/WATCMovie/%s",
                                        uid_suffix);
                        g_free (uid_suffix);

                        self->priv->previous_results =
                                        g_list_prepend (self->priv->previous_results,
                                                        movie);
                }
                i++;
        }
        object_paths[i] = NULL;

        g_signal_emit (self, btc_minfo_provider_service_signals[RESPONSE_RECEIVED],
                       0, movie_interface, object_paths);
        g_strfreev (object_paths);
        g_free (movie_interface);
        self->priv->search_id++;
}

static void
setup_dbus (BtcMInfoProviderService *self)
{
        DBusGProxy *proxy;
        guint request_name_result;
        GError *error = NULL;

        proxy = dbus_g_proxy_new_for_name (self->priv->connection,
                                           DBUS_SERVICE_DBUS,
                                           DBUS_PATH_DBUS,
                                           DBUS_INTERFACE_DBUS);

        if (!org_freedesktop_DBus_request_name (proxy,
                                                MINFO_PROVIDER_SERVICE_NAME,
                                                0, &request_name_result,
                                                &error)) {
                g_warning ("Unable to register service: %s", error->message);
                g_error_free (error);
        }

        dbus_g_connection_register_g_object (self->priv->connection,
                                             MINFO_PROVIDER_SERVICE_OBJECT_PATH,
                                             G_OBJECT (self));

        g_object_unref (proxy);
}

static void
btc_minfo_provider_service_set_property (GObject *object, guint property_id,
                                  const GValue *value, GParamSpec *pspec)
{
        BtcMInfoProviderService *self = BTC_MINFO_PROVIDER_SERVICE (object);

        switch (property_id) {
        case PROP_DBUSGCONN:
                if (!self->priv->connection) {
                        DBusGConnection *tmp = g_value_get_pointer (value);
                        if (tmp) {
                                self->priv->connection =
                                        dbus_g_connection_ref (tmp);
                                setup_dbus (self);
                        }
                }
                g_assert (self->priv->connection);
                break;
        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        }
}

static void
btc_minfo_provider_service_finalize (GObject *object)
{
        BtcMInfoProviderService *self = BTC_MINFO_PROVIDER_SERVICE (object);

        if (self->priv->connection) {
                dbus_g_connection_unref (self->priv->connection);
        }
        g_object_unref (self->priv->minfo_provider);
        clear_previous_results (self->priv->previous_results);

        G_OBJECT_CLASS (btc_minfo_provider_service_parent_class)->finalize (object);
}

static void
btc_minfo_provider_service_class_init (BtcMInfoProviderServiceClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (klass);

        g_type_class_add_private (klass, sizeof (BtcMInfoProviderServicePrivate));

        object_class->set_property = btc_minfo_provider_service_set_property;
        object_class->finalize = btc_minfo_provider_service_finalize;

        g_object_class_install_property
                (object_class, PROP_DBUSGCONN,
                 g_param_spec_pointer ("connection", "DBusGConnection",
                                       "DBus GConnection",
                                       G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));

         dbus_g_object_type_install_info (BTC_TYPE_MINFO_PROVIDER_SERVICE,
                                          &dbus_glib_btc_minfo_provider_service_object_info);

         btc_minfo_provider_service_signals[RESPONSE_RECEIVED] =
                         g_signal_new ("response-received",
                         G_TYPE_FROM_CLASS (klass),
                         G_SIGNAL_RUN_LAST,
                         0,
                         NULL,
                         NULL,
                         btc_marshal_VOID__STRING_POINTER,
                         G_TYPE_NONE,
                         2,
                         G_TYPE_STRING,
                         G_TYPE_STRV,
                         NULL);
}

static void
btc_minfo_provider_service_init (BtcMInfoProviderService *self)
{
        self->priv = GET_PRIVATE (self);
        self->priv->minfo_provider = btc_minfo_provider_new ();
        self->priv->connection = NULL;
        self->priv->search_id = 0;
        self->priv->previous_results = NULL;

        g_signal_connect (self->priv->minfo_provider, "response-received",
                          G_CALLBACK (response_received_cb), self);
}

BtcMInfoProviderService*
btc_minfo_provider_service_new (DBusGConnection *connection, GMainLoop *loop)
{
        BtcMInfoProviderService *self = g_object_new (BTC_TYPE_MINFO_PROVIDER_SERVICE,
                                                      "connection", connection, NULL);
        self->priv->main_loop = loop;

        return self;
}
