/*
 * btc-watc-movie-service.c
 *
 * This file is part of butaca
 * Copyright (C) 2010 Simón Pena <spenap@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 */

#include <dbus/dbus-glib-bindings.h>

#include "btc-watc-movie-service.h"

#define WATC_MOVIE_SERVICE_OBJECT_PATH "/WATCMovie"
#define WATC_MOVIE_SERVICE_NAME "com.simonpena.butaca.watcmovie"

G_DEFINE_TYPE (BtcWatcMovieService, btc_watc_movie_service, G_TYPE_OBJECT)

enum {
        PROP_0,
        PROP_ID,
        PROP_DBUSGCONN,
};

#define GET_PRIVATE(o) \
        (G_TYPE_INSTANCE_GET_PRIVATE ((o), BTC_TYPE_WATC_MOVIE_SERVICE, BtcWatcMovieServicePrivate))

struct _BtcWatcMovieServicePrivate {
        BtcWatcMovie *movie;
        DBusGConnection *connection;
        gchar *suffix;
};

gboolean
btc_watc_movie_service_get_name (BtcWatcMovieService *self, gchar **title,
                                  GError **error)
{
        *title = g_strdup (btc_watc_movie_get_name (self->priv->movie));
        return *title != NULL;
}

gboolean
btc_watc_movie_service_get_year (BtcWatcMovieService *self, gchar **year,
                                 GError **error)
{
        *year = g_strdup (btc_watc_movie_get_year(self->priv->movie));
        return *year != NULL;
}

gboolean
btc_watc_movie_service_get_stingers (BtcWatcMovieService *self, int *stingers,
                                     GError **error)
{
        *stingers = btc_watc_movie_get_stingers (self->priv->movie);
        return TRUE;
}

#include "btc-watc-movie-service-glue.h"

static void
setup_dbus (BtcWatcMovieService *self)
{
        DBusGProxy *proxy;
        guint request_name_result;
        GError *error = NULL;
        gchar *object_path = NULL;

        proxy = dbus_g_proxy_new_for_name (self->priv->connection,
                                           DBUS_SERVICE_DBUS,
                                           DBUS_PATH_DBUS,
                                           DBUS_INTERFACE_DBUS);

        if (!org_freedesktop_DBus_request_name (proxy,
                                                WATC_MOVIE_SERVICE_NAME,
                                                0, &request_name_result,
                                                &error)) {
                g_warning ("Unable to register service: %s", error->message);
                g_error_free (error);
        }

        object_path = g_strdup_printf (WATC_MOVIE_SERVICE_OBJECT_PATH "/%s",
                        self->priv->suffix);

        dbus_g_connection_register_g_object (self->priv->connection,
                                             object_path,
                                             G_OBJECT (self));

        g_free (object_path);
        g_object_unref (proxy);
}

static void
btc_watc_movie_service_set_property (GObject *object, guint property_id,
                                  const GValue *value, GParamSpec *pspec)
{
        BtcWatcMovieService *self = BTC_WATC_MOVIE_SERVICE (object);

        switch (property_id) {
        case PROP_DBUSGCONN:
                if (!self->priv->connection) {
                        DBusGConnection *tmp = g_value_get_pointer (value);
                        if (tmp) {
                                self->priv->connection =
                                        dbus_g_connection_ref (tmp);
                                setup_dbus (self);
                        }
                }
                g_assert (self->priv->connection);
                break;
        default:
                G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
        }
}

static void
btc_watc_movie_service_finalize (GObject *object)
{
        BtcWatcMovieService *self = BTC_WATC_MOVIE_SERVICE (object);

        if (self->priv->connection) {
                dbus_g_connection_unref (self->priv->connection);
        }
        g_free (self->priv->suffix);
        g_object_unref (self->priv->movie);
        G_OBJECT_CLASS (btc_watc_movie_service_parent_class)->finalize (object);
}

static void
btc_watc_movie_service_class_init (BtcWatcMovieServiceClass *klass)
{
        GObjectClass *object_class = G_OBJECT_CLASS (klass);

        g_type_class_add_private (klass, sizeof (BtcWatcMovieServicePrivate));

        object_class->set_property = btc_watc_movie_service_set_property;
        object_class->finalize = btc_watc_movie_service_finalize;

        g_object_class_install_property
                (object_class, PROP_DBUSGCONN,
                 g_param_spec_pointer ("connection", "DBusGConnection",
                                       "DBus GConnection",
                                       G_PARAM_WRITABLE));

         dbus_g_object_type_install_info (BTC_TYPE_WATC_MOVIE_SERVICE,
                                          &dbus_glib_btc_watc_movie_service_object_info);
}

static void
btc_watc_movie_service_init (BtcWatcMovieService *self)
{
        self->priv = GET_PRIVATE (self);
        self->priv->movie = NULL;
        self->priv->connection = NULL;
        self->priv->suffix = NULL;
}

BtcWatcMovieService*
btc_watc_movie_service_new (DBusGConnection *connection,
                BtcWatcMovie *movie, const gchar *suffix)
{
        BtcWatcMovieService *instance = g_object_new (BTC_TYPE_WATC_MOVIE_SERVICE, NULL);
        instance->priv->suffix = g_strdup(suffix);
        g_object_set (instance, "connection", connection, NULL);
        instance->priv->movie = movie;
        return instance;
}
