#include <QDebug>
#include <QBrush>
#include <QColor>
#include <QFontMetrics>
#include <QFont>
#include <QIcon>
#include <QSettings>
#include <QApplication>

#include "matchdaymodel.h"
#include "match.h"

MatchDayModel::MatchDayModel(QObject *parent, MatchDayBackend *backend) :
    QAbstractTableModel(parent),
    m_lastRowCount(0),
    m_settings(qApp->organizationName(), qApp->applicationName())
{
    m_backend = backend;

    connect(m_backend, SIGNAL(updateFinished(int)),
            this, SLOT(onUpdateFinished(int)));
}

int MatchDayModel::rowCount(const QModelIndex&) const
{
    int count = m_backend->matchList().count();

    return count;
}

int MatchDayModel::columnCount(const QModelIndex&) const
{
    return 11;
}

QVariant MatchDayModel::data(const QModelIndex& index, int role) const
{
    Match*       match;
    QFont        f;
    QSize        s;
    QIcon        i;

    if ((match = m_backend->matchList().at(index.row())) == NULL) {
        return QVariant(QVariant::Invalid);
    }

    // DisplayRole
    switch (role) {
    case Qt::BackgroundRole:
        return QBrush(QColor(20, 20, 20, 100));
        break;

    case Qt::DecorationRole:
        switch (index.column()) {
        case AwayIcon:
            i = match->awayEmblem().pixmap(25,25);
            break;
        case HomeIcon:
            i = match->homeEmblem().pixmap(25,25);
            break;
        case MatchState:
            switch(match->state()) {
            case Match::NotStarted:
                return QIcon(":/bullet-grey").pixmap(15,15);
                break;
            case Match::FirstHalf:
            case Match::SecondHalf:
                return QIcon(":/bullet-green").pixmap(15,15);
                break;
            case Match::HalfTime:
                return QIcon(":/bullet-yellow").pixmap(15,15);
                break;
            case Match::Finished:
                return QIcon(":/bullet-red").pixmap(15,15);
                break;
            default:
                return QVariant(QVariant::Invalid);
            }

            break;
        }
        return i;
        break;

    case Qt::DisplayRole:
        switch (index.column()) {
        case AwayTeam:
            return match->awayTeam();
            break;
        case AwayScore:
            if (match->state() == Match::NotStarted) {
                return "-";
            } else {
                return match->awayScore();
            }
            break;
        case HomeTeam:
            return match->homeTeam();
            break;
        case HomeScore:
            if (match->state() == Match::NotStarted) {
                return "-";
            } else {
                return match->homeScore();
            }
            break;
        case Seperator:
            return ":";
            break;
        case Date:
            return match->date().toString("ddd hh:mm");
            break;

        default:
            return QVariant(QVariant::Invalid);
            break;
        }
        break;

    case Qt::SizeHintRole:
        s.setHeight(25);
        switch (index.column()) {
        case Spacer:
        case Spacer2:
            s.setWidth(2);
            break;            
        case MatchState:
            s.setWidth(15);
            break;
        case AwayIcon:
            s.setWidth(29);
            break;
        case AwayTeam:
            s.setWidth(120);
            break;
        case AwayScore:
            s.setWidth(4);
            break;
        case HomeIcon:
            s.setWidth(29);
            break;
        case HomeTeam:
            s.setWidth(120);
            break;
        case HomeScore:
            s.setWidth(4);
            break;
        case Seperator:
            s.setWidth(5);
            break;
        case Date:
            s.setWidth(75);
            break;
        default:
            return QVariant(QVariant::Invalid);
            break;
        }
        return s;
        break;

    case Qt::TextAlignmentRole:
        if (index.column() < Seperator) {
            return 0x0002 | 0x0080;
        } else if (index.column() > Seperator) {
            return 0x0001 | 0x0080;
        } else {
            return Qt::AlignCenter;
        }
        break;

    case Qt::FontRole:
        if ((index.column() == HomeScore ||
            index.column() == AwayScore) &&
            (match->lastEvent().secsTo((QDateTime::currentDateTime())) < 300)) {
            f.setBold(true);
            f.setPixelSize(16);
        } else {
            f.setBold(false);
            f.setPixelSize(14);
        }

        return f;

    default:
        return QVariant(QVariant::Invalid);
    }

    return QVariant(QVariant::Invalid);
}


void MatchDayModel::onUpdateFinished(int)
{
    //remove all rows
    beginRemoveRows(QModelIndex(),
                    0,
                    m_lastRowCount);
    endRemoveRows();

    //add rows
    beginInsertRows(QModelIndex(),
                    0,
                    m_backend->matchList().count() - 1);
    endInsertRows();

    m_lastRowCount = m_backend->matchList().count() - 1;

    // invalidate complete data
    qDebug() << "MatchDayModel::emit dataChanged: " << rowCount(QModelIndex());
    emit dataChanged(index(0, 0),
                     index(rowCount(QModelIndex()) - 1, columnCount(QModelIndex()) - 1));

}
