#include <QtNetwork/QNetworkAccessManager>
#include <QtNetwork/QNetworkRequest>
#include <QUrl>
#include <QRegExp>
#include <QDebug>
#include <QStringList>
#include <QDateTime>
#include <QSettings>
#include <QApplication>

#include "backendkicker.h"

const int BackendKicker::INTERVAL_FAST = 180;

BackendKicker::BackendKicker(QObject *parent) :
    MatchDayBackend(parent),
    m_updateTimer(this),
    m_netaccmgr(new QNetworkAccessManager(this)),
    m_initialUpdateDone(false)
{
    QSettings settings(qApp->organizationName(), qApp->applicationName());

    connect(m_netaccmgr, SIGNAL(finished(QNetworkReply*)),
            this, SLOT(dlndFinished(QNetworkReply*)));

    connect(&m_updateTimer, SIGNAL(timeout()),
            this, SLOT(update()));

    m_updateTimer.setSingleShot(true);

    this->selectLeague(settings.value("League", "1. Bundesliga").toString());
}

Match* BackendKicker::getMatch(QString hometeam, QString awayteam, QDateTime date)
{
    QListIterator<Match*> iter(m_matchlist);
    Match*        match;

    while (iter.hasNext()) {
        match = iter.next();
        if (match->awayTeam() == awayteam &&
            match->homeTeam() == hometeam) {
            return match;
        }
    }

    match = new Match(hometeam, awayteam, date, this);
    m_matchlist.append(match);
    emit matchAdded(match);

    return match;
}

QList<Match*> BackendKicker::matchList()
{
    return m_matchlist;
}

static QDateTime parseDate(QString datehtml)
{
    static QDateTime    lastParsedDate;
    QStringList         tokens;
    QDate               date;

    int month, day, hour, minute;

    //qDebug() << "parseDate in: " << datehtml;

    tokens = datehtml.split(QRegExp("[>.&;:<\"]"), QString::SkipEmptyParts);
    date = QDate::currentDate();

    //qDebug() << tokens;
    if (tokens.count() < 6) {
        return lastParsedDate;
    }

    month  = (tokens.at(2)).toInt();
    day    = (tokens.at(1)).toInt();
    hour   = (tokens.at(4)).toInt();
    minute = (tokens.at(5)).toInt();

    lastParsedDate =  QDateTime(QDate(date.year(), month, day),
                                QTime(hour, minute));

    return lastParsedDate;
}

static QString parseTeam(QString teamhtml)
{
    QString team;

    //qDebug() << "parseTeam in: " << teamhtml;

    teamhtml.truncate(teamhtml.indexOf("</a>"));
    team = teamhtml.mid(teamhtml.lastIndexOf(">") + 1);

    //qDebug() << "parseTeam out: " << team;
    return team;
}

void BackendKicker::parseScore(Match* match, QString scorehtml)
{
    QStringList tokens;

    //qDebug() << "parseScore in: " << scorehtml;
    tokens = scorehtml.split(QRegExp("[>&();:<]"), QString::SkipEmptyParts);
    //qDebug() << tokens;

    if (tokens.count() == 7) {
        // no extra color tag -> either not started, halftime or finished
        if (tokens.at(4) == "-") {
            // no first half results -> match not started yet
            match->setState(Match::NotStarted, m_initialUpdateDone);
        } else if (tokens.at(1) == "-") {
            // second half has not been started but there are first
            // half results -> currently half time
            match->setScore(tokens.at(4).toInt(), tokens.at(5).toInt(),
                            m_initialUpdateDone);
            match->setState(Match::HalfTime, m_initialUpdateDone);
        } else {
            // no color tag and no "-" -> game is finished
            match->setScore(tokens.at(1).toInt(), tokens.at(2).toInt(),
                            m_initialUpdateDone);
            match->setState(Match::Finished, m_initialUpdateDone);
        }
    } else {
        // there is a color tag which means that either first
        // half or second half are currently running
        if (tokens.at(4).contains("color")) {
            // first half score marked red -> first half running
            match->setScore(tokens.at(5).toInt(), tokens.at(6).toInt(),
                            m_initialUpdateDone);
            match->setState(Match::FirstHalf, m_initialUpdateDone);
        } else if (tokens.at(1).contains("color")) {
            // second half score marked res -> second half running
            match->setState(Match::SecondHalf, m_initialUpdateDone);
            match->setScore(tokens.at(2).toInt(), tokens.at(3).toInt(),
                            m_initialUpdateDone);
        }

    }
    qDebug() << "match state: " << match->state();
    qDebug() << "match home: " << match->homeScore();
    qDebug() << "match away: " << match->awayScore();
    qDebug() << "notify: " << m_initialUpdateDone;
}

void BackendKicker::parsePage (QString htmlstr)
{
    QStringList     rawmatches;
    QString         hometeam, awayteam, tmp;
    QRegExp         rx;
    QDateTime       date;
    Match*          match;

    int             pos     = 0;
    int             count   = 0;

    //qDebug() << "parsePage in: " << htmlstr;

    rx.setPattern("<td class=\"first\">(.*)<td class=\"aligncenter last\">");
    rx.setMinimal(true);
    while ((pos = rx.indexIn(htmlstr, pos)) != -1) {
         ++count;
         rawmatches.append(htmlstr.mid(pos, rx.matchedLength()));
         //qDebug() << "MATCH " << count << ":" << htmlstr.mid(pos, rx.matchedLength()) << "\n\n";
         pos += rx.matchedLength();
     }

    rx.setPattern("<td.*>(.*)</td>");

    QStringList::iterator i;
    for (i = rawmatches.begin(); i != rawmatches.end(); ++i) {
        pos = 0;
        count = 0;
        while ((pos = rx.indexIn(*i, pos)) != -1) {
             ++count;
             tmp = (*i).mid(pos, rx.matchedLength());
             pos += rx.matchedLength();
             switch (count) {
             case 2: // date
                 date = parseDate(tmp);
                 break;
             case 3: // hometeam
                 hometeam = parseTeam(tmp);
                 break;
             case 5: // awayteam
                 awayteam = parseTeam(tmp);
                 match = getMatch(hometeam, awayteam, date);
                 break;
             case 6: // scores
                 parseScore(match, tmp);
                 break;
             default:
                ;;
             }
        }
    }
}

bool BackendKicker::selectLeague(QString league)
{
    bool leagueIsSupported = true;

    if (league == "1. Bundesliga") {
        m_URL = "http://www.kicker.de/news/fussball/bundesliga/spieltag/1-bundesliga/2010-11/spieltag.html";
    } else if (league == "2. Bundesliga") {
        m_URL = "http://www.kicker.de/news/fussball/bundesliga/spieltag/2-bundesliga/2010-11/spieltag.html";
    } else if (league == "tipp3 Bundesliga") {
        m_URL = "http://www.kicker.de/news/fussball/intligen/oesterreich/tipp3-bundesliga/2010-11/spieltag.html";
    } else {
        leagueIsSupported = false;
    }

    // delete last data
    m_initialUpdateDone = false;
    this->m_matchlist.clear();
    this->update();

    return leagueIsSupported;
}

void BackendKicker::update()
{
    emit updateStarted();

    qDebug() << "Start Update with URL: " << m_URL;
    m_netaccmgr->get(QNetworkRequest(QUrl(m_URL)));
}

void BackendKicker::dlndFinished(QNetworkReply *reply)
{
    QString         rawdata;
    int             secstonextupdate;

    if (reply->error() != QNetworkReply::NoError) {
        qWarning() << "Download failed with error: " << reply->error();
    } else {
        rawdata = reply->readAll();
        parsePage(rawdata);
    }

    secstonextupdate = secsToNextGame();
    if ((secstonextupdate == -1) ||
        (secstonextupdate > 6 * 3600)) {
        // all games finished for this matchday
        secstonextupdate = 6 * 3600;
    } else if (secstonextupdate < INTERVAL_FAST) {
        secstonextupdate = INTERVAL_FAST;
    }
    m_updateTimer.start(secstonextupdate * 1000);

    emit updateFinished(reply->error());
    m_initialUpdateDone = true;
    qDebug() << "Update finished, next update in: " << secstonextupdate << "seconds.";
}

int BackendKicker::secsToNextGame()
{
    QListIterator<Match*> iter(m_matchlist);
    Match*        match;
    int           secstogame;
    int           secstonextgame = -1;

    while (iter.hasNext()) {
        match = iter.next();
        if (match->state() == Match::FirstHalf ||
            match->state() == Match::SecondHalf ||
            match->state() == Match::HalfTime) {

            secstonextgame = 0;
            return secstonextgame;
            break;
        } else if (match->state() == Match::NotStarted) {
            secstogame = QDateTime::currentDateTime().secsTo(match->date());
            if (secstonextgame == -1) {
                secstonextgame = secstogame;
            } else if (secstogame < secstonextgame) {
                secstonextgame = secstogame;
            }
        }
    }

    return secstonextgame;
}
