/* Box2D Demo in SDL (couldn't figure out the current
 * OpenGLES situation
 * Copyright (c) 2009 Wilson Tang <Wil.7ang@gmail.com>
 * GPLv3 http://www.gnu.org/copyleft/gpl.html
 */

#include "SDL/SDL.h"
#include "SDL/SDL_gfxPrimitives.h"
#include "box2d/Box2D.h" //This is in the package libbox2d-dev
#include <cmath>
#include <stdio.h>
#include <vector>

//If on maemo use accelerometers
#ifdef MAEMO
#include <fstream>
#endif

#define MAX_BOXES 100
#define MAX_FPS 60
#define PI 3.14159265

using namespace std;

//If on maemo use accelerometers, else, return constant gravity downwards
b2Vec2 ReturnGravity()
{
#ifdef MAEMO
	ifstream data;
	float x, y, z;
	data.open("/sys/class/i2c-adapter/i2c-3/3-001d/coord");
	if(!data)
		return b2Vec2(0.0f, 10.0f);

	data >> x >> y >> z;
	data.close();
	printf("%f, %f, %f\n", x, y, z);
	return b2Vec2( -x/100.0f, -y/100.0f);
#else
	return b2Vec2(0.0f, 10.0f);
#endif
}


class Box
{
	public:
		Box();
		~Box();
		void Draw(SDL_Surface *screen);
		void SetPosition(float xPos, float yPos);
		void MoveToPos(float xPos, float yPos, float xr, float yr);
		bool ClickDelete();

	//private:
		float xPosition;
		float yPosition;

		b2BodyDef bodyDef;
		b2Body* body;
		b2PolygonDef shapeDef;
		b2CircleDef circleDef;

		bool deleteTimer;
		float clicksT;

		int type;

};

Box::Box()
{
	body = NULL;
	bodyDef.position.Set(0.0f, 0.0f);
	shapeDef.SetAsBox(2.0f, 2.0f);
	shapeDef.density = 1.0f;
	shapeDef.friction = 0.3f;

	circleDef.radius = 2.0f;
	circleDef.density = 1.0f;
	circleDef.friction = 0.3f;


	deleteTimer = false;
	clicksT = 0.0f;
	type = 0;
}

Box::~Box()
{
	body = NULL;
}

bool Box::ClickDelete()
{
	if( !deleteTimer )
	{
		deleteTimer = true;
		clicksT = SDL_GetTicks();
		return false;
	}
	else if(deleteTimer)
	{
		if(SDL_GetTicks() - clicksT <= 500.0f)
		{
			clicksT = 0.0f;
			deleteTimer = false;
			return true;
		}
	}
	return false;
}

void Box::MoveToPos( float xPos, float yPos, float xr, float yr )
{
	if(body != NULL)
	{
		body->SetXForm(b2Vec2(xPos, yPos), body->GetAngle());
		body->SetLinearVelocity(b2Vec2(xr, yr));
	}
}

void Box::SetPosition( float xPos, float yPos )
{
	bodyDef.position.Set(xPos, yPos);
	xPosition = xPos;
	yPosition = yPos;
}

void Box::Draw(SDL_Surface *screen)
{
	if(body != NULL)
	{
		if(type == 0)
		{
			float points[4][2];
			float centerX = body->GetPosition().x*10.0f;
			float centerY = body->GetPosition().y*10.0f;
			float angle = body->GetAngle();


			//Calculate the 4 points of the box
			points[0][0] = (sqrt(abs(pow(-20.0f, 2)) + abs(pow(20.0f, 2))) * sin( (-PI/4) + -angle )) + centerX; //x
			points[0][1] = (sqrt(abs(pow(-20.0f, 2)) + abs(pow(20.0f, 2))) * cos( (-PI/4) + -angle )) + centerY;//y

			points[1][0] = (sqrt(abs(pow(20.0f, 2)) + abs(pow(20.0f, 2))) * sin( (PI/4) + -angle )) + centerX; //x
			points[1][1] = (sqrt(abs(pow(20.0f, 2)) + abs(pow(20.0f, 2))) * cos( (PI/4) + -angle )) + centerY;//y

			points[2][0] = (sqrt(abs(pow(20.0f, 2)) + abs(pow(-20.0f, 2))) * sin( (3*PI/4) + -angle )) + centerX; //x
			points[2][1] = (sqrt(abs(pow(20.0f, 2)) + abs(pow(-20.0f, 2))) * cos( (3*PI/4) + -angle )) + centerY;//y

			points[3][0] = (sqrt(abs(pow(-20.0f, 2)) + abs(pow(-20.0f, 2))) * sin( (5*PI/4) + -angle )) + centerX; //x
			points[3][1] = (sqrt(abs(pow(-20.0f, 2)) + abs(pow(-20.0f, 2))) * cos( (5*PI/4) + -angle )) + centerY;//y

			//Draws the box
			aalineRGBA(screen, points[0][0], points[0][1], points[1][0], points[1][1], 255, 255, 255, 255);
			aalineRGBA(screen, points[1][0], points[1][1], points[2][0], points[2][1], 255, 255, 255, 255);
			aalineRGBA(screen, points[2][0], points[2][1], points[3][0], points[3][1], 255, 255, 255, 255);
			aalineRGBA(screen, points[3][0], points[3][1], points[0][0], points[0][1], 255, 255, 255, 255);

			pixelRGBA(screen, int(centerX), int(centerY), 255, 255, 0, 255);
			//float hypo = sqrt(abs(pow(20.0f, 2)) + abs(pow(20.0f, 2)));

			if(deleteTimer)
				circleRGBA(screen, centerX, centerY, 20, 255, 255, 255, 255);
		}
		else if(type == 1)
		{
			float centerX = body->GetPosition().x*10.0f;
			float centerY = body->GetPosition().y*10.0f;
			circleRGBA(screen, centerX, centerY, 20, 255, 255, 0, 255);
		}

		if(SDL_GetTicks() - clicksT > 500.0f)
		{
			clicksT = 0.0f;
			deleteTimer = false;
		}


	}
}

class Physics
{
	public:
		Physics();
		~Physics();
		void StepTime();
		void SetGravity(float x, float y);
		void SetGravity(b2Vec2 grav);
		void AddBox(int object, float x, float y);

		int GetBoxAt(float x, float y);
		void DragBox(int ID);
		bool DelBox( int ID );
		void DelInBox(int x1, int y1, int x2, int y2);

		void UpdateMousePos(float x, float y, float xr, float yr);
		void Draw(SDL_Surface *screen);
	private:
		b2AABB worldAABB;
		b2World *world;
		vector<Box> boxList;

		float scaleFac;

		b2BodyDef wallBodyDef[4];
		b2Body* wallBody[4];
		b2PolygonDef wallShapeDef[4];

		float mX;
		float mY;
		float mXr;
		float mYr;

};

Physics::Physics()
{
	mX = 0.0f;
	mY = 0.0f;

	scaleFac = 10.0f;
	b2AABB worldAABB;
	worldAABB.lowerBound.Set(-100.0f, -100.0f);
	worldAABB.upperBound.Set(100.0f, 100.0f);
	b2Vec2 gravity(0.0f, 10.0f);
	bool doSleep = true;
	world = new b2World(worldAABB, gravity, doSleep);

	//Create the four walls
	wallBodyDef[0].position.Set(40.0f, -0.5f);
	wallBodyDef[1].position.Set(40.0f, 48.5f);
	wallBodyDef[2].position.Set(-0.5f, 24.0f);
	wallBodyDef[3].position.Set(80.5f, 24.0f);

	wallBody[0] = world->CreateBody(&wallBodyDef[0]);
	wallBody[1] = world->CreateBody(&wallBodyDef[1]);
	wallBody[2] = world->CreateBody(&wallBodyDef[2]);
	wallBody[3] = world->CreateBody(&wallBodyDef[3]);

	wallShapeDef[0].SetAsBox(82.0f, 0.6f);
	wallShapeDef[1].SetAsBox(82.0f, 0.6f);
	wallShapeDef[2].SetAsBox(0.6f, 48.0f);
	wallShapeDef[3].SetAsBox(0.6f, 48.0f);

	wallBody[0]->CreateShape(&wallShapeDef[0]);
	wallBody[1]->CreateShape(&wallShapeDef[1]);
	wallBody[2]->CreateShape(&wallShapeDef[2]);
	wallBody[3]->CreateShape(&wallShapeDef[3]);

}

Physics::~Physics()
{
	delete world;
	world = NULL;
}

void Physics::UpdateMousePos(float x, float y, float xr, float yr)
{
	mX = x;
	mY = y;
	mXr = xr;
	mYr = yr;
}

int Physics::GetBoxAt(float x, float y)
{
	for(unsigned int i = 0; i < boxList.size(); i++)
	{
		if(   ( x > boxList[i].body->GetPosition().x-2.0f ) &&
			( x < boxList[i].body->GetPosition().x+2.0f ) &&
			( y > boxList[i].body->GetPosition().y-2.0f ) &&
			( y < boxList[i].body->GetPosition().y+2.0f ) )
		{
			return i;
		}
	}
	return -1;
}

void Physics::DelInBox(int x1, int y1, int x2, int y2)
{
	for(unsigned int i = 0; i < boxList.size(); i++)
	{
		if(   ( boxList[i].body->GetPosition().x > x1 ) &&
			( boxList[i].body->GetPosition().x < x2 ) &&
			( boxList[i].body->GetPosition().y > y1 ) &&
			( boxList[i].body->GetPosition().y < y2 ) )
		{
			world->DestroyBody( boxList[i].body );
			boxList.erase(boxList.begin()+i);
			i = -1;
		}
	}
}

bool Physics::DelBox(int ID)
{
	if(boxList[ID].ClickDelete())
	{
		world->DestroyBody( boxList[ID].body );
		boxList.erase(boxList.begin()+ID);
		return true;
	}
	return false;
}

void Physics::AddBox(int object, float x, float y)
{
	//if(boxList.size() < MAX_BOXES) //Maximum disabled
	{
		Box newBox;
		newBox.SetPosition(x, y);
		boxList.push_back(newBox);
		boxList[boxList.size()-1].body = world->CreateBody(&boxList[boxList.size()-1].bodyDef);

		if(object == 0)
			boxList[boxList.size()-1].body->CreateShape(&boxList[boxList.size()-1].shapeDef);
		else
			boxList[boxList.size()-1].body->CreateShape(&boxList[boxList.size()-1].circleDef);
		boxList[boxList.size()-1].body->SetMassFromShapes();

		boxList[boxList.size()-1].type = object;
	}
}

void Physics::Draw(SDL_Surface *screen)
{
	for(unsigned int i = 0; i < boxList.size(); i++)
	{
		boxList[i].Draw(screen);
	}
	SDL_Flip(screen);
	SDL_FillRect(screen, &screen->clip_rect, SDL_MapRGB(screen->format, 0, 0, 0));
}

void Physics::StepTime()
{
	float32 timeStep = 1.0f / 16.0f;
	int32 iterations = 10;
	world->Step(timeStep, iterations);
}

void Physics::SetGravity(float x, float y)
{
	world->SetGravity(b2Vec2(x, y));
	for( unsigned int i = 0; i < boxList.size(); i++ )
	{
		if(boxList[i].body->IsSleeping())
			boxList[i].body->WakeUp();
	}
}

void Physics::SetGravity(b2Vec2 grav)
{
	world->SetGravity(grav);
	for( unsigned int i = 0; i < boxList.size(); i++ )
	{
		if(boxList[i].body->IsSleeping())
			boxList[i].body->WakeUp();
	}
}

void Physics::DragBox(int ID)
{
	boxList[ID].MoveToPos(mX, mY, mXr, mYr);
}

int main()
{
	SDL_Init(SDL_INIT_EVERYTHING);
#ifdef MAEMO
	SDL_Surface *screen = SDL_SetVideoMode( 800, 480, 16, SDL_FULLSCREEN | SDL_HWSURFACE |SDL_DOUBLEBUF);
#else
	SDL_Surface *screen = SDL_SetVideoMode( 800, 480, 16, SDL_HWSURFACE |SDL_DOUBLEBUF );
#endif
	Physics myPhys;
	myPhys.AddBox(0, 40.0f, 10.0f);
	SDL_Event event;
	bool quit = false;
	bool clicked = false;

	float startT = 0.0f;
	float endT = 0.0f;

	float xLoc = 0.0f;
	float yLoc = 0.0f;
	float xVel = 0.0f;
	float yVel = 0.0f;

	int cX = 0;
	int cY = 0;

	int boxID = -1;

	int mode = 0; //Modes, 0 = add boxes, 1 = drag/delete boxes
	int object = 0; //Objects: 0 = boxes, 1 = circles


	int frame = 0;
	float fpsSta;
	float fpsEnd;


	fpsSta = SDL_GetTicks();
	while(!quit)
	{
		startT = SDL_GetTicks();

		xVel = 0.0f;
		yVel = 0.0f;
		while( SDL_PollEvent( &event ) )
		{
			if( event.type == SDL_KEYDOWN )
			{
				if( event.key.keysym.sym == SDLK_ESCAPE)
				{
					quit = true;
				}
				if( event.key.keysym.sym == SDLK_m)
				{
					if(mode == 0)
						mode = 1;
					else
						mode = 0;
				}
				if( event.key.keysym.sym == SDLK_1)
				{
					object = 0;
				}
				if( event.key.keysym.sym == SDLK_2)
				{
					object = 1;
				}
			}
			//Add another box when the mouse is clicked
			if( event.type == SDL_MOUSEBUTTONDOWN )
			{
				clicked = true;
				if(mode == 0)
				{
					xLoc = event.button.x;
					yLoc = event.button.y;

					myPhys.AddBox(object, xLoc/10.0f, yLoc/10.0f);
					boxID = myPhys.GetBoxAt(xLoc/10.0f, yLoc/10.0f);
				}
				else if(mode == 1)
				{
					cX = xLoc = event.button.x;
					cY = yLoc = event.button.y;
					boxID = myPhys.GetBoxAt(xLoc/10.0f, yLoc/10.0f);

					if(boxID >= 0)
					{
						if(myPhys.DelBox(boxID))
							clicked = false;
					}
				}



			}
			if( event.type == SDL_MOUSEBUTTONUP )
			{
				clicked = false;
				if(boxID == -1)
				{
					myPhys.DelInBox( cX/10.0f, cY/10.0f, xLoc/10.0f, yLoc/10.0f );
				}
			}
			if( event.type == SDL_MOUSEMOTION )
			{
				xLoc = event.motion.x;
				yLoc = event.motion.y;

				xVel = event.motion.xrel;
				yVel = event.motion.yrel;
			}
		}
		myPhys.UpdateMousePos(xLoc/10.0f, yLoc/10.0f, xVel*1.0f, yVel*1.0f);

		if(clicked)
		{
			if(boxID >= 0)
			{
				myPhys.DragBox(boxID);
			}
			else if(boxID == -1)
			{
				aalineRGBA(screen, cX, cY, xLoc, cY, 255, 0, 0, 255);
				aalineRGBA(screen, xLoc, cY, xLoc, yLoc, 255, 0, 0, 255);
				aalineRGBA(screen, cX, cY, cX, yLoc, 255, 0, 0, 255);
				aalineRGBA(screen, cX, yLoc, xLoc, yLoc, 255, 0, 0, 255);
			}
		}

		myPhys.SetGravity(ReturnGravity());
		myPhys.StepTime();
		myPhys.Draw(screen);

		//Frame rate limited to 60fps
		endT = SDL_GetTicks();
		if(endT - startT < (1000.0f/MAX_FPS) )
		{
			SDL_Delay( (1000.0f/MAX_FPS) - (endT - startT));
		}

		fpsEnd = SDL_GetTicks();
		if(fpsEnd - fpsSta>= 1000.0f)
		{
			printf("%i Frames per second.\n", frame);
			frame = 0;
			fpsSta = SDL_GetTicks();
		}
		else
			frame++;

	}

	return 0;
}
