#include "bootscreen.h"

#include <QtMaemo5>

BootScreen::BootScreen(QWidget *parent)
    : QMainWindow(parent)
{
    hildonWelcome = "/etc/hildon-welcome.d/";
    videosDir = "/opt/bootscreen/media/";
    settingsDir = "/opt/bootscreen/data/";
    checkSettings();

    filterGroup = new QActionGroup(this);
    filterGroup->setExclusive(true);
    mRandom = new QAction(tr("Random"), filterGroup);
    mRandom->setCheckable(true);
    mMultiple = new QAction(tr("Multiple"), filterGroup);
    mMultiple->setCheckable(true);

    //laod settings
    if(random)
        mRandom->setChecked(true);
    else
        mMultiple->setChecked(true);


    connect(mRandom, SIGNAL(toggled(bool)), this, SLOT(disableSelection(bool)));
    connect(mMultiple, SIGNAL(toggled(bool)), this, SLOT(selectMultiple(bool)));

    menuBar()->addActions(filterGroup->actions());

    menuBar()->addAction(tr("Add"), this, SLOT(addVideo()));
    menuBar()->addAction(tr("Remove"), this, SLOT(removeVideos()));
    menuBar()->addAction(tr("About"), this, SLOT(showAbout()));

    setWindowTitle("BootScreen Manager");

    createList();
}

void BootScreen::checkSettings()
{
    QDir dir(settingsDir);
    settingsFile.setFileName(dir.filePath("bootscreen.conf"));

    if (!settingsFile.exists())
        restoreSettings();
    else{
         QSettings tempSettings(settingsFile.fileName(),QSettings::IniFormat);
         random = tempSettings.value("random").toBool();
    }
}

void BootScreen::createList()
{
    QWidget *central = new QWidget();

    QVBoxLayout *lay = new QVBoxLayout(central);
    lay->setSpacing(8);
    lay->setContentsMargins(0, 0, 0, 15);

    list = new QListWidget();


    list->setSelectionMode(QAbstractItemView::MultiSelection);

    QDir dirHildon(hildonWelcome);
    dirHildon.setFilter(QDir::Files | QDir::NoSymLinks);
    dirHildon.setSorting(QDir::Name);

    QFileInfoList ls = dirHildon.entryInfoList();
    for (int i = 0; i < ls.size(); ++i) {
        QFileInfo fileInfo = ls.at(i);
        QSettings tempSettings(fileInfo.absoluteFilePath(),QSettings::IniFormat);
        tempSettings.beginGroup("hildon-welcome");

        QFileInfo prettyName(tempSettings.value("filename").toString());
        list->addItem(QString("%1").arg(prettyName.fileName()));
        mediaFiles[prettyName.fileName()] = tempSettings.value("filename").toString();


        qDebug() << fileInfo.absoluteFilePath() << endl;
        qDebug() << tempSettings.value("filename") << endl;
    }


    QListWidgetItem *temp;
    int listSize = list->count();
    //activate items
    for(int i = 0; i < listSize; i++){
        temp = list->item(i);
        if(temp)
            temp->setSelected(true);
    }

    if(random)
        list->setSelectionMode(QAbstractItemView::NoSelection);

    //Add hands video

    if(!mediaFiles.contains("Hands-v32-h264.avi")){
        QFileInfo handsTmp("/usr/share/hildon-welcome/media/Hands-v32-h264.avi");
        if(handsTmp.exists()){
            mediaFiles[handsTmp.fileName()] = handsTmp.absoluteFilePath();
            list->addItem(QString("%1").arg(handsTmp.fileName()));
        }
    }

    QDir dir(videosDir);
    dir.setFilter(QDir::Files | QDir::NoSymLinks);
    dir.setSorting(QDir::Name);
    //if empty show warning to add a file via menu

    ls = dir.entryInfoList();
    for (int i = 0; i < ls.size(); ++i) {
        QFileInfo fileInfo = ls.at(i);
        if(!mediaFiles.contains(fileInfo.fileName())){
            mediaFiles[fileInfo.fileName()] = fileInfo.absoluteFilePath();
            list->addItem(QString("%1").arg(fileInfo.fileName()));
        }
        qDebug() << fileInfo.absoluteFilePath() << endl;

    }

   // for(int i =0; i < 8; i++){
     //  list->addItem("bbr");
    //}

    lay->addWidget(list);

    QHBoxLayout *layButtons = new QHBoxLayout();
    up = new QPushButton("Up");
    down = new QPushButton("Down");
    save = new QPushButton("Save");
    save->setEnabled(false);

    if(random){
        up->setEnabled(false);
        down->setEnabled(false);
    }

    layButtons->addWidget(up);
    layButtons->addWidget(down);
    layButtons->addSpacing(50);
    layButtons->addWidget(save);
    lay->addLayout(layButtons);

    connect(up, SIGNAL(clicked()), this, SLOT(moveUp()));
    connect(down, SIGNAL(clicked()), this, SLOT(moveDown()));
    connect(save, SIGNAL(clicked()), this, SLOT(saveConfs()));
    connect(list, SIGNAL(itemActivated(QListWidgetItem*)), this, SLOT(setPendingChanges(QListWidgetItem*)));

    setCentralWidget(central);
}

void BootScreen::addVideo()
{
    QString filePath = QFileDialog::getOpenFileName(this,
        tr("Open file"), "/home/user/MyDocs", tr("Video Files (*.avi)"));
    qDebug() << filePath << endl;

    if(filePath != ""){
        QFile fileCopy(filePath);
        QFileInfo fileInfo(filePath);
        QString fileName = fileInfo.fileName();
        //boot player can't read files with spaces in the name
        fileName.replace(QString(" "), QString("_"));
        QString newFilePath = videosDir + fileName;
        qDebug() << newFilePath << endl;
        qDebug() << fileInfo.size() << endl;
        if(fileInfo.size() > 11100000)
            QMaemo5InformationBox::information(this, "<font color=black><b>Error:</b> The selected file is too big</font>",
                                                        QMaemo5InformationBox::DefaultTimeout);

        else{
            QFile tempTest(newFilePath);
            if(tempTest.exists())
                QMaemo5InformationBox::information(this, "<font color=black><b>Error:</b> The file already exists</font>",
                                                            QMaemo5InformationBox::DefaultTimeout);
            else{
                bool result = fileCopy.copy(newFilePath);
                if(result){
                    list->addItem(QString("%1").arg(fileName));
                    save->setEnabled(true);
                    //autoselect item, if not in random mode
                    if(!random){
                        QListWidgetItem *currentItem = list->item(list->count()-1);
                        currentItem->setSelected(true);
                    }
                }
                else
                    QMaemo5InformationBox::information(this, "<font color=black>Error copying file</font>",
                                                                QMaemo5InformationBox::DefaultTimeout);
            }
        }
    }

}

void BootScreen::removeVideos()
{
    QListWidgetItem *temp;
    QList <QListWidgetItem *> selectedItems = list->selectedItems();
    if(selectedItems.size()>0){
        QDir vidDir(videosDir);
        int ret = QMessageBox::information(this, tr("Remove files"), tr("Remove selected file(s) ?"), QMessageBox::Yes | QMessageBox::No);

        if(ret == QMessageBox::Yes){
            for(int i=0; i< selectedItems.size();i++){
                temp = selectedItems.at(i);
                bool result = vidDir.remove(temp->text());
                qDebug() << result << " :" << temp->text() << endl;

                if(result){
                    int rw = list->row(temp);
                    temp = list->takeItem(rw);
                    delete temp;
                }
            }
            save->setEnabled(true);
        }
    }

}

void BootScreen::showAbout()
{
    QString aboutText = "Written by Valerio Valerio <vdv100@gmail.com\n";
    aboutText += "Icon by Claes Norin <claes.norin@gmail.com>\n\n";
    QMessageBox::information(this, "BootScreen Manager", aboutText);
}

void BootScreen::moveUp()
{
    int currentRow = list->currentRow();
    if (currentRow == 0) return;
    QListWidgetItem * currentItem = list->takeItem(currentRow);
    list->insertItem(currentRow - 1, currentItem);
    list->setCurrentRow(currentRow - 1);
    save->setEnabled(true);
}

void BootScreen::moveDown()
{
    int currentRow = list->currentRow();
    if (currentRow >= list->count()-1) return;
    QListWidgetItem * currentItem = list->takeItem(currentRow);
    list->insertItem(currentRow + 1, currentItem);
    list->setCurrentRow(currentRow + 1);
    save->setEnabled(true);
}

void BootScreen::disableSelection(bool state)
{
    if(state){
        up->setEnabled(false);
        down->setEnabled(false);
        QListWidgetItem *temp;
        int listSize = list->count();
        //disable items
        QTime fixTime(0, 0, 0);
        qsrand(fixTime.secsTo(QTime::currentTime()));
        int randomNumb = qrand() % listSize;
        qDebug() << randomNumb << endl;
        for(int i = 0; i < listSize; i++){
            temp = list->item(i);
            if(temp){
                if(i == randomNumb)
                    temp->setSelected(true);
                else
                    temp->setSelected(false);
            }
        }
        list->setSelectionMode(QAbstractItemView::NoSelection);
        QMaemo5InformationBox::information(this, "<font color=black><b>Random mode:</b> Selection disabled</font>",
                                                    QMaemo5InformationBox::DefaultTimeout);
    }

    random = true;
    save->setEnabled(true);
}

void BootScreen::selectMultiple(bool state)
{
    if(state){
        up->setEnabled(true);
        down->setEnabled(true);
        list->setSelectionMode(QAbstractItemView::MultiSelection);
        random = false;
    }
}

void BootScreen::writeSettings()
{
    QSettings settings(settingsFile.fileName(),QSettings::IniFormat);
    settings.setValue("random", random);
}

void BootScreen::restoreSettings()
{
     qDebug() << settingsFile.fileName() << endl;

     if (!settingsFile.open(QIODevice::ReadWrite | QIODevice::Text)){
        qWarning("Cannot create the settings file"); //abord
        QMaemo5InformationBox::information(this, "<font color=black><b>Error:</b> Cannot create the settings file</font>",
                                                    QMaemo5InformationBox::DefaultTimeout);
     }

     QTextStream out(&settingsFile);
     out << "random=false" << endl;

     random = false;
}

void BootScreen::saveConfs(){

    //remove old files
    QDir dir(hildonWelcome);
    dir.setFilter(QDir::Files);

    QFileInfoList ls = dir.entryInfoList();
    for (int i = 0; i < ls.size(); ++i) {
        QFileInfo fileInfo = ls.at(i);
        bool result = dir.remove(fileInfo.fileName());
        qDebug() << result << " :" << fileInfo.fileName() << endl;
    }

    QListWidgetItem *temp;
    for(int i=0; i< list->count();i++){
        temp = list->item(i);
        if(temp->isSelected())
            if(random)
                 createFile(temp->text(), 0);
            else
                createFile(temp->text(), i);
    }
    save->setEnabled(false);
}

void BootScreen::createFile(QString filename, int index)
{
    QString ind = QString::number(index);
    if(mediaFiles.contains(filename)){
        QFile confFile(hildonWelcome + ind + ".conf");
        qDebug() << confFile.fileName() << endl;
        if (!confFile.open(QIODevice::ReadWrite | QIODevice::Text)){
           qWarning("Cannot create the settings file"); //abord
           QMaemo5InformationBox::information(this, "<font color=black><b>Error:</b> Cannot create the configurations file</font>",
                                                       QMaemo5InformationBox::DefaultTimeout);
       }
        else{
            QString filePath = mediaFiles.value(filename);
            QTextStream out(&confFile);
            out << "[hildon-welcome]" << endl;
            out << "filename=" << filePath << endl;
        }
    }
    else{
        qDebug() << filename << endl;
        qDebug() << "File already exists" << endl;
    }
}

void BootScreen::setPendingChanges(QListWidgetItem *item)
{
    if(!random)
        save->setEnabled(true);
}

void BootScreen::closeEvent(QCloseEvent *event)
{
    if(save->isEnabled()){
        int ret = QMessageBox::information(this, tr("There are pending changes"), tr("Do you want to save your changes?"),
                                           QMessageBox::Save | QMessageBox::Discard | QMessageBox::Cancel);

        if(ret == QMessageBox::Save){
            saveConfs();
            writeSettings();
            event->accept();

        }
        else if(ret == QMessageBox::Discard){
            writeSettings();
            event->accept();
        }
        else{
            event->ignore();
        }
    }
    else{
        writeSettings();
        event->accept();
    }
}

BootScreen::~BootScreen()
{
}


