/*
 *  Birthday application for Maemo.
 *  Copyright (C) 2010 Roman Moravcik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>

#include <libosso.h>

#include <glib.h>
#include <glib/gi18n.h>

#include <gtk/gtk.h>
#include <hildon/hildon.h>

#include <libebook/e-book.h>
#include <libosso-abook/osso-abook.h>

enum
{
	COLUMN_AVATAR = 0,
	COLUMN_DISPLAY,
	COLUMN_FULLNAME,
	COLUMN_NEXT_BIRTHDAY,
	NUM_COLS
};

/* Application UI data struct */
typedef struct _BirthdayData BirthdayData;
struct _BirthdayData {
	GtkWidget *window;
	GtkWidget *label;
	GtkWidget *view;
	GtkWidget *search;

	GtkTreeModel *sorted;
	GtkTreeModel *filter;

	gchar *searched_name;
	gboolean found;
};

static gboolean
birthday_filered_view_visible_func (GtkTreeModel *model,
				    GtkTreeIter *iter,
				    gpointer data)
{
	BirthdayData *priv;
	gchar *fullname = NULL, *ascii_searched_name = NULL, *ascii_fullname = NULL;
	gboolean found = FALSE;

	g_return_val_if_fail (data, FALSE);
	priv = (BirthdayData *) data;

	if (priv->searched_name == NULL) {
		priv->found = TRUE;
		return TRUE;
	}

	ascii_searched_name = g_ascii_strdown (priv->searched_name, strlen (priv->searched_name));

	gtk_tree_model_get (model, iter, COLUMN_FULLNAME, &fullname, -1);
	if (fullname) {
		ascii_fullname = g_ascii_strdown (fullname,  strlen (fullname));
		g_free (fullname);
	}

	if (g_strstr_len (ascii_fullname, strlen (ascii_fullname), ascii_searched_name) != NULL)
		found = TRUE;

	if (ascii_searched_name)
		g_free (ascii_searched_name);

	if (ascii_fullname)
		g_free (ascii_fullname);

	if (found)
		priv->found = TRUE;

	return found;
}

static void
sort_by_name_clicked (GtkButton *button,
                      gpointer data)
{
	BirthdayData *priv;

	g_return_if_fail (data);
	priv = (BirthdayData *) data;

	if (priv->sorted) {
		gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (priv->sorted),
						      COLUMN_FULLNAME, GTK_SORT_ASCENDING);
	}
}

static void
sort_by_date_clicked (GtkButton *button,
                      gpointer data)
{
	BirthdayData *priv;

	g_return_if_fail (data);
	priv = (BirthdayData *) data;

	if (priv->sorted) {
		gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (priv->sorted),
						      COLUMN_NEXT_BIRTHDAY, GTK_SORT_ASCENDING);
	}
}

static void
on_search_entry_changed (GtkEditable *editable,
			 gpointer data)
{
	BirthdayData *priv;
	GtkWidget *pannable;

	g_return_if_fail (data);
	priv = (BirthdayData *) data;

	priv->found = FALSE;

	if (priv->searched_name)
		g_free (priv->searched_name);
	priv->searched_name = g_strdup (gtk_entry_get_text (GTK_ENTRY (editable)));

	/* refilter tree view */
	gtk_tree_model_filter_refilter (GTK_TREE_MODEL_FILTER (priv->filter));

	if (priv->found) {
		/* hide label */
		gtk_widget_hide (priv->label);

		/* show tree view */
		gtk_widget_show (priv->view);
	} else {
		/* hide label */
		gtk_widget_show (priv->label);
		gtk_label_set_text (GTK_LABEL (priv->label), _("No search results"));

		/* show tree view */
		gtk_widget_hide (priv->view);
	}

	pannable = g_object_get_data (G_OBJECT (priv->view), "pannable");
	hildon_pannable_area_jump_to (HILDON_PANNABLE_AREA (pannable), 0, 0);
}

static void
on_search_close_clicked (GtkButton *button,
			 gpointer data)
{
	BirthdayData *priv;
	GtkWidget *pannable;

	g_return_if_fail (data);
	priv = (BirthdayData *) data;

	/* hide search bar */
	gtk_widget_hide (priv->search);

	/* hide label */
	gtk_widget_hide (priv->label);

	/* show tree view */
	gtk_widget_show (priv->view);

	/* clear searched name */
	if (priv->searched_name)
		g_free (priv->searched_name);
	priv->searched_name = NULL;

	/* refilter tree view */
	gtk_tree_model_filter_refilter (GTK_TREE_MODEL_FILTER (priv->filter));

	pannable = g_object_get_data (G_OBJECT (priv->view), "pannable");
	hildon_pannable_area_jump_to (HILDON_PANNABLE_AREA (pannable), 0, 0);
}


static gboolean
on_key_press_event (GtkWidget *widget,
		    GdkEventKey *event,
		    gpointer data)
{
	BirthdayData *priv;

	g_return_val_if_fail (data, TRUE);
	priv = (BirthdayData *) data;

	if ((event->keyval > GDK_space) && (event->keyval <= GDK_stricteq) && !GTK_WIDGET_VISIBLE (priv->search)) {
		GtkWidget *entry;

		/* show search bar */
		gtk_widget_show (priv->search);

		/* focus on search entry */
		entry  = g_object_get_data (G_OBJECT (priv->search), "entry");
		gtk_entry_set_text (GTK_ENTRY (entry), "");
		gtk_widget_grab_focus (GTK_WIDGET (entry));

		/* refilter tree view */
		gtk_tree_model_filter_refilter (GTK_TREE_MODEL_FILTER (priv->filter));
	}

	return FALSE;
}

static unsigned int
calc_age (EContactDate *bdate)
{
	struct tm tm_bday;
	struct tm *tm_age;
	time_t t_age;
	int age = 0;

	tm_bday.tm_sec = 0;
	tm_bday.tm_min = 0;
	tm_bday.tm_hour = 0;
	tm_bday.tm_mday = bdate->day;
	tm_bday.tm_mon = bdate->month - 1;
	tm_bday.tm_year = bdate->year - 1900;

	t_age = time (NULL) - mktime (&tm_bday);
	tm_age = gmtime (&t_age);
	age = tm_age->tm_year - 70;

	if (age < 0)
		age = 0;

	return age;
}

static unsigned int
calc_next_bday (EContactDate *bdate)
{
	struct tm tm_current_bday, tm_next_bday;
	struct tm *tm_current_date;
	time_t t_current_date, t_current_bday, t_next_bday;

	t_current_date = time (NULL);
	tm_current_date = localtime (&t_current_date);

	tm_current_bday.tm_sec = 0;
	tm_current_bday.tm_min = 0;
	tm_current_bday.tm_hour = 0;
	tm_current_bday.tm_mday = bdate->day;
	tm_current_bday.tm_mon = bdate->month - 1;
	tm_current_bday.tm_year = tm_current_date->tm_year;
	t_current_bday = mktime (&tm_current_bday);

	if (t_current_date > t_current_bday) {
		tm_next_bday.tm_sec = 0;
		tm_next_bday.tm_min = 0;
		tm_next_bday.tm_hour = 0;
		tm_next_bday.tm_mday = bdate->day;
		tm_next_bday.tm_mon = bdate->month - 1;
		tm_next_bday.tm_year = tm_current_date->tm_year + 1;
		t_next_bday = mktime (&tm_next_bday);
	} else {
		t_next_bday = t_current_bday;
	}

	return (t_next_bday - t_current_date) / 86400;
}

static gchar *
get_text_font_by_name (const gchar *name)
{
	GtkSettings *settings;
	GtkStyle *style;

	settings = gtk_settings_get_default ();
	style = gtk_rc_get_style_by_paths (settings, name, NULL, G_TYPE_NONE);
	return pango_font_description_to_string (style->font_desc);
}

static gchar *
get_text_color_by_name (const gchar *name)
{
	GtkSettings *settings;
	GtkStyle *style;
	GdkColor color;

	settings = gtk_settings_get_default ();
	style = gtk_rc_get_style_by_paths (settings, "GtkButton", "osso-logical-colors", G_OBJECT_TYPE(gtk_button_new()));
	gtk_style_lookup_color (style, name, &color);
	return gdk_color_to_string (&color);
}

static GtkListStore *
create_bday_liststore (GList *contacts)
{
	GtkListStore *store;
	GtkTreeIter iter;
	GList *contact;
	gchar *text_font = NULL;
	gchar *text_color = NULL;

	text_font = get_text_font_by_name ("SmallSystemFont");
	text_color = get_text_color_by_name ("SecondaryTextColor");

	store = gtk_list_store_new(NUM_COLS,
				   GDK_TYPE_PIXBUF,	/* COLUMN_AVATAR */
				   G_TYPE_STRING,	/* COLUMN_DISPLAY */
				   G_TYPE_STRING,	/* COLUMN_FULLNAME */
				   G_TYPE_INT);		/* COLUMN_NEXT_BIRTHDAY */

	for (contact = contacts; contact != NULL; contact = contact->next) {
		EContactDate *bdate = NULL;

		bdate = e_contact_get (E_CONTACT (contact->data), E_CONTACT_BIRTH_DATE);
		if (bdate) {
			EContactPhoto *photo = NULL;
			GError *error = NULL;
			GdkPixbuf *avatar = NULL;
			gchar *fullname = NULL;
			guint age = 0, next_birthday = 0;
			gchar *display_column = NULL;
			gchar *next_birthday_text = NULL;

			photo = e_contact_get (E_CONTACT (contact->data), E_CONTACT_PHOTO);
			if (photo) {
				if (photo->type == E_CONTACT_PHOTO_TYPE_INLINED) {
					GdkPixbufLoader *loader;

					loader = gdk_pixbuf_loader_new ();
					if (gdk_pixbuf_loader_write (loader, (guchar *) photo->data.inlined.data, photo->data.inlined.length, NULL))
						avatar = gdk_pixbuf_loader_get_pixbuf (loader);

				} else {
					gchar *avatar_filename = NULL;

					avatar_filename = g_filename_from_uri (photo->data.uri, NULL, NULL);
					if (avatar_filename) {
						avatar = gdk_pixbuf_new_from_file (avatar_filename, &error);
						g_free (avatar_filename);
					}
				}

				if (avatar) {
					gint height = gdk_pixbuf_get_height (avatar);
					if (height != 48) {
						gint new_height = 48;
						gint new_width = new_height * gdk_pixbuf_get_width (avatar) / height;
						avatar = gdk_pixbuf_scale_simple (avatar, new_width, new_height, GDK_INTERP_BILINEAR);
					}
				}
				e_contact_photo_free (photo);
				photo = NULL;
			} else {
				avatar = gdk_pixbuf_new_from_file ("/usr/share/icons/hicolor/48x48/hildon/general_default_avatar.png", &error);
			}

			fullname = e_contact_get (E_CONTACT (contact->data), E_CONTACT_FULL_NAME);
			if (!fullname) {
				fullname = e_contact_get (E_CONTACT (contact->data), E_CONTACT_GIVEN_NAME);
				if (!fullname) {
					fullname = e_contact_get (E_CONTACT (contact->data), E_CONTACT_FAMILY_NAME);
					if (!fullname) {
						fullname = e_contact_get (E_CONTACT (contact->data), E_CONTACT_NICKNAME);
						if (!fullname) {
							fullname = e_contact_get (E_CONTACT (contact->data), E_CONTACT_ORG);
						}
					}
				}
			}

			age = calc_age(bdate);
			next_birthday = calc_next_bday(bdate);
			next_birthday_text = g_strdup_printf(ngettext ("next birthday in %d day", "next birthday in %d days", next_birthday), next_birthday);
			display_column = g_strdup_printf("%s <span font_desc=\"%s\" foreground=\"%s\"><sup>(%d)</sup>\n%02d.%02d.%04d, %s</span>",
							 fullname, text_font, text_color, age, bdate->day, bdate->month, bdate->year, next_birthday_text);

			gtk_list_store_append (store, &iter);
			gtk_list_store_set (store, &iter,
					    COLUMN_AVATAR, avatar,
					    COLUMN_DISPLAY, display_column,
					    COLUMN_FULLNAME, fullname,
					    COLUMN_NEXT_BIRTHDAY, next_birthday,
					    -1);

			if (display_column)
				g_free (display_column);
			display_column = NULL;

			if (fullname)
				g_free (fullname);
			fullname = NULL;

			if (next_birthday_text)
				g_free (next_birthday_text);
			next_birthday_text = NULL;

			e_contact_date_free (bdate);
		}
		bdate = NULL;
	}

	if (text_font)
		g_free (text_font);

	if (text_color)
		g_free (text_color);

	return store;
}

static void
create_search_bar (BirthdayData *priv)
{
	GtkWidget *entry, *button;
	GtkEntryCompletion *completion;

	g_return_if_fail (priv);

	/* search hbox */
	priv->search = gtk_hbox_new (FALSE, HILDON_MARGIN_DEFAULT);

	/* search entry */
	entry = hildon_entry_new (HILDON_SIZE_FINGER_HEIGHT);
	hildon_gtk_entry_set_input_mode (GTK_ENTRY (entry), HILDON_GTK_INPUT_MODE_FULL);
	gtk_box_pack_start (GTK_BOX (priv->search), entry, TRUE, TRUE, 0);

	completion = gtk_entry_completion_new ();
	gtk_entry_completion_set_inline_completion (completion, TRUE);
	gtk_entry_completion_set_popup_completion (completion, FALSE);
	gtk_entry_set_completion (GTK_ENTRY (entry), completion);

	/* clear button */
	button = GTK_WIDGET (gtk_tool_button_new (gtk_image_new_from_icon_name
			     ("general_close", (GtkIconSize) HILDON_ICON_PIXEL_SIZE_FINGER), "Clear"));
	gtk_box_pack_end (GTK_BOX (priv->search), button, FALSE, TRUE, 0);

	/* search signals */
	g_signal_connect (entry, "changed", G_CALLBACK (on_search_entry_changed), priv);
	g_signal_connect (button, "clicked", G_CALLBACK (on_search_close_clicked), priv);

	g_object_set_data (G_OBJECT (priv->search), "entry", entry);
}

static void
create_main_menu (BirthdayData *priv)
{
	HildonAppMenu *menu;
	GtkWidget *filter;

	g_return_if_fail (priv);

	menu = HILDON_APP_MENU (hildon_app_menu_new ());
	hildon_window_set_app_menu (HILDON_WINDOW (priv->window), menu);

	filter = hildon_gtk_radio_button_new (HILDON_SIZE_FINGER_HEIGHT , NULL);
	gtk_button_set_label (GTK_BUTTON (filter), _("Name"));
	gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (filter), FALSE);
	g_signal_connect_after (filter, "clicked", G_CALLBACK (sort_by_name_clicked), priv);
	hildon_app_menu_add_filter (menu, GTK_BUTTON (filter));

	filter = hildon_gtk_radio_button_new_from_widget (HILDON_SIZE_FINGER_HEIGHT , GTK_RADIO_BUTTON (filter));
	gtk_button_set_label (GTK_BUTTON (filter), _("Date"));
	gtk_toggle_button_set_mode (GTK_TOGGLE_BUTTON (filter), FALSE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (filter), TRUE);
	g_signal_connect_after (filter, "clicked", G_CALLBACK (sort_by_date_clicked), priv);
	hildon_app_menu_add_filter (menu, GTK_BUTTON (filter));

	gtk_widget_show_all (GTK_WIDGET (menu));
}

static void
create_main_window (BirthdayData *priv, GtkListStore *store)
{
	HildonProgram *program = NULL;
	GtkWidget *main_vbox, *alignment, *pannable, *tree_view;
	GtkTreeModel *filter;
	GtkTreeViewColumn *column;
	GtkCellRenderer *renderer;

	g_return_if_fail (priv);

	program = hildon_program_get_instance ();
	g_set_application_name (_("Birthday"));

	/* main window */
	priv->window = hildon_stackable_window_new ();
	hildon_program_add_window (program, HILDON_WINDOW (priv->window));

	/* create main menu */
	create_main_menu (priv);

	/* aligment */
	alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding (GTK_ALIGNMENT (alignment),
				   HILDON_MARGIN_HALF, 0, HILDON_MARGIN_DEFAULT, HILDON_MARGIN_DEFAULT);
	gtk_container_add (GTK_CONTAINER (priv->window), alignment);

	/* main vbox */
	main_vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (alignment), main_vbox);

	/* no_search_result label */
	priv->label = gtk_label_new (_("No contacts with set birthdate"));
	hildon_helper_set_logical_color (priv->label, GTK_RC_FG,
					 GTK_STATE_NORMAL, "SecondaryTextColor");
	hildon_helper_set_logical_font (priv->label, "LargeSystemFont");
	gtk_box_pack_start (GTK_BOX (main_vbox), priv->label, TRUE, TRUE, 0);

	/* alignment for pannable area */
	priv->view = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
	gtk_alignment_set_padding (GTK_ALIGNMENT (priv->view),
				   0, 0, HILDON_MARGIN_DEFAULT, 0);
	gtk_box_pack_start (GTK_BOX (main_vbox), priv->view, TRUE, TRUE, 0);

	/* pannable for tree view */
	pannable = hildon_pannable_area_new ();
	g_object_set (G_OBJECT (pannable), "mov-mode", HILDON_MOVEMENT_MODE_VERT, NULL);
	g_object_set_data (G_OBJECT (priv->view), "pannable", pannable);
	gtk_container_add (GTK_CONTAINER (priv->view), pannable);

	/* sort list by next birthdays */
	gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (store),
					      COLUMN_NEXT_BIRTHDAY, GTK_SORT_ASCENDING);
	priv->sorted = GTK_TREE_MODEL (store);

	/* filtered view */
	filter = gtk_tree_model_filter_new (GTK_TREE_MODEL (store), NULL);
	gtk_tree_model_filter_set_visible_func (GTK_TREE_MODEL_FILTER (filter),
						birthday_filered_view_visible_func,
						priv,
						NULL);
	gtk_tree_model_filter_refilter (GTK_TREE_MODEL_FILTER (filter));
	priv->filter = GTK_TREE_MODEL (filter);

	/* tree view */
	tree_view = hildon_gtk_tree_view_new_with_model (HILDON_UI_MODE_NORMAL, filter);
	gtk_container_add (GTK_CONTAINER (pannable), tree_view);

	/* display column */
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_fixed_width (column, 709);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_attributes (column, renderer,
					     "markup", COLUMN_DISPLAY,
					     NULL);
	g_object_set (G_OBJECT (renderer), "xpad", 10, NULL);

	gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);

	/* avatar column */
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_fixed_width (column, 48);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_end (column, renderer, FALSE);
	gtk_tree_view_column_set_attributes (column, renderer,
					     "pixbuf", COLUMN_AVATAR,
					     NULL);
	gtk_tree_view_append_column (GTK_TREE_VIEW (tree_view), column);

	/* search bar */
	create_search_bar(priv);
	gtk_box_pack_end (GTK_BOX (main_vbox), priv->search, FALSE, FALSE, 0);

	/* window signals */
	g_signal_connect (G_OBJECT (priv->window), "destroy", G_CALLBACK (gtk_main_quit), NULL);
	g_signal_connect (G_OBJECT (priv->window), "key-press-event", G_CALLBACK (on_key_press_event), priv);

	gtk_widget_show_all (GTK_WIDGET (priv->window));
	gtk_widget_hide (GTK_WIDGET (priv->label));
	gtk_widget_hide (GTK_WIDGET (priv->search));
}

int main (int argc, char **argv)
{
	BirthdayData *data;
	osso_context_t *osso_context;
	EBook *ebook;
	EBookQuery *query;
	GError *error = NULL;
	GtkWidget *window;
	GtkListStore *store;
	GList *contacts;

	hildon_gtk_init (&argc, &argv);

	/* create application data */
	data = g_new0 (BirthdayData, 1);
	data->searched_name = NULL;
	data->found = TRUE;

	/* initialize localization */
	setlocale(LC_ALL, "");
	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);

	/* initialize osso */
	osso_context = osso_initialize ("birthday", "0.1", TRUE, NULL);
	if (osso_context == NULL) {
		g_critical ("Error initializing osso");
		return 1;
	}

	/* init abook */
	if (!osso_abook_init (&argc, &argv, osso_context)) {
		g_critical ("Error initializing libosso-abook");
		goto exit;
	}

	ebook = e_book_new_system_addressbook (&error);
	if (!ebook) {
		g_warning ("Error opening system address book: %s", error->message);
		g_error_free (error);
		goto exit;
	}

	if (!e_book_open (ebook, TRUE, &error)) {
		g_warning ("Error opening system address book: %s", error->message);
		g_error_free (error);
		goto exit;
	}

	query = e_book_query_any_field_contains ("");

	if (!e_book_get_contacts (ebook, query, &contacts, &error)) {
		g_warning ("Error getting contacts: %s", error->message);
		g_error_free (error);
		goto exit;
	}

	store = create_bday_liststore (contacts);
	create_main_window (data, store);

	gtk_main ();

exit:
	osso_deinitialize (osso_context);
	return 0;
}
