/*
 *  Copyright 2010 Ruediger Gad
 *
 *  This file is part of Binary Clock.
 *
 *  Binary Clock is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  Binary Clock is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Binary Clock.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <QCoreApplication>
#include <QIcon>
#include <QSettings>

#include "bcdclockwidget.h"
#include "binaryclockwidget.h"
#include "unixtimewidget.h"
#include "unixtimebchwidget.h"

#include "mainwidget.h"
#include "settingsdialog.h"

MainWidget::MainWidget(QWidget *parent) :
    QWidget(parent
#ifndef Q_WS_MAEMO_5
            , Qt::SplashScreen
            | Qt::FramelessWindowHint
#endif
            )
{
    clockWidget = NULL;

    QCoreApplication::setOrganizationName("Binary Clock");
    QCoreApplication::setOrganizationDomain("https://garage.maemo.org/projects/binaryclock");
    QCoreApplication::setApplicationName("Binary Clock");

    layout = new QVBoxLayout();
    layout->setSpacing(0);
    layout->setMargin(0);
    setLayout(layout);

#ifdef Q_WS_MAEMO_5
    setAttribute(Qt::WA_TranslucentBackground);
#else
    tray = NULL;

    QSettings settings;
    move(settings.value(SettingsDialog::POSITION,
                        SettingsDialog::POSITION_DEFAULT).toPoint());
    createMenu();

    if(QSystemTrayIcon::isSystemTrayAvailable()){
        createTray();
    }
#endif

    applySettings();

    layout->addWidget(clockWidget);
}

MainWidget::~MainWidget(){
    delete clockWidget;
    delete layout;

#ifndef Q_WS_MAEMO_5
    delete tray;

    delete closeAction;
    delete hideAction;
    delete settingsAction;

    delete menu;

    QSettings settings;
    settings.setValue(SettingsDialog::POSITION, pos());
    settings.sync();
#endif
}

void MainWidget::applySettings(){
    QSettings settings;
    setClock(settings.value(SettingsDialog::CLOCK_STYLE,
                            SettingsDialog::CLOCK_STYLE_DEFAULT).toInt());

#ifndef Q_WS_MAEMO_5
    transparency = settings.value(SettingsDialog::TRANSPARENCY,
                   SettingsDialog::TRANSPARENCY_DEFAULT).toBool();
    setAttribute(Qt::WA_TranslucentBackground, transparency);

    Qt::WindowFlags flags = windowFlags();

    switch(settings.value(SettingsDialog::WIDGET_PLACEMENT,
                          SettingsDialog::WIDGET_PLACEMENT_DEFAULT).toInt())
    {
    case 0:
        flags &= ~ Qt::WindowStaysOnBottomHint;
        flags |= Qt::WindowStaysOnTopHint;
        break;
    case 1:
        flags &= ~ Qt::WindowStaysOnBottomHint;
        flags &= ~ Qt::WindowStaysOnTopHint;
        break;
    case 2:
        flags |= Qt::WindowStaysOnBottomHint;
        flags &= ~ Qt::WindowStaysOnTopHint;
        break;
    default:
        qDebug("Unsupported setting for widget placement!");
        break;
    }

    setWindowFlags(flags);
    show();
#endif
}

#ifndef Q_WS_MAEMO_5
void MainWidget::close(){
    QWidget::close();
    QCoreApplication::exit();
}

void MainWidget::createMenu(){
    menu = new QMenu(this);

    settingsAction = new QAction(tr("Settings"), this);
    connect(settingsAction, SIGNAL(triggered()), this, SLOT(openSettingsDialog()));

    hideAction = new QAction(tr("Hide"), this);
    connect(hideAction, SIGNAL(triggered()), this, SLOT(triggerHide()));

    closeAction = new QAction(tr("Close"), this);
    connect(closeAction, SIGNAL(triggered()), this, SLOT(close()));

    menu->addAction(settingsAction);
    menu->addAction(hideAction);
    menu->addAction(closeAction);
}

void MainWidget::createTray(){
    tray = new QSystemTrayIcon(this);
    tray->setContextMenu(menu);
    tray->setIcon(QIcon(":/icon_transparency.png"));
    tray->setVisible(true);
    tray->setToolTip("Q Binary Clock");
}

void MainWidget::mouseMoveEvent(QMouseEvent *event){
    if (event->buttons() & Qt::LeftButton) {
        move(event->globalPos() - position);
        event->accept();
    }
}

void MainWidget::mousePressEvent(QMouseEvent *event){
    if (event->button() == Qt::LeftButton) {
        position = event->globalPos() - frameGeometry().topLeft();
    } else if (event->button() == Qt::RightButton){
        menu->exec(QCursor::pos());
    }
}

void MainWidget::triggerHide(){
    setHidden(! isHidden());
    hideAction->setText((isHidden()) ? "Show" : "Hide");
}

#endif

void MainWidget::openSettingsDialog(){
    SettingsDialog dialog;
    dialog.setAttribute(Qt::WA_QuitOnClose, false);

    if(dialog.exec()){
        applySettings();
    }
}

void MainWidget::paintEvent(QPaintEvent */*event*/){
    QPainter painter(this);
    painter.setRenderHint(QPainter::Antialiasing, true);
    painter.setPen(Qt::NoPen);

#ifdef Q_WS_MAEMO_5
    painter.setBrush(QColor(255, 255, 255, 128));
    painter.drawRoundedRect(rect(), 25, 25);
#else
    if(transparency){
        painter.setBrush(QColor(0, 0, 0, 128));
        painter.drawRoundedRect(rect(), 25, 25);
    }else{
        painter.setBrush(QColor(128, 128, 128));
        painter.drawRect(rect());
    }
#endif

    painter.end();
}

void MainWidget::setClock(int clockStyle){
    if(clockWidget != NULL){
        disconnect(this, SIGNAL(resumeClock(bool)), clockWidget, SLOT(resumeClock(bool)));
        delete clockWidget;
    }

    switch(clockStyle){
    case 0:
        clockWidget = new BinaryClockWidget();
        break;
    case 1:
        clockWidget = new BcdClockWidget();
        break;
    case 2:
        clockWidget = new UnixTimeWidget();
        break;
    case 3:
        clockWidget = new UnixTimeBchWidget();
        break;
    default:
        clockWidget = new BinaryClockWidget();
        break;
    }

    connect(this, SIGNAL(resumeClock(bool)), clockWidget, SLOT(resumeClock(bool)));

    layout->addWidget(clockWidget);
    setFixedSize(clockWidget->sizeHint());
}
