// @abstract(Systems management for Valkyrie)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(June 6, 2004)
// @lastmod(Jan 14, 2006)
//
//  @html <div class="license">
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  @html </div>
{$INCLUDE valkyrie.inc}
unit vini_broken;
interface
uses vdata, vutil, vstream;

// Callback procedure for mass parsing ini entries.
type TINIFileCallbackProcedure = procedure (nkey,nvalue : string);

// INI Entry Type type ;-).
type TINIEntryType = (typeString,typeNumber,typeBoolean,typeFlags);

// INI Entry type. EValue holds either a Integer, Boolean or Flags, or a AnsiString
type TINIEntry = record
                   // The type of the Entry.
                   EType  : TINIEntryType;
                   // The value. Typecast it to whatever EType holds.
                   EValue : Pointer;
                 end;
     // Pointer to @link(TINIEntry).
     PINIEntry = ^TINIEntry;
     
// TINIEntry AssocArray. Best used via the default property.
type TINIEntryAssocArray = class(TAbstractAssocArray)
       procedure addINIEntry(const Str : Ansistring; const Value : TINIEntry);
       function  getINIEntry(const Str : Ansistring) : TINIEntry;
       property INIEntry[index : Ansistring] : TINIEntry read getINIEntry write addINIEntry; default;
       protected
       procedure Rewrite(Entry : PAssocArrayEntry; Value : Pointer); override;
       procedure RemoveEntry(Entry : PAssocArrayEntry); override;
     end;
     
// The INI file class.
type TINI = class(TVObjectAssocArray)
       // Initializes a plain INI file data holder. Useful for constructing an
       // INI file in memory and then printing it to a file.
       constructor Init;
       // Loads a INI file.
       constructor Load(const FileName : string);
       // Creates a new section of the given name.
       procedure addSection(const Name : string); virtual;
       // Disposes properly of the allocated data.
       destructor Done;
       // Changes current section.
       procedure setSection(const Name : string);
       // Tries to change section. Returns wether change succeded (Section found).
       function trySetSection(const Name : string) : boolean;
       // Adds a string entry to given section.
       procedure add(const nKey : string; const nValue : Ansistring; Section : TINIEntryAssocArray = nil );virtual;
       // Adds a boolean entry to given section.
       procedure add(const nKey : string; const nValue : boolean; Section : TINIEntryAssocArray = nil );virtual;
       // Adds a integer entry to given section.
       procedure add(const nKey : string; const nValue : integer; Section : TINIEntryAssocArray = nil );virtual;
       // Adds a flags entry to given section.
       procedure add(const nKey : string; const nValue : TFlags; Section : TINIEntryAssocArray = nil );virtual;
       // Returns wether the key is defined in the current section.
       function Defined(const nkey : string) : boolean;
       // Returns a number from the current section.
       function getNumber(const nkey : string) : LongInt;
       // Returns a boolean from the current section.
       function getBoolean(const nkey : string) : Boolean;
       // Returns a String from the current section.
       function getString(const nkey : string) : AnsiString;
       // Returns Flags from the current section.
       function getFlags(const nkey : string) : TFlags;
       // Returns entry converted to string.
       function get(const nkey : string) : AnsiString;
       // Resolves a flagstring based on the [FLAGS] section.
       function ResolveFlags(const flagString : string) : TFlags;
       // Queries current section with given callback.
       procedure QuerySection(inicallback : TINIFileCallbackProcedure);
       // Converts entry of any type to string.
       function EntryToString(const Entry : TINIEntry) : Ansistring;
       property Num    [index : string] : LongInt read getNumber  write add;
       property Bool   [index : string] : Boolean read getBoolean write add;
       property Str    [index : string] : AnsiString  read getString  write add;
       property Flag   [index : string] : TFlags  read getFlags   write add;
       property General[index : string] : AnsiString  read get; default;
       protected
       Selected  : TINIEntryAssocArray;
       Flags     : TINIEntryAssocArray;
       Constants : TINIEntryAssocArray;
     end;

implementation

uses sysutils;

procedure TINIEntryAssocArray.addINIEntry(const Str : Ansistring; const Value : TINIEntry);
var p : PINIEntry;
begin
  New(p);
  p^ := Value;
  AddEntry(Str,p)
end;

function  TINIEntryAssocArray.getINIEntry(const Str : Ansistring) : TINIEntry;
var Entry : PINIEntry;
begin
  Entry := PINIEntry(GetEntry(Str));
  if Entry = nil then CritError('Entry "'+Str+'" not found in ini-file!');
  Exit(Entry^);
end;

procedure TINIEntryAssocArray.Rewrite(Entry : PAssocArrayEntry; Value : Pointer);
begin
  if PINIEntry(Entry^.Value)^.EType = typeFlags then
    Dispose(PFlags(PINIEntry(Entry^.Value)^.EValue));
  Dispose(PINIEntry(Entry^.Value));
  Entry^.Value := Value;
end;

procedure TINIEntryAssocArray.RemoveEntry(Entry : PAssocArrayEntry);
begin
  if PINIEntry(Entry^.Value)^.EType = typeFlags then
    Dispose(PFlags(PINIEntry(Entry^.Value)^.EValue));
  Dispose(PINIEntry(Entry^.Value));
  inherited RemoveEntry(Entry);
end;

constructor TINI.Init;
begin
  inherited Create;
  Selected := nil;
end;

constructor TINI.Load(const FileName : string);
var INI : Text;
    S,V : String;
    IP  : Byte;
    Entry : TINIEntry;
begin
  Create;
  if not FileExists(FileName) then CritError('File "'+Filename+'" not found!');
  Assign(INI,FileName);
  Reset(INI);
  while not EOF(INI) do
  begin
    Readln(INI,S);
    if S[1] = '#' then
    begin
      if Parameter(s,1,' ') = '#include' then Load(Parameter(s,1,'""'));
      Continue;
    end;
    IP := pos('//',S);
    if IP <> 0 then Delete(S,IP,Length(S)-IP+1);
    S := Trim(S);
    if S[1] = '#' then Continue;
    if length(s) < 3 then Continue;
    if (S[1] = '[') and (S[length(S)] = ']') then
    begin
      AddSection(Parameter(S,1,'[]'));
      Continue;
    end;
    if Selected = nil then CritError('Key without section!');
    S := Parameter(S,1,';');
    V := S;
    S := UpCase(Trim(Parameter(S,1,'=')));
    Delete(V,1,Pos('=',V));
    V := Trim(V);
    case V[1] of
      '#' : begin
              if length(V) < 2 then CritError('INI file error: '''+S+' => '+V+''' - const value error!');
              Delete(V,1,1);
              Entry := Constants[UpCase(V)];
              if Entry.EType <> typeNumber then CritError('Constant "'+V+'" not a number!');
              Add(S,Integer(Entry.EValue));
            end;
      't','T','f','F' : Add(S,StrToBool(V));
      '0'..'9' : Add(S,StrToInt(V));
      '"' : Add(S,Parameter(V,1,'""'));
      '[' : Add(S,ResolveFlags(Parameter(V,1,'[]')));
    end;
  end;
  Close(INI);
  Selected := nil;
end;

procedure TINI.addSection(const Name : string);
var SectName : string[40];
begin
  SectName := UpCase(Name);
  if Exists(SectName) then CritError('Trying to re-create section '+SectName+'!');
  AddObject(SectName,TINIEntryAssocArray.Create);
  Selected := TINIEntryAssocArray(LastEntry^.Value);
  if SectName = 'FLAGS'     then Flags  := Selected;
  if SectName = 'CONSTANTS' then Constants := Selected;
end;

procedure TINI.setSection(const Name : string);
var SectName : string[40];
begin
  SectName := UpCase(Name);
  if not TrySetSection(SectName) then CritError('Section ['+SectName+'] not found!');
end;

function TINI.trySetSection(const Name : string) : boolean;
var SectName : string[40];
begin
  SectName := UpCase(Name);
  trySetSection := Exists(SectName);
  if trySetSection then Selected := TINIEntryAssocArray(LastEntry^.Value);
end;

function TINI.Defined(const nKey : string) : boolean;
begin
  if Selected = nil then CritError('No Section selected when quering "'+nKey+'"!');
  Exit(Selected.Exists(nKey));
end;

procedure TINI.add(const nKey : string; const nValue : Ansistring; Section : TINIEntryAssocArray = nil );
var Entry : TINIEntry;
begin
  if Section = nil then Section := Selected;
  Selected := Section;
  if Selected = nil then CritError('No Section selected when adding "'+nKey+'"!');
  Entry.EType  := typeString; Entry.EValue := Pointer(nValue);
  Section.AddINIEntry(UpCase(nKey),Entry);
end;

procedure TINI.add(const nKey : string; const nValue : Boolean; Section : TINIEntryAssocArray = nil );
var Entry : TINIEntry;
begin
  if Section = nil then Section := Selected;
  Selected := Section;
  if Selected = nil then CritError('No Section selected when adding "'+nKey+'"!');
  Entry.EType  := typeBoolean; Entry.EValue := Pointer(LongBool(nValue));
  Section.AddINIEntry(UpCase(nKey),Entry);
end;

procedure TINI.add(const nKey : string; const nValue : Integer; Section : TINIEntryAssocArray = nil );
var Entry : TINIEntry;
begin
  if Section = nil then Section := Selected;
  Selected := Section;
  if Selected = nil then CritError('No Section selected when adding "'+nKey+'"!');
  Entry.EType  := typeNumber; Entry.EValue := Pointer(nValue);
  Section.AddINIEntry(UpCase(nKey),Entry);
end;

procedure TINI.add(const nKey : string; const nValue : TFlags; Section : TINIEntryAssocArray = nil );
var Entry : TINIEntry;
begin
  if Section = nil then Section := Selected;
  Selected := Section;
  if Selected = nil then CritError('No Section selected when adding "'+nKey+'"!');
  
  Entry.EType  := typeFlags; Entry.EValue := New(PFlags); PFlags(Entry.EValue)^ := nValue;
  Section.AddINIEntry(UpCase(nKey),Entry);
end;

function TINI.getNumber(const nkey : string) : LongInt;
var Entry : TINIEntry;
begin
  if Selected = nil then CritError('No Section selected when quering "'+nKey+'"!');
  Entry := Selected.GetINIEntry(UpCase(nkey));
  if Entry.EType <> typeNumber then CritError('Key "'+nkey+'" should be of type "Number"!');
  Exit(LongInt(Entry.EValue));
end;

function TINI.getBoolean(const nkey : string) : Boolean;
var Entry : TINIEntry;
begin
  if Selected = nil then CritError('No Section selected when quering "'+nKey+'"!');
  Entry := Selected.GetINIEntry(UpCase(nkey));
  if Entry.EType <> typeBoolean then CritError('Key "'+nkey+'" should be of type "Boolean"!');
  Exit(LongBool(Entry.EValue));
end;

function TINI.getString(const nkey : string) : AnsiString;
var Entry : TINIEntry;
begin
  if Selected = nil then CritError('No Section selected when quering "'+nKey+'"!');
  Entry := Selected.GetINIEntry(UpCase(nkey));
  if Entry.EType <> typeString then CritError('Key "'+nkey+'" should be of type "String"!');
  Exit(Ansistring(Entry.EValue));
end;

function TINI.getFlags(const nkey : string) : TFlags;
var Entry : TINIEntry;
begin
  if Selected = nil then CritError('No Section selected when quering "'+nKey+'"!');
  Entry := Selected.GetINIEntry(UpCase(nkey));
  if Entry.EType <> typeFlags then CritError('Key "'+nkey+'" should be of type "Flags"!');
  Exit(PFlags(Entry.EValue)^);
end;

function TINI.ResolveFlags(const flagString : string) : TFlags;
var FFlag      : string[40];
    FFlags     : TFlags;
    FCount     : Word;
    FlagStr    : string;
    FEntry     : TINIEntry;
begin
  if Flags = nil then CritError('No Flags Section selected when quering flagstring "'+flagstring+'"!');
  FFlags := [];
  if FlagString = '' then Exit(FFlags);
  FlagStr := UpCase(FlagString);
  FCount := 0;
  repeat
    Inc(FCount);
    FFlag := Trim(Parameter(FlagStr,FCount,'|'));
    if FFlag = '' then Break;
    FEntry := Flags[FFlag];
    if FEntry.EType <> typeNumber then CritError('Flag "'+FFlag+'" not a number!');
    Include(FFlags,LongInt(FEntry.EType));
  until FFlag = '';
  Exit(FFlags);
end;


procedure TINI.QuerySection(inicallback : TINIFileCallbackProcedure);
var Count : Word;
begin
  if Selected = nil then CritError('No Section selected!');
  with Selected do
  for Count := 0 to 95 do
  begin
    LastEntry := Entries[Count];
    while LastEntry <> nil do
    begin
      inicallback(LastEntry^.Key,EntryToString(PINIEntry(LastEntry^.Value)^));
      LastEntry := LastEntry^.Next;
    end;
  end;
end;

// Converts entry of any type to string.
function TINI.EntryToString(const Entry : TINIEntry) : Ansistring;
var c : byte;
    f : TFlags;
begin
  case Entry.EType of
    typeString  : Exit(AnsiString(Entry.EValue));
    typeNumber  : Exit(IntToStr(Integer(Entry.EValue)));
    typeBoolean : Exit(BoolToStr(LongBool(Entry.EValue)));
    typeFlags   : begin
                    f := PFlags(Entry.EValue)^;
                    EntryToString := '';
                    for c := 0 to 255 do
                      if c in f then
                        EntryToString += IntToStr(c)+ ' ';
                    Delete(EntryToString,Length(EntryToString),1);
                  end;
  end;
end;

// Returns entry converted to string.
function TINI.get(const nkey : string) : AnsiString;
begin
  if Selected = nil then CritError('No Section selected when quering "'+nKey+'"!');
  Exit(EntryToString(Selected.getINIEntry(UpCase(nkey))));
end;


destructor TINI.Done;
begin
  inherited Destroy;
end;

end.

