{$INCLUDE valkyrie.inc}
// @abstract(Unique Identification class for Valkyrie)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(May 7, 2004)
// @lastmod(Jan 14, 2006)
//
// Contains class and singleton for managing Unique Identification
// Numbers.
//
//  @html <div class="license">
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  @html </div>

unit vuid;
interface
uses vutil, vnode, vsystem;

// Manager class for UIDs. Used through the singleton @link(UIDs).
type TUIDStore = class(TSystem)
       // Standard constructor.
       constructor Create; override;
       // UID lookup. Returns the @link(TNode) referenced by given UID.
       function    Get(nUID : TUID) : TNode;
       // Frees the given UID for later use.
       procedure   Remove(nUID : TUID);
       // Registers the Node with given UID.
       procedure   Register(Node : TNode; nUID : TUID);
       // Returns a unused UID number.
       function    GetFreeID : TUID;
       // Standard destructor.
       destructor  Destroy; override;
       // Property for Get
       property UIDs[nUID : TUID] : TNode read Get; default;
     private
       IDCount : TUID;
       Data    : array [$0000..$FFFF] of TNode;
     end;

// Singleton for using @link(TUIDStore).
const UIDs : TUIDStore = nil;

implementation

constructor TUIDStore.Create;
var Count : DWord;
begin
  inherited Create;
  for Count := $0000 to $FFFF do Data[Count] := nil;
  IDCount := 0;
end;

function TUIDStore.Get(nUID : TUID) : TNode;
begin
  Get := Data[nUID];
end;

procedure TUIDStore.Remove(nUID : TUID);
begin
  Data[nUID] := nil;
end;

procedure TUIDStore.Register(Node : TNode; nUID : TUID);
begin
  Data[nUID] := Node;
end;

function TUIDStore.GetFreeID : TUID;
var OverFlow : LongInt;
begin
  Inc(IDCount);
  if Data[IDCount] = nil then Exit(IDCount);
  OverFlow := 0;
  repeat
    Inc(IDCount);
    if IDCount = $FFFF then IDCount := 1;
    if Data[IDCount] = nil then Exit(IDCount);
    Inc(OverFlow);
  until OverFlow = $FFFF;
  CritError('TUIDStore : Ran out of free UID''s!');
end;

destructor TUIDStore.Destroy;
begin
  inherited Destroy;
end;

end.
