{$INCLUDE valkyrie.inc}
// @abstract(SDL Sound system for Valkyrie)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(May 03, 2009)
// @lastmod(May 03, 2009)
//
// Implements an SDL sound system for Valkyrie
//
// Default behaviour for music is stoping the previous song before
// playing a new one.
//
// TODO : Currently the sound system opens and closes SDL. Make a separate SDL
//   from SDLVideo system

unit vsdlsound;

interface

uses Classes, SysUtils, vsdl, vsound;

// The basic sound class, published as the singleton @link(Sound).
// Should be initialized and disposed via TSystems.
type

{ TSound }

{ TSDLSound }

TSDLSound = class(TSound)
       // Initializes the Sound system.
       constructor Create; override;
       // Deinitializes the Sound system.
       destructor Destroy; override;
     protected
       // Implementation of Music Loading
       function LoadMusic( const aFileName : AnsiString; Streamed : Boolean ) : Pointer; override;
       // Implementation of Sound Loading
       function LoadSound( const aFileName : AnsiString ) : Pointer; override;
       // Implementation of Music Loading
       function LoadMusicStream( Stream : TStream; Size : DWord; Streamed : Boolean ) : Pointer; override;
       // Implementation of Sound Loading
       function LoadSoundStream( Stream : TStream; Size : DWord ) : Pointer; override;
       // Implementation of Music Freeing
       procedure FreeMusic( aData : Pointer; const aType : String ); override;
       // Implementation of Sound Freeing
       procedure FreeSound( aData : Pointer ); override;
       // Implementation of get error
       function GetError( ) : AnsiString; override;
       // Implementation of play Sound
       procedure PlaySound( aData : Pointer; aVolume : Byte; aPan : Integer = -1 ); override;
       // Implementation of play Sound
       procedure PlayMusic( aData : Pointer; const aType : string; aRepeat : Boolean = True ); override;
       // Implementation of StopMusic
       procedure StopMusic( aData : Pointer; const aType : string ); override;
       // Implementation of VolumeMusic
       procedure VolumeMusic( aData : Pointer; const aType : string; aVolume : Byte ); override;
     end;

implementation

uses SDL, SDL_Mixer, Math;

function Mix_LoadMUS_RW( src : PSDL_RWops; freesrc : integer ) : PMix_Music;  cdecl;
external SDL_MixerLibName; {$EXTERNALSYM Mix_LoadMUS}

{ TSDLSound }

constructor TSDLSound.Create;
begin
  inherited Create;
  if SDL_Init(SDL_INIT_AUDIO) < 0 then
    CritError('Can''t open SDL_Audio!');
  if Mix_OpenAudio(MIX_DEFAULT_FREQUENCY,MIX_DEFAULT_FORMAT,2,512) < 0 then
    CritError('Can''t open SDL_Mixer!');
  Mix_VolumeMusic(GetMusicVolume);
end;

destructor TSDLSound.Destroy;
begin
  inherited Destroy;
  Mix_CloseAudio();
  SDL_Quit();
end;

function TSDLSound.LoadMusic(const aFileName: AnsiString; Streamed : Boolean): Pointer;
begin
  Exit( Mix_LoadMUS( PChar( aFileName ) ) )
end;

function TSDLSound.LoadSound(const aFileName: AnsiString): Pointer;
begin
  Exit( Mix_LoadWAV( PChar( aFileName ) ) );
end;

function TSDLSound.LoadMusicStream(Stream: TStream; Size: DWord; Streamed : Boolean): Pointer;
var Data : Pointer;
begin
  if Streamed then
  begin
    Data := GetCacheMem( Size );
    Stream.Read( Data^, Size );
    Exit( Mix_LoadMUS_RW( SDL_RWFromMem( Data, Size ), 0 ) );
  end
  else
    Exit( Mix_LoadMUS_RW( RWopsFromStream( Stream, Size ), 0 ) );
end;

function TSDLSound.LoadSoundStream(Stream: TStream; Size: DWord): Pointer;
begin
  Exit( Mix_LoadWAV_RW( RWopsFromStream( Stream, Size ), 0 ) );
end;

procedure TSDLSound.FreeMusic(aData: Pointer; const aType : String );
begin
  Mix_FreeMusic(PMix_Music(aData));
end;

procedure TSDLSound.FreeSound(aData: Pointer);
begin
  Mix_FreeChunk(PMix_Chunk(aData));
end;

function TSDLSound.GetError(): AnsiString;
var iError : AnsiString;
begin
  iError := Mix_GetError();
  Exit( iError );
end;

procedure TSDLSound.PlaySound(aData: Pointer; aVolume: Byte; aPan: Integer);
var iChannel : Integer;
begin
  iChannel := Mix_PlayChannel( -1, PMix_Chunk(aData), 0 );
  Mix_Volume( iChannel, aVolume );
  if aPan <> -1 then Mix_SetPanning( iChannel, 255-aPan, aPan );
end;

procedure TSDLSound.PlayMusic(aData: Pointer; const aType : string; aRepeat: Boolean);
begin
  if aRepeat then
    Mix_PlayMusic( PMix_Music(aData), -1 )
  else
    Mix_PlayMusic( PMix_Music(aData), 1 );
end;

procedure TSDLSound.StopMusic(aData: Pointer; const aType : string );
begin
  Mix_HaltMusic();
end;

procedure TSDLSound.VolumeMusic(aData: Pointer; const aType : string; aVolume: Byte );
begin
  Mix_VolumeMusic(aVolume);
end;

initialization
  { SDL_Mixer (at least under 32-bit Linux) may make floating point 
    exceptions. Observed when trying to play midi from doomrl.
    Interesting backtrace part is:
    
    ...
    #4  0xb7e88728 in ?? () from /usr/lib/libSDL_mixer-1.2.so.0
    #5  0xb7e88b4c in ?? () from /usr/lib/libSDL_mixer-1.2.so.0
    #6  0xb7e88ee4 in load_missing_instruments () from /usr/lib/libSDL_mixer-1.2.so.0
    #7  0xb7e8b4d0 in Timidity_Start () from /usr/lib/libSDL_mixer-1.2.so.0
    #8  0xb7e83efb in ?? () from /usr/lib/libSDL_mixer-1.2.so.0
    #9  0xb7e84042 in Mix_FadeInMusicPos () from /usr/lib/libSDL_mixer-1.2.so.0
    #10 0xb7e840a2 in Mix_PlayMusic () from /usr/lib/libSDL_mixer-1.2.so.0
    #11 0x08120687 in VSDLSOUND_TSDLSOUND_$__PLAYMUSIC$POINTER$SHORTSTRING$BOOLEAN ()
    ...
    
    These exceptions can, and should, be ignored 
    (otherwise they mess up some state). }
  {$if defined(cpui386) or defined(cpux86_64)}
  SetExceptionMask([exInvalidOp, exDenormalized, exZeroDivide,exOverflow, exUnderflow, exPrecision]);
  {$endif}
end.
