// @abstract(SDL Output system for Valkyrie Output)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(Jan 22, 2007)
// @lastmod(Jan 22, 2007)
//
// Implements an SDL based output system for Valkyrie.
//
//  @html <div class="license">
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  @html </div>


{$INCLUDE valkyrie.inc}
unit vsdloutput;
interface
uses vutil, vsdl, voutput, vsdlsurface;

// The basic output class, published as the singleton @link(Output).
// Should be initialized and disposed via TSystems.
type

{ TSDLOutput }

TSDLOutput = class(TOutput)
       // Initializes the Video system.
       constructor Create(FullScreen : Boolean = False);
       // Updates (redraws) the screen.
       procedure Update; override;
       // Updates the characters.
       procedure UpdateChars;
       // Hides the cursor.
       procedure HideCursor; override;
       // Shows the cursor.
       procedure ShowCursor; override;
       // Moves the cursor to thr desired position.
       procedure MoveCursor(x,y : byte); override;
       // Puts character char to screen coordinate x,y, with color atr.
       procedure DrawChar(x,y : byte; atr : byte; chr : char; Color : TTrueColor = 0); override;
       // Draws a @link(TPicture).
       procedure DrawPicture(x,y : byte; pic : TPicture; Color : TTrueColor = 0); override;
       // Clears the whole screen memory.
       procedure Clear; override;
       // Clears a rectangle of screen memory.
       procedure ClearRect(x1,y1,x2,y2 : byte); override;
       // Clears a rectangle of screen memory with given color.
       procedure ClearRectColor(x1,y1,x2,y2,atr : byte; Color : TTrueColor = 0);override;
       // Deinitializes the Video system.
       destructor Destroy; override;
       // Outputs a single character onto the OpenGL screen (no color check)
       procedure OutputChar(x,y : Byte; c : char); inline;
       // Sets the application title
       procedure SetTitle(Long,Short : Ansistring); override;
       private
       // Cursor position
       CursorX,CursorY : Byte;
       // Cursor visibility
       CursorV : Boolean;
       // Width and height of a single character in GL mode
       WW,HH : Real;
       Font : TSurface;
       Buf  : array[1..80,1..25] of record c,a : Byte; end;
     end;

const GLColors : array[0..15] of array[0..2] of Single = (
      ( 0.0, 0.0, 0.0 ),
      ( 0.0, 0.0, 0.6 ),
      ( 0.0, 0.6, 0.0 ),
      ( 0.0, 0.6, 0.6 ),
      ( 0.6, 0.0, 0.0 ),
      ( 0.6, 0.0, 0.6 ),
      ( 0.6, 0.6, 0.0 ),
      ( 0.8, 0.8, 0.8 ),
      ( 0.5, 0.5, 0.5 ),
      ( 0.0, 0.0, 1.0 ),
      ( 0.0, 1.0, 0.0 ),
      ( 0.0, 1.0, 1.0 ),
      ( 1.0, 0.0, 0.0 ),
      ( 1.0, 0.0, 1.0 ),
      ( 1.0, 1.0, 0.0 ),
      ( 1.0, 1.0, 1.0 )
      );

implementation

uses math, vsystems, GL, GLU, SDL, SysUtils;


const ClearCell : Word = Ord(' ')+(LightGray shl 8);

constructor TSDLOutput.Create(FullScreen : Boolean = False);
begin
  inherited Create;
  if Fullscreen then Systems.Add(ValkyrieSDL,TValkyrieSDL.Create([VSDL800x600,VSDLOPENGL,VSDLFULLSCREEN]))
                else Systems.Add(ValkyrieSDL,TValkyrieSDL.Create([VSDL800x600,VSDLOPENGL]));
  Font := TSurface.Create('font10x18.png');
  Font.SetColorKey(0,0,0);
  Font.RenderGL;
  
  WW := Font.GLWidth / 32;
  HH := Font.GLHeight / 7;
end;

procedure TSDLOutput.UpdateChars;
var x,y,v : Byte;
    time  : UInt32;
    phase : Word;
    cv,ch : Byte;
    DrawC : Boolean;
begin
  DrawC := False;
  if CursorV then
  begin
    time := SDL_GetTicks;
    phase := time mod 600;
    if (phase > 300) then DrawC := true;
  end;
  
  
  glBindTexture(GL_TEXTURE_2D, Font.GLTexture);
  glBegin(GL_QUADS);
  glColor3f(0.0,0.0,0.0);
  v := 0;
  for x := 1 to 80 do
    for y := 1 to 25 do
    with Buf[x,y] do
    begin
      if (c = 0) or (a = 0) then Continue;
      //if a = 0 then Continue;
      if (a mod 16) <> v then
      begin
        v := a mod 16;
        glColor3f(GLColors[v][0],GLColors[v][1],GLColors[v][2]);
      end;
      OutputChar(x,y,char(c+32));
    end;
    if DrawC then
    begin
      glColor3f(1.0,1.0,1.0);
      OutputChar(CursorX,CursorY,Chr(219));
    end;
  glEnd();
end;

procedure TSDLOutput.Update;
begin
  glClearColor(0.0,0.0,0.0,1.0);
  glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);

  glEnable( GL_TEXTURE_2D );
  glDisable( GL_DEPTH_TEST );
  glEnable( GL_BLEND );

  glMatrixMode( GL_PROJECTION );
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0, ValkyrieSDL.Width-1, ValkyrieSDL.Height-1,0);

  glMatrixMode( GL_MODELVIEW );
  glPushMatrix();
  glLoadIdentity();
  glColor4f(1.0,1.0,1.0,1.0);
  glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );

  if UI <> nil then UI.Draw;
  UpdateChars;

  SDL_GL_SwapBuffers();
end;

procedure TSDLOutput.HideCursor;
begin
  CursorV := False;
end;

procedure TSDLOutput.ShowCursor;
begin
  CursorV := True;
end;

procedure TSDLOutput.MoveCursor(x,y : byte);
begin
  CursorX := x;
  CursorY := y;
end;

procedure TSDLOutput.DrawChar(x,y : byte; atr : byte; chr : char; Color : TTrueColor = 0);
begin
  if not (x in [1..80]) then Exit;
  if not (y in [1..25]) then Exit;
  if not (Ord(chr) in [32..255]) then Exit;
  Buf[x,y].c := Ord(chr)-32;
  Buf[x,y].a := atr;
end;

procedure TSDLOutput.DrawPicture(x,y : byte; pic : TPicture; Color : TTrueColor = 0);
begin
  DrawChar(x,y,pic div 256, Char(pic mod 256));
end;

procedure TSDLOutput.Clear;
var x,y : byte;
begin
  for x := 1 to 80 do
    for y := 1 to 25 do
    begin  
      Buf[x,y].c := 0;
      Buf[x,y].a := 0;
    end;  
end;

procedure TSDLOutput.ClearRect(x1,y1,x2,y2 : byte);
var x,y : byte;
begin
  for y := y1 to y2 do
    for x := x1 to x2 do
    begin  
      Buf[x,y].c := 0;
      Buf[x,y].a := 0;
    end;  
end;

procedure TSDLOutput.ClearRectColor(x1,y1,x2,y2,atr : byte; Color : TTrueColor = 0);
var x,y : byte;
begin
  for y := y1 to y2 do
    for x := x1 to x2 do
    begin  
      Buf[x,y].c := 0;
      Buf[x,y].a := atr;
    end;  
end;

destructor TSDLOutput.Destroy;
begin
//  FreeAndNil(Font);
  inherited Destroy;
end;

procedure TSDLOutput.OutputChar(x, y: Byte; c: char); inline;
var cv, ch : Byte;
begin
  cv := (Ord(c)-32) mod 32;
  ch := (Ord(c)-32) div 32;
  glTexCoord2f(cv*WW,ch*HH);         glVertex2i((x-1)*10, (y-1)*24);
  glTexCoord2f(cv*WW,(ch+1)*HH);     glVertex2i((x-1)*10, y*24-6);
  glTexCoord2f((cv+1)*WW,(ch+1)*HH); glVertex2i(x*10, y*24-6);
  glTexCoord2f((cv+1)*WW,ch*HH);     glVertex2i(x*10, (y-1)*24);
end;

procedure TSDLOutput.SetTitle(Long, Short: Ansistring);
begin
  if Short = '' then Short := Long;
  SDL_WM_SetCaption(PChar(Long),PChar(Short));
end;


end.

{ $Log:
}
