//  @abstract(FreePascal Valkyrie SDL - Valkyrie Font)
//  @author  (Kornel Kisielewicz <charon@magma-net.pl>)
//  
//  This library will provide several methods of SDL font rendering. Currently
//  just rendering via SDL_ttf is supported. Mono-fonts and bitmap fonts are
//  planned.
//  
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  
//  ToDo@br
//   -- BitmapFont@br
//   -- MonoFont@br
//   -- Implement additional capabilities


{$INCLUDE valkyrie.inc}
unit vsdlfont;
interface
uses vds, vnode, vsdl, vsystem, SDL, SDL_ttf, GL, GLu;

// Font quality type (for description see SDL_ttf documentation)
type TFontQuality = (FQSolid,FQShaded,FQBlended);

// Abstract base class for all Valkyrie Font classes
// All new font systems should inherit this class. All methods are abstract and
// should be overloaded.
type

{ TFont }

TFont = class(TVObject)
       // Creates a surface with the given text
       function TypeText(const text : ansistring; fg : TColor) : PSDL_Surface; virtual; abstract;
       // Returns the estimated text width
       function GetWidth(const text : ansistring) : Integer; virtual; abstract;
       // Returns the estimated text height
       function GetHeight(const text : ansistring) : Integer; virtual; abstract;
     end;

// Class for TTF Fonts - rendered using SDL_TTF
// See SDL_ttf for specific behaviour. Remember that the result of the typed
// text is a NEW surface, and disposing it is the responsibility of the user.
type TTTFFont = class(TFont)
       // Loads the given TTF font
       constructor Create(const filename : ansistring; ptsize : integer; quality : TFontQuality);
       // Creates a surface with the given text
       function TypeText(const text : ansistring; fg : TColor) : PSDL_Surface; override;
       // Returns the estimated text width
       function GetWidth(const text : ansistring) : Integer; override;
       // Returns the estimated text height
       function GetHeight(const text : ansistring) : Integer; override;
       // Frees memory
       destructor Destroy; override;
       private
       FontQuality : TFontQuality;
       FontData    : PTTF_Font;
     end;


type TFontArray = specialize TManagedArray<TFont>;

// The System class for the Font Engine
// The TVFonts class must be initaited before any font classes are used.
// Currently except taking care of initialization and finalization this system
// has no other uses.
type

{ TFonts }

TFonts = class(TSystem)
       private
       function getFont(index : DWord) : TFont;
       procedure setFont(index : DWord; const nFont : TFont);
       public
       // Registers the font system.
       constructor Create; override;
       // Closes the font system.
       destructor Destroy; override;
       // Access property for handling fonts.
       property Font[index : DWord] : TFont read getFont write setFont; default;
       FontArray : TFontArray;
     end;

const Fonts : TFonts = nil;

implementation
uses SysUtils;

type A4GLfloat = array[0..3] of GLfloat;



function TFonts.GetFont(index : DWord) : TFont;
begin
  if FontArray[index] = nil then CritError('Font ID#'+IntToStr(index)+' not found!');
  Exit(FontArray[index])
end;

procedure TFonts.SetFont(index : DWord; const nFont : TFont);
begin
  if FontArray[index] <> nil       then CritError('Trying to reassign font ID#'+IntToStr(index)+'!');
  if not nFont.InheritsFrom(TFont) then CritError('Non font class passed to TFonts!');
  FontArray[index] := nFont;
end;

constructor TFonts.Create;
begin
  inherited Create;
  TTF_Init;
  FontArray := TFontArray.Create;
end;

destructor TFonts.Destroy;
begin
  FreeAndNil(FontArray);
  TTF_Quit;
  inherited Destroy;
end;


constructor TTTFFont.Create(const filename : ansistring; ptsize : integer; quality : TFontQuality );
begin
  FontData    := TTF_OpenFont(PChar(filename), ptsize);
  FontQuality := Quality;
end;

function TTTFFont.TypeText(const text : ansistring; fg : TColor) : PSDL_Surface;
const BLACK : TSDL_Color = (r:0;g:0;b:0;);
begin
  case FontQuality of
    FQSolid   : Exit(TTF_RenderText_Solid( FontData , PChar(text) , fg.SDLColor ));
    FQShaded  : Exit(TTF_RenderText_Shaded( FontData , PChar(text) , fg.SDLColor , BLACK));
    FQBlended : Exit(TTF_RenderText_Blended( FontData , PChar(text) , fg.SDLColor ));
  end;
end;

// Returns the estimated text width
function TTTFFont.GetWidth(const text : ansistring) : Integer;
var temp : Integer;
begin
  TTF_SizeText( FontData, PChar(text), GetWidth, temp );
end;

// Returns the estimated text height
function TTTFFont.GetHeight(const text : ansistring) : Integer;
var temp : Integer;
begin
  TTF_SizeText( FontData, PChar(text), temp, GetHeight);
end;


destructor TTTFFont.Destroy;
begin
  TTF_CloseFont(FontData);
end;

end.

