// @abstract(SDL Output system for Valkyrie Output)
// @author(Kornel Kisielewicz <kisiel@fulbrightweb.org>)
// @created(Jan 22, 2007)
// @lastmod(Jan 22, 2007)
//
// Implements an SDL based output system for Valkyrie.
//
//  @html <div class="license">
//  This library is free software; you can redistribute it and/or modify it
//  under the terms of the GNU Library General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or (at your
//  option) any later version.
//
//  This program is distributed in the hope that it will be useful, but WITHOUT
//  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
//  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
//  for more details.
//
//  You should have received a copy of the GNU Library General Public License
//  along with this library; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//  @html </div>


{$INCLUDE valkyrie.inc}
unit vgloutput;
interface
uses Classes, SysUtils, vsystems, GL, GLU, SDL, vutil, vsdlsurface, vsdl, voutput;

// The basic output class, published as the singleton @link(Output).
// Should be initialized and disposed via TSystems.
type

{ TSDLOutput }

{ TGLOutput }

TGLOutput = class(TOutput)
       // Initializes the Video system.
       constructor Create(FullScreen : Boolean = False; Widescreen : Boolean = false);  reintroduce;
       // Initializes the Video system.
       constructor Create(aWidth, aHeight : Word; FullScreen : Boolean = False);  reintroduce;
       // Updates (redraws) the screen.
       procedure Update; override;
       // Updates the characters.
       procedure UpdateChars;
       // Hides the cursor.
       procedure HideCursor; override;
       // Shows the cursor.
       procedure ShowCursor; override;
       // Moves the cursor to thr desired position.
       procedure MoveCursor(x,y : byte); override;
       // Puts character char to screen coordinate x,y, with color atr.
       procedure DrawChar(x,y : byte; atr : byte; chr : char; Color : TTrueColor = 0); override;
       // Draws a @link(TPicture).
       procedure DrawPicture(x,y : byte; pic : TPicture; Color : TTrueColor = 0); override;
       // Clears the whole screen memory.
       procedure Clear; override;
       // Clears a rectangle of screen memory.
       procedure ClearRect(x1,y1,x2,y2 : byte); override;
       // Clears a rectangle of screen memory with given color.
       procedure ClearRectColor(x1,y1,x2,y2,atr : byte; Color : TTrueColor = 0);override;
       // Deinitializes the Video system.
       destructor Destroy; override;
       // Outputs a single character onto the OpenGL screen (no color check)
       procedure OutputChar(x,y : Byte; c : char); inline;
       // Sets the application title
       procedure SetTitle(Long,Short : Ansistring); override;
       // Toggles fullscreen
       procedure ToggleFullScreen; override;
       // Creates a screenshot file at the given name
       function ScreenShot(const FileName : string; stype : byte = 0) : Boolean; override;
       // Returns the extension for the screenshots files.
       function ScreenShotExt(stype : byte = 0) : string; override;

       public
       LineHeight   : Byte;
       LineSkip     : Byte;
       HCollumnWidth: Byte;
       HLineHeight  : Byte;
       CollumnWidth : Byte;
       GylphHeight  : Byte;
       GylphWidth   : Byte;
       SkipY        : Byte;
       function TranslateRect(const Rect : TRect) : TRect;
       function TranslateX(const X : Integer) : Integer;
       function TranslateY(const Y : Integer) : Integer;
       procedure Quad(Rect : TRect);
       procedure Frame(Rect : TRect);
       procedure Line(x1,y1,x2,y2 : Integer);
       private
       procedure Setup;
       private
       // Cursor position
       CursorX,CursorY : Byte;
       // Cursor visibility
       CursorV : Boolean;
       // Width and height of a single character in GL mode
       WW,HH : Real;
       Font : TSurface;
       Buf  : array of array of record c : Byte; color : TTrueColor; end;
       
     end;

const GLColors : array[0..15] of array[0..3] of Single = (
      ( 0.0, 0.0, 0.0, 1.0 ),
      ( 0.0, 0.0, 0.6, 1.0 ),
      ( 0.0, 0.6, 0.0, 1.0 ),
      ( 0.0, 0.6, 0.6, 1.0 ),
      ( 0.6, 0.0, 0.0, 1.0 ),
      ( 0.6, 0.0, 0.6, 1.0 ),
      ( 0.6, 0.6, 0.0, 1.0 ),
      ( 0.8, 0.8, 0.8, 1.0 ),
      ( 0.5, 0.5, 0.5, 1.0 ),
      ( 0.0, 0.0, 1.0, 1.0 ),
      ( 0.0, 1.0, 0.0, 1.0 ),
      ( 0.0, 1.0, 1.0, 1.0 ),
      ( 1.0, 0.0, 0.0, 1.0 ),
      ( 1.0, 0.0, 1.0, 1.0 ),
      ( 1.0, 1.0, 0.0, 1.0 ),
      ( 1.0, 1.0, 1.0, 1.0 )
      );
var GLColorsPacked : array[0..15] of DWord;

var GLOutput : TGLOutput;

implementation

uses vdebug, png;

//const ClearCell : Word = Ord(' ')+(LightGray shl 8);

constructor TGLOutput.Create(FullScreen : Boolean = False;  Widescreen : Boolean = false);
begin
  SetLength(Buf,80+1,25+1);
  inherited Create;
  if Fullscreen then Systems.Add(ValkyrieSDL,TValkyrieSDL.Create([VSDL800x600,VSDLOPENGL,VSDLFULLSCREEN]))
                else Systems.Add(ValkyrieSDL,TValkyrieSDL.Create([VSDL800x600,VSDLOPENGL]));


  if WideScreen then
  begin
    LineHeight   := 18;
    LineSkip     := 0;
    SkipY        := (600-LineHeight*25) div 2;
  end
  else
  begin
    LineHeight   := 24;
    LineSkip     := 6;
    SkipY        := 0;
  end;

  Setup;
end;

constructor TGLOutput.Create(aWidth, aHeight: Word; FullScreen: Boolean);
begin
  ScreenSizeX  := aWidth  div 10;
  ScreenSizeY  := aHeight div 18;
  ScreenSize   := ScreenSizeX * ScreenSizeY;

  SetLength(Buf,ScreenSizeX+1,ScreenSizeY+1);

  inherited Create;
  if Fullscreen then Systems.Add(ValkyrieSDL,TValkyrieSDL.Create(aWidth,aHeight,32,[VSDLOPENGL,VSDLFULLSCREEN]))
                else Systems.Add(ValkyrieSDL,TValkyrieSDL.Create(aWidth,aHeight,32,[VSDLOPENGL]));

  LineHeight   := 18;
  LineSkip     := 0;
  SkipY        := 0;

  Setup;
end;

procedure TGLOutput.UpdateChars;
var x,y   : Byte;
    v     : DWord;
    time  : UInt32;
    phase : Word;
    DrawC : Boolean;
begin
  DrawC := False;
  if CursorV then
  begin
    time := SDL_GetTicks;
    phase := time mod 600;
    if (phase > 300) then DrawC := true;
  end;
  
  
  glBindTexture(GL_TEXTURE_2D, Font.GLTexture);
  glBegin(GL_QUADS);
  glColor3f(0.0,0.0,0.0);
  v := 0;
  for x := 1 to ScreenSizeX do
    for y := 1 to ScreenSizeY do
    with Buf[x,y] do
    begin
      if (c = 0) or (color = 0) then Continue;
      //if a = 0 then Continue;
      if color <> v then
      begin
        v := color;
        if TTrueColorRec(color)[3] <> 255
          then glColor4ubv(@color)
          else glColor3ubv(@color);
      end;
      OutputChar(x,y,char(c+32));
    end;
    if DrawC then
    begin
      glColor3f(1.0,1.0,1.0);
      OutputChar(CursorX,CursorY,Chr(219));
    end;
  glEnd();
end;

procedure TGLOutput.Update;
begin
  glClearColor(0.0,0.0,0.0,1.0);
  glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);

  glEnable( GL_TEXTURE_2D );
  glDisable( GL_DEPTH_TEST );
  glEnable( GL_BLEND );

  glMatrixMode( GL_PROJECTION );
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0, ValkyrieSDL.Width-1, ValkyrieSDL.Height-1,0);

  glMatrixMode( GL_MODELVIEW );
  glPushMatrix();
  glLoadIdentity();
  glColor4f(1.0,1.0,1.0,1.0);
  glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );

  if UI <> nil then UI.Draw;
  UpdateChars;

  SDL_GL_SwapBuffers();
end;

procedure TGLOutput.HideCursor;
begin
  CursorV := False;
end;

procedure TGLOutput.ShowCursor;
begin
  CursorV := True;
end;

procedure TGLOutput.MoveCursor(x,y : byte);
begin
  CursorX := x;
  CursorY := y;
end;

procedure TGLOutput.DrawChar(x,y : byte; atr : byte; chr : char; Color : TTrueColor = 0);
begin
  if not (x in [1..ScreenSizeX]) then Exit;
  if not (y in [1..ScreenSizeY]) then Exit;
  if not (Ord(chr) in [32..255]) then Exit;
  Buf[x,y].c := Ord(chr)-32;
  if color = 0 then
    Buf[x,y].color := GLColorsPacked[atr mod 16]
  else
    Buf[x,y].color := color;
end;

procedure TGLOutput.DrawPicture(x,y : byte; pic : TPicture; Color : TTrueColor = 0);
begin
  DrawChar(x,y,pic div 256, Char(pic mod 256),color);
end;

procedure TGLOutput.Clear;
var x,y : byte;
begin
  for x := 1 to ScreenSizeX do
    for y := 1 to ScreenSizeY do
    begin  
      Buf[x,y].c := 0;
      Buf[x,y].color := 0;
    end;  
end;

procedure TGLOutput.ClearRect(x1,y1,x2,y2 : byte);
var x,y : byte;
begin
  for y := y1 to y2 do
    for x := x1 to x2 do
    begin  
      Buf[x,y].c := 0;
      Buf[x,y].color := 0;
    end;  
end;

procedure TGLOutput.ClearRectColor(x1,y1,x2,y2,atr : byte; Color : TTrueColor = 0);
var x,y : byte;
begin
  if color = 0 then color := GLColorsPacked[atr mod 16];
  for y := y1 to y2 do
    for x := x1 to x2 do
    begin  
      Buf[x,y].c := 0;
      Buf[x,y].color := color;
    end;  
end;

destructor TGLOutput.Destroy;
begin
//  FreeAndNil(Font);
  inherited Destroy;
end;

procedure TGLOutput.OutputChar(x, y: Byte; c: char); inline;
var cv, ch : Byte;
begin
  cv := (Ord(c)-32) mod 32;
  ch := (Ord(c)-32) div 32;
  glTexCoord2f(cv*WW,ch*HH);         glVertex2i((x-1)*CollumnWidth, (y-1)*LineHeight);
  glTexCoord2f(cv*WW,(ch+1)*HH);     glVertex2i((x-1)*CollumnWidth, y*LineHeight-LineSkip);
  glTexCoord2f((cv+1)*WW,(ch+1)*HH); glVertex2i(x*CollumnWidth, y*LineHeight-LineSkip);
  glTexCoord2f((cv+1)*WW,ch*HH);     glVertex2i(x*CollumnWidth, (y-1)*LineHeight);
end;

procedure TGLOutput.SetTitle(Long, Short: Ansistring);
begin
  if Short = '' then Short := Long;
  SDL_WM_SetCaption(PChar(Long),PChar(Short));
end;

procedure TGLOutput.ToggleFullScreen;
begin
  ValkyrieSDL.ToggleFullScreen;
  Font.RenderGL;
  Update;
end;

procedure user_error_fn(png_ptr: png_structp; error_msg: png_const_charp); cdecl;
begin
  raise Exception.Create(error_msg);
end;

procedure user_read_data(png_ptr: png_structp; data: png_bytep; length: png_size_t); cdecl;
var
  inFile: TFileStream;
begin
  inFile := TFileStream(png_get_io_ptr(png_ptr));
  inFile.Read(data^, length);
end;

procedure user_write_data(png_ptr: png_structp; data: png_bytep; length: png_size_t); cdecl;
var
  outFile: TFileStream;
begin
  outFile := TFileStream(png_get_io_ptr(png_ptr));
  outFile.Write(data^, length);
end;

procedure user_flush_data(png_ptr: png_structp); cdecl;
begin
end;

function TGLOutput.ScreenShot(const FileName: string; stype: byte) : Boolean;
var
  w, h     : Integer;
  fname    : AnsiString;
  png_ptr  : png_structp;
  info_ptr : png_infop;
  PNGFile  : TFileStream;
  Count    : Integer;
  Data     : array of png_bytep;
  pixels   : PByte;
begin
  ScreenShot := False;
  fname := FileName;
  w := ValkyrieSDL.width;
  h := ValkyrieSDL.height;

  try
    pngFile := TFileStream.Create(fName, fmCreate);
  except
    Log(LERROR,'ScreenShot "@1" create failed!',[fname]);
    Exit;
  end;

  pixels := GetMem(w * h * 3);
  glReadPixels(0, 0, w, h, GL_RGB, GL_UNSIGNED_BYTE, pixels);

  // setup data
  SetLength(Data, h);
  for Count := 0 to h-1 do
    Data[Count] := @PChar(pixels)[(h-Count-1) * w * 3];

  png_ptr := nil;

  try
    png_ptr := png_create_write_struct(PNG_LIBPNG_VER_STRING, nil, @user_error_fn, nil);

    if png_ptr = nil then Exit(False);

    info_ptr := png_create_info_struct(png_ptr);

    png_set_write_fn(png_ptr, pngFile, @user_write_data, @user_flush_data);
    png_set_IHDR(
      png_ptr, info_ptr,
      w, h,
      8,
      2{PNG_COLOR_TYPE_RGB},
      0{PNG_INTERLACE_NONE},
      0{PNG_COMPRESSION_TYPE_DEFAULT},
      0{PNG_FILTER_TYPE_DEFAULT}
    );

    png_write_info(png_ptr, info_ptr);
    png_write_image(png_ptr, png_bytepp(Data));
    png_write_end(png_ptr, nil);

    ScreenShot := True;
  except on e : Exception do
    Log( LERROR, 'ScreenShot "@1" error: @2', [fname, e.message]);
  end;

  FreeMem(pixels);

  SetLength(Data, 0);

  if png_ptr <> nil then
    png_destroy_write_struct(@png_ptr, nil);

  pngFile.Free;
end;

function TGLOutput.ScreenShotExt(stype: byte): string;
begin
  Exit('png');
end;

function TGLOutput.TranslateRect(const Rect : TRect) : TRect;
begin
  TranslateRect.x1 := Rect.x1*CollumnWidth-HCollumnWidth;
  TranslateRect.y1 := Rect.y1*LineHeight-HLineHeight;
  TranslateRect.x2 := Rect.x2*CollumnWidth-HCollumnWidth;
  TranslateRect.y2 := Rect.y2*LineHeight-HLineHeight;
end;

function TGLOutput.TranslateX(const X: Integer): Integer;
begin
  TranslateX := X*CollumnWidth-HCollumnWidth;
end;

function TGLOutput.TranslateY(const Y: Integer): Integer;
begin
  TranslateY := Y*LineHeight-HLineHeight;
end;

procedure TGLOutput.Quad(Rect: TRect);
begin
  Rect := TranslateRect(Rect);
  {glTexCoord2f(tcx1,tcy1); }glVertex2i(Rect.x1, Rect.y1);
  {glTexCoord2f(tcx1,tcy2); }glVertex2i(Rect.x1, Rect.y2);
  {glTexCoord2f(tcx2,tcy2); }glVertex2i(Rect.x2, Rect.y2);
  {glTexCoord2f(tcx2,tcy1); }glVertex2i(Rect.x2, Rect.y1);
end;

procedure TGLOutput.Frame(Rect: TRect);
begin
  Rect := TranslateRect(Rect);
  glVertex2i(Rect.x1,Rect.y1);
  glVertex2i(Rect.x1,Rect.y2);
  glVertex2i(Rect.x1,Rect.y2);
  glVertex2i(Rect.x2,Rect.y2);
  glVertex2i(Rect.x2,Rect.y2);
  glVertex2i(Rect.x2,Rect.y1);
  glVertex2i(Rect.x2,Rect.y1);
  glVertex2i(Rect.x1,Rect.y1);
end;

procedure TGLOutput.Line(x1, y1, x2, y2 : Integer);
begin
  glVertex2i(x1*CollumnWidth-HCollumnWidth, y1*LineHeight-HLineHeight);
  glVertex2i(x2*CollumnWidth-HCollumnWidth, y2*LineHeight-HLineHeight);
end;

procedure TGLOutput.Setup;
var Count : Byte;
    CRec  : TTrueColorRec;
begin
  Font := TSurface.Create('font10x18.png');
  Font.SetColorKey(0,0,0);
  Font.RenderGL;

  CollumnWidth := 10;
  GylphHeight  := 18;
  GylphWidth   := 10;

  HLineHeight   := LineHeight div 2;
  HCollumnWidth := CollumnWidth div 2;

  WW := Font.GLWidth / 32;
  HH := Font.GLHeight / 7;

  for Count := 0 to 15 do
  begin
    CRec[0] := Round(GLColors[Count][0]*255);
    CRec[1] := Round(GLColors[Count][1]*255);
    CRec[2] := Round(GLColors[Count][2]*255);
    CRec[3] := 255;
    GLColorsPacked[Count] := DWord(CRec);
  end;

  GLOutput := Self;
end;


end.

{ $Log:
}
