// @abstract(BerserkRL -- User Interface/CharGen class)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(Jan 9, 2007)
// @lastmod(Jan 9, 2007)
//
// This unit holds the User Interface Chargen class of Berserk!. It's a subclass
// created to cut the size of the BerserkUI class.
//
//  @html <div class="license">
//  This file is part of BerserkRL.
//
//  BerserkRL is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  BerserkRL is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with BerserkRL; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//  @html </div>unit brui;

unit bruichar;
{$mode objfpc}
interface
uses brdata;

// Result of TBerserk.CharGen.Stats
type TChargenStats = array[1..5] of Byte;
// Type for TBerserk.CharGen.Stats
type TChargenUpgradeType = (statsMassacre,statsFirst,statsUpgrade);

// The Berserk chargen class. Static.
type

{ TBerserkCharGen }

TBerserkCharGen = class
    // Choosing the name.
    class function Name : string;
    // Choosing stats.
    class function Stats(Initial : TChargenStats; UpgradeType : TChargenUpgradeType) : TChargenStats;
    // Choosing battlefield.
    class function Arena : Byte;
    // Gamemode choice menu.
    class function GameMode : TGameMode;
    // Show character screen.
    class procedure Screen;
    // Skill upgrade screen.
    class procedure Skill;
    // Creates a TChargenStats array from the given values.
    class function NewStats(Strength, Dexterity, Endurance, Willpower, Points : Word) : TChargenStats;
    private
    class procedure SkillDesc(SkillID : byte);
  end;

implementation

uses SysUtils, vutil, voutput, vinput, vtextut, brui, brmain, brplayer;

class function TBerserkChargen.Name : string;
begin
  Name := '';
  UI.DrawFire;
  UI.Window(30,11,50,14);
  Output.DrawString(32,12,LightGray,'What''s your name?');
  Name := Trim(TextInputField(32,13,16));
  UI.Clear;
end;

class function TBerserkChargen.Stats(Initial : TChargenStats; UpgradeType : TChargenUpgradeType) : TChargenStats;
type TStat = record Name : string; Cost : byte; end;
var Choice,Key : Byte;
    Choices     : array[1..5] of TStat = (
     ( Name : 'Strength';  Cost : 1; ),
     ( Name : 'Dexterity'; Cost : 2; ),
     ( Name : 'Endurance'; Cost : 1; ),
     ( Name : 'Willpower'; Cost : 1; ),
     ( Name : 'Done';      Cost : 0; ));
  procedure DrawStats;
  var cnt : byte;
  begin
    Output.DrawString(16,10,LightGray,'Points : @<@1   ',[Stats[5]]);
    for cnt := 1 to 5 do
    begin
      if Choice = cnt then Output.DrawString(16,11+cnt,White,Choices[cnt].Name)
                      else Output.DrawString(16,11+cnt,DarkGray,Choices[cnt].Name);
      if cnt < 5      then Output.DrawString(27,11+cnt,DarkGray,'[@L@1@d]',[Stats[cnt]]);
    end;
    Output.ClearRect(37,11,65,18);
    if Choice < 5 then Output.DrawString(37,12,DarkGray,'@L@1 @l( Cost: @L@2@l )   ',[Choices[Choice].Name,Choices[Choice].Cost]);
    case Choice of
      1 : Output.DrawString(39,14,LightGray,'-- affects damage');
      2 : begin
            Output.DrawString(39,14,LightGray,'-- affects hit chance');
            Output.DrawString(39,15,LightGray,'-- affects accuracy');
            Output.DrawString(39,16,LightGray,'-- affects dodge');
            Output.DrawString(39,17,LightGray,'-- affects speed');
          end;
      3 : begin
            Output.DrawString(39,14,LightGray,'-- increases hitpoints');
            Output.DrawString(39,15,LightGray,'-- increases energy');
            Output.DrawString(39,16,LightGray,'-- decreases knockback');
          end;
      4 : begin
            Output.DrawString(39,14,LightGray,'-- increases energy');
            Output.DrawString(39,15,LightGray,'-- quickens pain recovery');
            Output.DrawString(39,16,LightGray,'-- quickens energy recovery');
            Output.DrawString(39,17,LightGray,'-- affects berserking');
          end;
      5 : if UpgradeType = statsMassacre
            then Output.DrawString(37,12,LightGray,'Press @<Enter@> to accept the chosen stats. Excess points will be lost.',30)
            else Output.DrawString(37,12,LightGray,'Press @<Enter@> to accept the chosen stats. Excess points will be kept.',30);
    end;
  end;
begin
  UI.DrawFire;
  UI.Window(14,5,66,18);
  Output.ClearRect(37,11,65,18);
  case UpgradeType of
    statsFirst    : Output.DrawString(16,6,LightGray,'Choose your basic statistics. Up and down to navigate, right to increase, and left lower. Unspent points may be used in the next avancement.',47);
    statsMassacre : Output.DrawString(16,7,LightGray,'Choose your basic statistics. Up and down to navigate, right to increase, and left lower.',46);
    statsUpgrade  : Output.DrawString(16,6,LightGray,'Choose which statistics to upgrade. Up and down to navigate, right to increase, and left lower. Unspent points will be kept.',47);
  end;

  for Choice := 1 to 5 do Stats[Choice] := Initial[Choice];

  Output.HideCursor;
  Choice := 1;
  DrawStats;
  repeat
    Key := Input.GetKey([VKEY_UP, VKEY_DOWN, VKEY_LEFT, VKEY_RIGHT, VKEY_ENTER]);
    case Key of
      VKEY_UP    : if Choice = 1 then Continue else Dec(Choice);
      VKEY_DOWN  : if Choice = 5 then Continue else Inc(Choice);
      VKEY_LEFT  : if Choice < 5 then with Choices[Choice] do if Stats[Choice] > Initial[Choice] then begin Stats[5] += Cost; Dec(Stats[Choice]); end;
      VKEY_RIGHT : if Choice < 5 then with Choices[Choice] do if Stats[5] >= Cost                then begin Stats[5] -= Cost; Inc(Stats[Choice]); end;
    end;
    DrawStats;
  until (Key = VKEY_ENTER) and (Choice = 5);

  UI.Clear;
  Output.ShowCursor;
end;

class function TBerserkChargen.Arena : Byte;
var Menu : TTextMenu;
begin
  // Screen 3 -- Battlefield
  UI.DrawFire;
  UI.Window(10,11,60,16);
  Output.DrawString(12,12, LightGray,'Choose your the arena you want to fight');
  Output.DrawString(12,13, LightGray,'on. Fields are empty, with only a few');
  Output.DrawString(12,14, LightGray,'rocks for cover. The forest has trees,');
  Output.DrawString(12,15, LightGray,'The city has lots of cover.');

  UI.Window(65,11,75,18);

  Menu := TTextMenu.Create(67,12);
  Menu.Add('Fields');
  Menu.Add('Forest');
  Menu.Add('Town');
  Menu.Add('River');
  Menu.Add('Bridge');
  Menu.Add('Snow');
  Arena := Menu.Run;
  FreeAndNil(Menu);

  UI.Clear;
  Output.ShowCursor;
end;

class function TBerserkChargen.GameMode : TGameMode;
var Menu : TTextMenu;
    Choice : Byte;
begin
  UI.DrawFire;
  UI.Window(10,10,60,16);
  Output.DrawString(12,11,LightGray,'Please choose game mode. @<Endless@> is a mode');
  Output.DrawString(12,12,LightGray,'where you fight waves of monsters each night,');
  Output.DrawString(12,13,LightGray,'gaining experience after each survived arena.');
  Output.DrawString(12,14,LightGray,'@<Massacre@> is just what it says -- pure bloodshed');
  Output.DrawString(12,15,LightGray,'without any distractions.');

  UI.Window(65,10,76,14);

  Menu := TTextMenu.Create(67,11);
//  Menu.Add(ModeToString(modeCampaign));
  Menu.Add(ModeToString(modeEndless));
  Menu.Add(ModeToString(modeMassacre));
  Choice := Menu.Run;
  case Choice+1 {/////////////////////} of
    1 : GameMode := modeCampaign;
    2 : GameMode := modeEndless;
    3 : GameMode := modeMassacre;
  end;
  FreeAndNil(Menu);
  UI.Clear;
end;

class procedure TBerserkCharGen.Screen;
var Count,C2 : Byte;
begin
with Player do
begin
  UI.DrawFire;
  UI.Window(7,3,73,23);
  Output.DrawString(8,3,LightGray,Padded('-',65,'-'));
  Output.DrawString(10,5,LightGray,'Name      : @<@1@>',[Name]);
  Output.DrawString(10,6,LightGray,'Game type : @<@1@>',[ModeToString(Player.Mode)]);
  
  Output.DrawString(42,5,LightGray,'Monsters killed : @<@1@>',[Kills.All]);
  if Player.Mode <> modeMassacre then
  Output.DrawString(42,6,LightGray,'Nights survived : @<@1@>',[Player.Night-1]);

  Output.DrawString(10,8 ,LightGray,'Strength : @<@1@>  Dexterity: @<@2@>',[Strength,Dexterity]);
  Output.DrawString(10,9 ,LightGray,'Endurance: @<@1@>  Willpower: @<@2@>',[Endurance,Willpower]);
  Output.DrawString(10,10,LightGray,'Weight   : @<@2@>  Speed    : @<@1@>' ,[Speed,Weight]);
  Output.DrawString(10,11,LightGray,'Base damage - @<@1@>d6+@<@2@>',[getDamageDice,getDamageMod+5]);
  
  Output.DrawString(42,8 ,LightGray,'Crossbow  : @<@1@>/@<@2@>  Cannon : @<@3@>/@<@4@>',[Crossbow,CrossMis,Cannon,CannonChg]);
  Output.DrawString(42,9 ,LightGray,'Knives    : @<@1@>     Bombs  : @<@2@>',[Knives,Bombs]);
  Output.DrawString(42,10,LightGray,'Fairydust : @<@1@>',[FairyDust]);

  Output.DrawString(8,13,LightGray,Padded('-- @<Skills@> ',34,'-'));
  Output.DrawString(40,13,LightGray,Padded('-- @<Achievements@> ',37,'-'));

  C2 := 15;
  for Count := 1 to MAXSKILLS do
    if Skills[Count] > 0 then
    begin
      Output.DrawString(10,C2,LightGray,'@1 (level @<@2@>)',[SkillData[Count].Name,Skills[Count]]);
      Inc(C2);
    end;

  Output.DrawString(42,15,LightGray,'Kills in 1 turn   : @<@1',[Kills.BestTurn]);
  Output.DrawString(42,16,LightGray,'Killing sequence  : @<@1',[Kills.BestSpree]);
  Output.DrawString(42,17,LightGray,'Sequence duration : @<@1t',[Kills.BestSpreeLength]);
  Output.DrawString(42,18,LightGray,'Survived for      : @<@1t',[TurnCount]);

  Output.DrawString(8,23,LightGray,Padded('-- Press <@<Enter@>> to exit... ',65+4,'-'));
  Input.GetKey([VKEY_ENTER,VKEY_ESCAPE]);
  UI.Clear;
end;
end;

class procedure TBerserkCharGen.Skill;
var Menu   : TTextMenu;
    Choice : Byte;
  function LevStr(Level : byte) : string;
  begin
    if Level = 0 then Exit('');
    Exit(' (@<'+IntToStr(Level+1)+'@>)');
  end;
begin
  UI.DrawFire;
  UI.Window(4,1,55,22);
  Output.DrawString(6,2,LightGray,'Choose a skill you want to upgrade. Most skills offer some otherwise unattainable option in the game. Further levels increase that option''s effectiveness.',50);

  UI.Window(58,1,77,MAXSKILLS+2);

  Menu := TTextMenu.Create(60,2,[VKEY_ENTER],@SkillDesc,True);
  for Choice := 1 to MAXSKILLS do
    with SkillData[Choice] do
      Menu.Add(Name+LevStr(Player.Skills[Choice]),Player.SkillReqsMet(Choice) and (Player.Skills[Choice] < MaxLevel));

  Choice := Menu.Run;
  
  Inc(Player.Skills[Choice]);

  FreeAndNil(Menu);
  UI.Clear;
end;

class function TBerserkCharGen.NewStats(Strength, Dexterity, Endurance,
  Willpower, Points : Word) : TChargenStats;
begin
  NewStats[1] := Strength;
  NewStats[2] := Dexterity;
  NewStats[3] := Endurance;
  NewStats[4] := Willpower;
  NewStats[5] := Points;
end;

class procedure TBerserkCharGen.SkillDesc(SkillID : byte);
var Lines : Byte;
begin
  Output.ClearRect (4, 7,55,22);
  Output.DrawString(6, 8,LightGray,'Name      : @<'+SkillData[SkillID].Name);
  Output.DrawString(6, 9,LightGray,'Max level : @<'+IntToStr(SkillData[SkillID].MaxLevel));
  Output.DrawString(6,10,LightGray,'Requires  : ');
  Output.DrawString(45,8,LightGray,SkillData[SkillID].Picture);

  Lines := 1;
  with SkillData[SkillID] do
    if Reqs[1][1] = 0 then
    begin
      Output.DrawString(18,10,Green,'nothing');
      Inc(Lines);
    end
    else
    repeat
      if Player.SkillReqMet(Reqs[Lines]) then Output.DrawString(18, 9+Lines,Green,ReqToString(Reqs[Lines]))
                                         else Output.DrawString(18, 9+Lines,Red,ReqToString(Reqs[Lines]));
      Inc(Lines);
    until (Lines = MAXREQS+1) or (Reqs[Lines][1] = 0);
  
  Output.DrawString(6,9+Lines+1,LightGray,SkillData[SkillID].Description,50);
end;


end.

