// @abstract(BerserkRL -- TPlayer class unit)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(Oct 16, 2006)
// @lastmod(Oct 22, 2006)
//
// This unit just holds the TPlayer class -- a class representing the Player in
// Berserk. It is a descendant of TBeing (brbeing.pas) and overrides several
// it's methods to allow proper usage.
//
//  @html <div class="license">
//  This file is part of BerserkRL.
//
//  BerserkRL is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  BerserkRL is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with BerserkRL; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//  @html </div>

unit brplayer;
{$mode objfpc}
interface
uses SysUtils, vrltools, brbeing, brdata, brui;

const
  // Target mode for ChooseTarget - look mode
  TM_LOOK  = 1;
  // Target mode for ChooseTarget - fire mode
  TM_FIRE  = 2;
  // Target mode for ChooseTarget - throw bomb
  TM_THROW = 3;

type

// The Berserk player object. Should be only one, and always referenced by the
// Player pointer. TPlayer is a descendant of TBeing, and overrides it's methods
// where needed.

{ TPlayer }

TPlayer = class(TBeing)
  // Amount of ready missiles in the cannon.
  Cannon    : Byte;
  // Amount of cannon charges
  CannonChg : Byte;
  // Amount of ready missiles in the crossbow.
  Crossbow  : Byte;
  // Missiles in the quiver for the crossbow.
  CrossMis  : Byte;
  // Amount of throwing Knives
  Knives    : Byte;
  // Amount of bombs
  Bombs     : Byte;

  // Berserk points
  BerserkPT   : Word;

  // Amount of Fairy dust
  FairyDust : Byte;

  // Bonus to defense. Applicable when moving, or waiting.
  DefBonus  : Byte;


  // Holds the ID value of the last attacker, used to check who killed the player.
  LastAttack   : Byte;
  // Count of turns passed. Based on actions of the player.
  TurnCount    : DWord;

  // The kills tables.
  Kills        : TKillTable;
  
  // The *S*kills table.
  Skills       : TSkills;
  
  // Unused character generation points.
  Points       : Word;
  
  // Holds the night number for Endless mode.
  Night     : Word;
  // Defines the game mode - Massacre, Nights or Campaign
  Mode      : TGameMode;

  // Creates a new player. It automaticaly sets the id and nid of the player
  // reference to the Player.
  constructor Create( pos : TCoord2D );
  // Handles character creation. Should be called after Create.
  procedure CreateCharacter;
  // Resets the player for a new day, and runs advancement.
  procedure Advance;
  // Writes a post mortem and shows it on the screen.
  procedure WriteMortem;
  // Overriden Die procedure. Contrary to TBeing.Die, the players Die procedure
  // DOES NOT free the player object!
  procedure Die; override;
  // Handle berserk time.
  procedure BerserkTick;
  // Overriden action procedure -- handles Input parsing, and acting accordingly.
  procedure Action; override;
  // Returns True.
  function IsPlayer : Boolean; override;
  // Interface function for choosing target. Returns true when in Fire mode
  // target is accepted.
  function ChooseTarget(TargetMode : Byte = TM_LOOK; FireCmd : byte = COMMAND_FIRE ) : boolean;
  // Returns damage dice based on Strength + Berserk!
  function getDamageDice : Byte; override;
  // Ampplies damage to the player.
  procedure ApplyDamage(Amount : Word; DType : Byte); override;
  // Returns the number of adjacent to the player
  function enemiesAround : byte;
  // Fires the appropriate missle
  procedure Fire(Command : Byte);
  // Returns wether the given skill requirement is met.
  function SkillReqMet(SkillReq : TSkillReq) : Boolean;
  // Returns wether the requirements for the given skill are met.
  function SkillReqsMet(SkillID : Byte) : Boolean;
  // Frees all player data.
  destructor Destroy; override;
  // Returns the current light radius based on the players status.
  function LightRadius : Byte;
  // Returns status of BF_RUNNING flag - wether the player is running
  function isRunning : boolean;
  // Returns status of BF_BERSERK flag - wether the player is in berserk mode
  function isBerserk : boolean;
end;

var Player : TPlayer = nil;


implementation
uses vutil, bruichar, brmain, brlevel, brhof, math;


{ TPlayer }

constructor TPlayer.Create( pos : TCoord2D );
begin
  inherited Create(1, 'yourself', pos);
  TurnCount := 0;

  LastAttack := 0;
  HPMax := 100;
  HP := HPMax;

  Strength  := 10;
  Dexterity := 10;
  Endurance := 10;
  Willpower := 10;

  Armor     := 2;
  
  Cannon    := 1;
  CannonChg := 2;
  
  Crossbow  := 12;
  Crossmis  := 60;
  Knives    := 10;
  FairyDust := 3;
  Bombs     := 3;
  
  DefBonus  := 0;
  
  BerserkPT    := 0;
  
  Points := 0;

  Kills.Clear;
end;

procedure TPlayer.CreateCharacter;
var Stats       : TChargenStats;
    ModeNum     : Byte;
begin
  if QuickStart then
  begin
    Mode := modeMassacre;
    Name := 'Epyon';
    Strength  := 16;
    Dexterity := 14;
    Strength  := 14;
    Dexterity := 12;
    Endurance := 10;
    Willpower := 10;
    Points    := 0;
    Berserk.Arena := ARENA_SNOW;
    Player.Skills[SKILL_IRONMAN] += 3;
    HPMax  := 115;
    ENMax  := 100;
    Speed := 95+Dexterity;
    Weight := Endurance;
    HP := HPMax;
    EN := ENMax;
    Exit;
  end;

  Mode := UI.CharGen.GameMode;


  Name := '';
  if not Option_AlwaysRandomName then
    Name  := UI.CharGen.Name;
  if Name = '' then
  case Random(8) of
    0..1 : Name := 'Guts';
    2    : Name := 'Glowie';
    3    : Name := 'Turgor';
    4    : Name := 'Fingerzam';
    5    : Name := 'Malek';
    6    : Name := 'Jorge';
    7    : Name := 'Thomas';
  end;

  if Mode = modeMassacre then
    Stats := UI.CharGen.Stats(UI.Chargen.NewStats(10,10,10,10,14),statsMassacre)
  else
    Stats := UI.CharGen.Stats(UI.Chargen.NewStats(10,10,10,10,8),statsFirst);

  Strength  := Stats[1];
  Dexterity := Stats[2];
  Endurance := Stats[3];
  Willpower := Stats[4];
  Points    := Stats[5];

  if Mode = modeMassacre then
  begin
    Berserk.Arena := UI.CharGen.Arena;
    UI.CharGen.Skill;
    UI.CharGen.Skill;
    UI.CharGen.Skill;
  end
  else
  begin
    UI.CharGen.Skill;
    Crossmis  := 36;
    Fairydust := 2;
    Bombs     := 2;
    Knives    := 5;
    CannonChg := 1;
  end;

  HPMax  := 100 + (Endurance-10)*5 + Skills[SKILL_IRONMAN] * 5;
  ENMax  := 100 + (Endurance-10)*5 + (Willpower-10)*5;
  Speed := 95+Dexterity;
  Weight := Endurance;


  HP := HPMax;
  EN := ENMax;
end;

procedure TPlayer.Advance;
var Sum   : byte;
    Stats : TChargenStats;

begin
  // Run advancement screens
  Stats := UI.CharGen.Stats(
    UI.Chargen.NewStats(Strength,Dexterity,Endurance,Willpower,Points+1),
    statsUpgrade);

  // Apply advancements
  Strength  := Stats[1];
  Dexterity := Stats[2];
  Endurance := Stats[3];
  Willpower := Stats[4];
  Points    := Stats[5];

  // Choose skill
  UI.CharGen.Skill;


  // Recalculate stats
  HPMax  := 100 + (Endurance-10)*5 + Skills[SKILL_IRONMAN] * 5;
  ENMax  := 100 + (Endurance-10)*5 + (Willpower-10) * 5;
  Speed  := 95+Dexterity;
  Weight := Endurance;

  // Update current stats.
  EN := ENMax;
  HP := Min(HPMax,HP+Endurance*5);

  // Free healing if Fairydust was max.
  if FairyDust = 5 then HP := HPMax;

  // Update equipment
  Sum := Crossbow + Crossmis;
  Sum := Min(Sum+36,12+60);

  Crossbow := 12;
  Crossmis := Sum-12;

  Sum := Cannon+CannonChg;
  Sum := Min(Sum+1,1+3);
  
  Cannon    := 1;
  CannonChg := Sum-1;
  
  Knives    := Min(Knives+3,10);
  FairyDust := Min(FairyDust+1,5);
  Bombs     := Min(Bombs+2,5);

  // Game stats reset
  tid := 0;
  TargetCoord := ZeroCoord2D;
  LastAttack  := 0;
  Flags       := [];
  BerserkPT   := 0;
  DefBonus    := 0;
  
  Pain := 0;
  Freeze := 0;
end;

procedure TPlayer.WriteMortem;
var Mortem : Text;
    Count  : byte;
    Count2 : byte;
function ChrAT(b : byte) : string;
begin
  if Chr(b) = '@' then Exit('@@') else Exit(chr(b));
end;

begin
  Assign(Mortem,'mortem.txt');
  Rewrite(Mortem);
  Flags := []; // reset berserk and running
  Writeln(Mortem,Padded('-- Berserk! ('+Version+') Post Mortem ',70,'-'));
  Writeln(Mortem,'');
  Writeln(Mortem,'  Character name  : '+Name);
  Writeln(Mortem,'  Game type       : '+ModeToString(Mode));
  if Mode <> modeMassacre then
  Writeln(Mortem,'  Nights survived : '+IntToStr(Night));
  Writeln(Mortem,'  Monsters killed : '+IntToStr(Kills.All));

  Writeln(Mortem,'');
  Writeln(Mortem,'  STR: ',Strength, '  DEX: ',Dexterity,'  END: ',Endurance,'  WIL: ',Willpower);
  Writeln(Mortem,'  Speed: ',Speed,'  HP: ',HP,'/',HPMax,'  EN: ',EN,'/',ENMax);
  Writeln(Mortem,'  Base damage: ',getDamageDice,'d6+',getDamageMod+5,'  Weight: ',Weight);
  Writeln(Mortem,'');
  Writeln(Mortem,Padded('-- Graveyard ',70,'-'));
  Writeln(Mortem,'');
  for Count2 := 1 to MAP_MAXY do
  begin
    Write(Mortem,'  ');
    for Count := 1 to MAP_MAXX do
        if Level.Being[ NewCoord2D(Count,Count2) ] <> nil then Write(Mortem,ChrAT(Level.Being[ NewCoord2D(Count,Count2) ].Picture mod 256))
                                                          else Write(Mortem,Chr(Level.Terrain[ NewCoord2D(Count,Count2) ].Picture mod 256));

    Writeln(Mortem,'');
  end;
    Writeln(Mortem,'');
  Writeln(Mortem,Padded('-- Weapons left ',70,'-'));
  Writeln(Mortem,'');
  if (Cannon <> 0) or (CannonChg <> 0) then Write(Mortem,'  Cannon (',Cannon,'/',CannonChg,')');
  if (Crossbow <> 0) or (CrossMis <> 0)  then Write(Mortem,'  Crossbow (',Crossbow,'/',CrossMis,')');
  if (Knives <> 0) then Write(Mortem,'  Knives (',Knives,')');
  if (Bombs <> 0)  then Write(Mortem,'  Bombs (',Bombs,')');
  if (FairyDust <> 0)  then Write(Mortem,'  FairyDust (',FairyDust,')');
  if (Knives+Bombs+FairyDust+Cannon+CannonChg+Crossbow+CrossMis > 0) then Writeln(Mortem,'');
  Writeln(Mortem,'');
  Writeln(Mortem,Padded('-- Kills ('+IntToStr(Kills.All)+') ',70,'-'));
  Writeln(Mortem,'');
  for Count := 2 to 100 do
    if Kills.Get[Count] <> 0 then
      if Kills.Get[Count] = 1 then Writeln(Mortem,'  1 ',Berserk.Lua.IndexedTable['beings',Count,'name'])
                              else Writeln(Mortem,'  ',Kills.Get[Count],' ',Berserk.Lua.IndexedTable['beings',Count,'namep']);
  Writeln(Mortem,'');
  Count2 := 0;
  for Count := 1 to MAXSKILLS do
    Count2 += Skills[Count];
  Writeln(Mortem,Padded('-- Skills ('+IntToStr(Count2)+') ',70,'-'));
  Writeln(Mortem,'');
  for Count := 1 to MAXSKILLS do
    if Skills[Count] > 0 then
      Writeln(Mortem,'  ',SkillData[Count].Name,' (level ',Skills[Count],')');
  Writeln(Mortem,'');
  Writeln(Mortem,Padded('-- Messages ',70,'-'));
  Writeln(Mortem,'');
  UI.MsgDump(Mortem);
  Writeln(Mortem,'');
  Writeln(Mortem,Padded('-- Achievements ',70,'-'));
  Writeln(Mortem,'');
  Writeln(Mortem,'  Max kills in one turn    : ',Kills.BestTurn);
  Writeln(Mortem,'  Longest killing sequence : ',Kills.BestSpree,' kills in ',Kills.BestSpreeLength,' turns.');
  Writeln(Mortem,'  Survived for             : ',TurnCount,' turns');
  Writeln(Mortem,'  Died on                  : ',ArenaToString(Level.Arena));
  Write  (Mortem,'  Reason of death          : ');
  if LastAttack = 1 then Writeln(Mortem,'suicide')
    else if (LastAttack > 1) and (LastAttack < 100) then Writeln(Mortem,'killed by a ',Berserk.Lua.IndexedTable['beings',LastAttack,'name'])
    else Writeln(Mortem,'unknown');
  Writeln(Mortem,'');
  Close(Mortem);
  UI.Text.Mortem;
end;

procedure TPlayer.Die;
var Count : Word;
begin
  Kills.Update(TurnCount);
  Kills.Update(TurnCount);
  UI.Blink(Red,200);
  UI.Msg('You die!...');
  UI.Msg('Press <@<Enter@>>');
  UI.Draw;
  UI.PressEnter;
  Berserk.Escape := True;
  UI.Screen := Menu;
  if UI.Playback then Exit;
  HOF.Add(Mode,Name,Kills.All,TurnCount,LastAttack,Level.Arena,Night);
  WriteMortem;
end;

procedure TPlayer.BerserkTick;
begin
  if isBerserk then
  begin
    if Kills.ThisTurn = 0 then
    begin
      Inc(BerserkPT);
      if BerserkPT = HPMax then
      begin
        if HP <= Willpower then Dec(BerserkPT)
        else
        begin
          UI.Msg('You calm down.');
          Exclude(Flags,BF_BERSERK);
          BerserkPT   := 0;
        end;
      end;
    end;
  end
  else
  begin
    if (BerserkPT < HPMax div 2) and (((Kills.ThisTurn > 0) and (enemiesAround > 0)) or (TurnCount mod Max(1,(25-Willpower)) = 0)) then Inc(BerserkPT);
    if BerserkPT > HP then
    begin
      UI.Blink(Red,150);
      UI.Delay(150);
      UI.Blink(Red,200);
      UI.Msg('You go BERSERK!');
      Include(Flags,BF_BERSERK);
    end;
  end;
  
  if isBerserk then
  begin
    DefBonus := enemiesAround div 2;
    Pain := 0;
    Freeze := 0;
  end;
end;

procedure TPlayer.Action;
var Command    : Byte;
    MoveCoord  : TCoord2D;
    MoveResult : Word;
    Direction  : TDirection;
    Amount     : Word;
    dx,dy      : Word;
    dxv,dyv    : ShortInt;
    AFlags     : TFlags;
    Cost       : Word;
    Slip       : Boolean;
begin
  Inc(TurnCount);
  DefBonus  := 0;

  BerserkTick;
  Kills.Update(TurnCount);

repeat
  if EN < 10 then Exclude(Flags,BF_RUNNING);
  UI.Draw;
  AttackCount := 0;
  
  Slip := False;
  
  if ( TF_ICE in Level.getFlags( Position ) ) and (Dice(3,6) > Dexterity) then
    Slip := True;

  if ( TF_FREEZE in Level.getFlags( Position ) ) then begin ApplyDamage(4,DAMAGE_FREEZE); UI.Msg('Freezing!'); end;

  if not Slip then
  begin
    Command := UI.GetCommand;
    if Command = 0 then UI.Msg('Press @<'+UI.GetKeybinding(COMMAND_HELP)+'@> for help.');
  end;
  
  if (Command in COMMANDS_MOVE) or Slip then
  begin
    if Slip then
    begin
      Direction.Create( Random(9)+1 );
      UI.Msg('You slip!');
      SpeedCount -= 500;
    end
    else Direction := UI.CommandDirection(Command);
    MoveCoord := Position + Direction;
    MoveResult := TryMove( MoveCoord );
    if MoveResult <= 1 then
    begin
      Move( MoveCoord );
      DefBonus  := Max(1,DefBonus);
      if isRunning then
      begin
        if Skills[SKILL_RUNNING] = 0 then
        begin
          Dec(EN,10);
          Cost := 750;
        end else begin
          Dec(EN,7-Skills[SKILL_RUNNING]);
          Cost := 700-50*Skills[SKILL_RUNNING];
        end;
        DefBonus  := Max(2,DefBonus);
      end else Cost := 1000;
      SpeedCount -= Round((Cost / 200)*(Level.getMoveCost( Position )+Level.getMoveCost( MoveCoord )));
    end
    else
    if (MoveResult <= MAXBEINGS) then
    begin
      if (Level.Being[ MoveCoord ].AI <> AI_MONSTER) and (not isBerserk) then UI.Msg('Let them live... a little longer.')
      else
      begin
        Attack(Level.Being[ MoveCoord ]);
        SpeedCount -= 1000;
      end;
    end;
     // TEMPORARY
  end;
  
  case Command of
    COMMAND_WAIT : begin SpeedCount -= 1000; DefBonus := Max(3,DefBonus); end;
    COMMAND_LOOK : begin
        UI.Msg('Look mode, @<ESC@> to exit.');
        UI.Msg('You see:');
        ChooseTarget;
        UI.MsgKill;
        UI.MsgKill;
      end;
      
    COMMAND_KNIFE : Fire(COMMAND_KNIFE);
    COMMAND_BOMB  : Fire(COMMAND_BOMB);
    COMMAND_FIRE  : Fire(COMMAND_FIRE);
    COMMAND_RUNNING :
      //if Skills[SKILL_RUNNING] = 0 then UI.Msg('You don''t have the Running skill!') else
      if EN < 10                   then UI.Msg('You''re to exhausted to run!') else
      if isRunning  then Exclude(Flags,BF_RUNNING)
                    else Include(Flags,BF_RUNNING);
                  
    COMMAND_CRRELOAD :
      if isBerserk          then UI.Msg('No time, need to KILL!') else
      if Crossbow      = 12 then UI.Msg('Magazine already full!') else
      if CrossMis      = 0  then UI.Msg('You have no more bolts!') else
      if enemiesAround > 0  then UI.Msg('Can''t reload in combat!') else
      begin
        Amount := Min(Min(12,CrossMis),12-Crossbow);
        UI.Msg('You reload the crossbow.');
        Crossbow += Amount;
        CrossMis -= Amount;
        SpeedCount -= 2000;
      end;
      
    COMMAND_HELP      : UI.Text.Help;
    COMMAND_PLAYERINFO: UI.Chargen.Screen;
    COMMAND_MESSAGES  : UI.MsgPast;

    COMMAND_QUIT      : begin
        Berserk.Escape := True;
        SpeedCount -= 1000;
      end;
      
    COMMAND_FAIRYDUST :
      if FairyDust = 0 then UI.Msg('You have no more fairydust!')
      else
      begin
        UI.Msg('You use the fairydust...');
        if HP = HPMax then UI.Msg('Nothing happens.')
        else
        begin
          HP := HPMax;
          EN := ENMax;
          UI.Blink(LightGreen);
          UI.Msg('You feel a lot better!');
          Pain := 0;
          Freeze := 0;
          if isBerserk then
          begin
            UI.Msg('You calm down.');
            Exclude(Flags,BF_BERSERK);
            BerserkPT := 0;
          end;
        end;
        Dec(Fairydust);
        SpeedCount -= 1000;
      end;
      
    COMMAND_SKILLSWEEP :
      if Skills[SKILL_SWEEP] = 0       then UI.Msg('You don''t have the Sweep Attack skill!') else
      if EN < 25-5*Skills[SKILL_SWEEP] then UI.Msg('You''re to exhausted do that!') else
      begin
        UI.Msg('Sweep: Choose direction...');
        Direction := UI.ChooseDirection;
        UI.MsgKill;
        if Direction.Code <> 0 then
        begin
          Dec(EN,25-5*Skills[SKILL_SWEEP]);
          dxv := Direction.x;
          dyv := Direction.y;
          if dxv*dyv = 0 then
            if dxv = 0 then for dx := Position.x-1 to Position.x+1 do Attack( NewCoord2D( dx,Position.y+dyv ) ,[AF_SWEEP])
              {dyv = 0}else for dy := Position.y-1 to Position.y+1 do Attack( NewCoord2D( Position.x+dxv,dy ) ,[AF_SWEEP])
          else
            for dx := Position.x+Min(0,dxv) to Position.x+Max(0,dxv) do
              for dy := Position.y+Min(0,dyv) to Position.y+Max(0,dyv) do
                if not ((dx = Position.x) and (dy = Position.y)) then
                   Attack( NewCoord2D(dx,dy) ,[AF_SWEEP]);
          SpeedCount -= 1600-Skills[SKILL_SWEEP]*200;
        end;
      end;
      
    COMMAND_SKILLWHIRL :
      if Skills[SKILL_WHIRLWIND] = 0         then UI.Msg('You don''t have the Whirlwind Attack skill!') else
      if EN < 60-10*Skills[SKILL_WHIRLWIND] then UI.Msg('You''re to exhausted do that!') else
      begin
        Dec(EN,60-10*Skills[SKILL_WHIRLWIND]);
        for dx := Position.x-1 to Position.x+1 do
          for dy := Position.y-1 to Position.y+1 do
            if not ((dx = Position.x) and (dy = Position.y)) then
              Attack(NewCoord2D(dx,dy),[AF_SWEEP]);
        SpeedCount -= 2250-Skills[SKILL_WHIRLWIND]*250;
      end;
      
    COMMAND_SKILLIMPALE:
      if Skills[SKILL_IMPALE] = 0        then UI.Msg('You don''t have the Impale skill!') else
      if EN < 50-10*Skills[SKILL_IMPALE] then UI.Msg('You''re to exhausted do that!') else
      begin
        UI.Msg('Impale: Choose direction...');
        Direction := UI.ChooseDirection;
        UI.MsgKill;
        if Direction.Code <> 0 then
        begin
          MoveCoord := Position + Direction;
          if TryMove( MoveCoord ) = SCAN_OK then
          begin
            Dec(EN,50-10*Skills[SKILL_IMPALE]);
            Move( MoveCoord );
            DefBonus := Max(1,DefBonus);
            Attack(Position+Direction,[AF_IMPALE]);
            SpeedCount -= 1600-Skills[SKILL_IMPALE]*200;
          end else UI.Msg('No space to do a Impale attack!');
        end;
      end;

    COMMAND_SKILLJUMP:
      if Skills[SKILL_JUMP] = 0        then UI.Msg('You don''t have the Jump skill!') else
      if EN < 50-10*Skills[SKILL_JUMP] then UI.Msg('You''re to exhausted do that!') else
      begin
        UI.Msg('Jump: Choose direction...');
        Direction := UI.ChooseDirection;
        UI.MsgKill;
        if Direction.Code <> 0 then
        begin
          MoveCoord := Position + Direction;
          if (TryMove( MoveCoord ) <= MAXBEINGS)
          and (TryMove( MoveCoord + Direction ) = SCAN_OK) then
          begin
            Dec(EN,50-10*Skills[SKILL_JUMP]);
            Attack(Position+Direction,[AF_NOKNOCKBACK]);
            Move( MoveCoord + Direction );
            DefBonus := Max(1,DefBonus);
            SpeedCount -= 1600-Skills[SKILL_JUMP]*200;
          end else UI.Msg('No space to do a Jump attack!');
        end;
      end;

    COMMAND_CARELOAD:
      if isBerserk         then UI.Msg('No time, need to KILL!') else
      if Cannon        = 1 then UI.Msg('Cannon already loaded!') else
      if CannonChg     = 0 then UI.Msg('You have no more charges!') else
      if enemiesAround > 0 then UI.Msg('Can''t reload in combat!') else
      begin
        UI.Msg('You reload the cannon.');
        Cannon     += 1;
        CannonChg  -= 1;
        SpeedCount -= 3000;
      end;

    COMMAND_CANNON:
      if Cannon = 0 then UI.Msg('Cannon not loaded!') else
      begin
        UI.Msg('Choose direction...');
        Direction := UI.ChooseDirection;
        UI.MsgKill;
        if Direction.Code <> 0 then
        begin
          UI.Blink(Red,200);
          UI.Blink(Yellow,50);
          Level.Breath(Position,Direction,Red,18,9,30);
          Knockback(Direction.Reversed,20);
          Dec(Cannon);
          SpeedCount -= 1000;
        end;
      end;
  end;
until SpeedCount <= SPEEDLIMIT;
end;


function TPlayer.IsPlayer : Boolean;
begin
  Exit(True);
end;

function TPlayer.ChooseTarget( TargetMode : Byte = TM_LOOK; FireCmd : byte = COMMAND_FIRE ) : boolean;
var Dir     : TDirection;
    Key     : Byte;
    Targets : TAutoTarget;
    Target  : TCoord2D;
    scx,scy : Integer;
    AISet   : TAITypeSet;
begin
  Targets := TAutoTarget.Create( Position );
  
  if isBerserk then AISet := [AI_MONSTER,AI_CIVILIAN,AI_ALLY]
               else AISet := [AI_MONSTER];
               
  for scx := Max(1,Position.x-LightRadius-1) to Min(MAP_MAXX,Position.x+LightRadius+1) do
    for scy := Max(1,Position.y-LightRadius-1) to Min(MAP_MAXY,Position.y+LightRadius+1) do
    begin
      Target := NewCoord2D( scx,scy );
      if Level.Being[ Target ] <> nil then
        if Level.Being[ Target ].AI in AISet then
         if isInLOS( Target ) then
           Targets.AddTarget( Target );
    end;

  UI.Msg('');

  if TargetMode in [TM_FIRE,TM_THROW] then
    if tid <> 0 then
      if Level.Beings[tid] <> nil then
        with Level.Beings[tid] do
          Targets.PriorityTarget( Position );

  Target := Targets.Current;
  
  repeat
    UI.DrawLevel;
    if TargetMode = TM_FIRE then
      UI.Target(Target,RED);
    if TargetMode = TM_THROW then
    begin
      if Distance(Position,Target) > BOMBDISTANCE
        then UI.Target(Target,DarkGray)
        else UI.Target(Target,Red);
    end;
    UI.Focus(Target);
    UI.MsgCoord(Target);

    if TargetMode  in [TM_FIRE,TM_THROW] then
      Key := UI.GetCommand(COMMANDS_MOVE+[COMMAND_ESCAPE,COMMAND_RUNNING,FireCmd])
    else
      Key := UI.GetCommand(COMMANDS_MOVE+[COMMAND_ESCAPE,COMMAND_RUNNING]);
    if Key = COMMAND_ESCAPE   then begin Target.x := 0; Break; end;
    if Key = COMMAND_RUNNING  then Target := Targets.Next;
    if (Key in COMMANDS_MOVE) then
    begin
      Dir := UI.CommandDirection(Key);
      if Level.properCoord(Target + Dir) then
        Target += Dir;
    end;
  until Key in [FireCmd];

  TargetCoord := Target;

  UI.MsgKill;
  FreeAndNil(Targets);

  if TargetCoord.x = 0 then Exit(False);
  if Level.Being[ TargetCoord ] <> nil then tid := Level.Being[ TargetCoord ].nid;
  Exit(True);
end;


function TPlayer.getDamageDice : Byte;
begin
  if isBerserk then Exit(2* inherited getDamageDice)
               else Exit(inherited getDamageDice);
end;

procedure TPlayer.ApplyDamage(Amount : Word; DType : Byte);
begin
  if isBerserk then Amount := Amount div 2;
  inherited ApplyDamage(Amount,DType);
end;

function TPlayer.enemiesAround : byte;
var sx,sy : Byte;
begin
  enemiesAround := 0;
  for sx := Position.x-1 to Position.x+1 do
    for sy := Position.y-1 to Position.y+1 do
      if Level.Being[ NewCoord2D( sx,sy ) ] <> nil then Inc(enemiesAround);
  Dec( enemiesAround ); // Player
end;

procedure TPlayer.Fire(Command : Byte);
var TargetType : Byte;
begin

  case Command of
    COMMAND_KNIFE : if Knives   = 0 then begin UI.Msg('You have no more knives!'); exit end else TargetType := TM_FIRE;
    COMMAND_FIRE  : if Crossbow = 0 then begin UI.Msg('Magazine empty!'); exit; end else TargetType := TM_FIRE;
    COMMAND_BOMB  : if Bombs    = 0 then begin UI.Msg('You have no more bombs!'); exit; end else TargetType := TM_THROW;
  end;

  if Command = COMMAND_FIRE then
  begin
    if enemiesAround > 0 then begin UI.Msg('You can''t fire in combat!'); exit; end;
    UI.Msg('Choose target, @<'+UI.GetKeybinding(Command)+'@> to fire:');
  end
  else UI.Msg('Choose target, @<'+UI.GetKeybinding(Command)+'@> to throw:');
  
  if ChooseTarget(TargetType,Command) then
  begin
    if (Command = COMMAND_BOMB) and (Distance( TargetCoord,Position) > BOMBDISTANCE)
      then begin UI.Msg('That''s too far!'); exit end;
    UI.MsgKill;
    if TargetCoord = Position then begin  UI.Msg('Suicide? Too easy!'); Exit; end;
    case Command of
      COMMAND_KNIFE: begin
          Dec(Knives);
          SendMissile(TargetCoord,MTKNIFE);
        end;
      COMMAND_FIRE : begin
          Dec(Crossbow,3);
          SendMissile(TargetCoord,MTBOLT);
          SendMissile(TargetCoord,MTBOLT);
          SendMissile(TargetCoord,MTBOLT);
        end;
      COMMAND_BOMB : begin
          Dec(Bombs);
          SendMissile(TargetCoord,MTBOMB);
        end;
    end;
    SpeedCount -= 1000; // TEMPORARY
  end
  else
    UI.MsgKill;
end;

function TPlayer.SkillReqMet(SkillReq : TSkillReq) : Boolean;
begin
  SkillReqMet := True;
  case SkillReq[1] of
    SKILLREQ_STAT :
      case SkillReq[2] of
        STAT_STR : Exit(Strength  >= SkillReq[3]);
        STAT_DEX : Exit(Dexterity >= SkillReq[3]);
        STAT_WIL : Exit(Willpower >= SkillReq[3]);
        STAT_END : Exit(Endurance >= SkillReq[3]);
      end;
    SKILLREQ_SKILL : Exit(Skills[SkillReq[2]] >= SkillReq[3]);
  else Exit(True);
  end;
end;

function TPlayer.SkillReqsMet(SkillID : Byte) : Boolean;
var Count : byte;
begin
  SkillReqsMet := True;
  for Count := 1 to MAXREQS do
    if not SkillReqMet(SkillData[SkillID].Reqs[Count]) then Exit(False);
end;

function TPlayer.LightRadius : Byte;
begin
  if HP < 1 then Exit(1);
  if HP < (HPMax div 5) then
    LightRadius := 1 + Round((HP/HPMax)*45)
  else Exit(10);
end;

function TPlayer.isRunning : boolean;
begin
  Exit(BF_RUNNING in Flags);
end;

function TPlayer.isBerserk : boolean;
begin
  Exit(BF_BERSERK in Flags);
end;

destructor TPlayer.Destroy;
begin
  inherited Destroy;
  Player := nil;
end;

end.

