// @abstract(BerserkRL -- TLevel class unit)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(Oct 16, 2006)
// @lastmod(Oct 22, 2006)
//
// This unit holds the TLevel class -- the singleton class to represent
// the Berserk levels. It also holds additional data structures used by TLevel.
//
//  @html <div class="license">
//  This file is part of BerserkRL.
//
//  BerserkRL is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  BerserkRL is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with BerserkRL; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//  @html </div>

unit brlevel;
{$mode objfpc}
interface
uses SysUtils, vnode, vmath, brdata, brbeing, vutil, brlevgen,vvision, vrltools;

type
  // Data record on a single Map Cell on the map.
  TMapCell = record
               // Information about beings on this cell. It's an index to the
               // TLevel.Beings array.
               Being    : Word;
               // Information on terrain. Index to the TerraData array.
               Terrain  : Word;
               // Information on the original terrain. Used only in graphics mode for overlays.
               TerrainB : Word;
               // Unused as yet.
               Feature  : Word;
               // How much damage the cell has sustained.
               Damage   : Word;
             end;
  // The map array, used by TLevel
  TMap     = array[1..MAP_MAXX,1..MAP_MAXY] of TMapCell;
  
  // The beings array, used by TLevel
  TBeings  = array[1..MAXBEINGS] of TBeing;

  // Level class. Berserk doesn't take advantage of Valkyries Node and Event
  // system, because of simplicity issues. That's why TLevel inherits only
  // TVObject. Also, that means that TLevel is a singleton, accessible from
  // the whole program. Creating and Destroying TLevel is a responsibility
  // of TBerserk.
  TLevel = class(TVObject,IVisionQuery)
    // Our level map object. Instead of holding pointers to TBeing's it holds
    // indexes to the Beings array. Cleaned by the method TLevel.Clear.
    Map    : TMap;
    // Holds a fixed array of TBeing objects. It's the only place where TBeing
    // pointers are stored. Remember that each TBeing object MUST have a correct
    // nid number, that references to the index in this array! Beings[1] is
    // always the Player.
    Beings : TBeings;
    // Describes the level of spawning -- the higher, the more often and worse
    // monsters appear.
    SpawnLevel : Word;
    // The game mode of the level.
    Mode       : TGameMode;
    // The type of the level.
    Arena      : Byte;
    // Applicable only to Campaign and Endless mode -- states wether the level
    // has been completed.
    Cleared    : Boolean;
    // Applicable only to Campaign and Endless mode -- states wether the level
    // should spawn monsters.
    NoSpawn    : Boolean;
    // Count of ticks passed. Based on each speed incrementation.
    TickCount : DWord;
    // Level generator - also handles "TICKS".
    Generator : TLevelGenerator;
    // Number of beings left alive. If all enemies are killed, then this will
    // equal to 1!
    BeingsLeft : Word;
    // Currently only used in the graphical version. Stores a random value for
    // each level.
    Seed : Word;
    // Used to decide on the base sprite for overlays
    SpriteBase : Word;
    // Holds the Valkyrie LOS algorithm
    Vision : TVision;
    // Area holding the level boundaries. Do not change!
    Area : TArea;
    // Creates the TLevel object. A TLevel should be created once -- to enter
    // another Level use Clear and fill the map anew. Remember to remove the
    // Player from the Beings list tough!
    constructor Create;
    // Generates the level using vdungen.
    procedure Generate(nArena : byte; nMode : TGameMode; nSpawnLevel : Word);
    // Clears the map and being array and Destroys all Beings. To preserve the
    // Player, you need to remove him from the Beings array.
    procedure Clear;
    // A single level Tick. Calls Tick on each being in the Beings array. Also
    // handles Respawn, and all Level time scheduled operations.
    procedure Tick;
    // Handeles bleeding -- amount is the information for greater
    // recursive blood flow.
    procedure Bleed(const Coord : TCoord2D; amount : byte = 1);
    // function needed for the vvision interface
    function blocksVision( const Coord : TCoord2D ) : boolean;
    // Spawns 'amount' beings of the id = 'id' on random places of the map border.
    // Returns last monster placed.
    function Spawn(const id : AnsiString;  amount : byte = 1) : TBeing;
    // Spawns 'amount' beings of the id = 'id' on random places over the map
    function SpawnOnMap(const id : AnsiString; amount : byte = 1) : TBeing;
    // Creates and places the being at the given coordinate. Demonlevel is optional
    // and used only if the summoned creature is a demon.
    function Summon(const id : AnsiString; Coord : TCoord2D; demonlevel : byte = 0) : TBeing;
    // Damages the tile
    procedure DamageTile( Coord : TCoord2D; damage : Word);
    // Renders and calculates an explosion.
    procedure Explosion( Where : TCoord2D; Color : byte; Range : byte; Strength : byte; DrawDelay : Word = 50; DamageType : Byte = DAMAGE_FIRE);
    // Renders and calculates a breath weapon/cannon shot.
    procedure Breath( Where : TCoord2D; Direction : TDirection; Color : byte; Range : byte; Strength : byte; DrawDelay : Word = 50);
    // Disposes of all alocated structures, and the level itself. If you want to
    // keep the player, remove it from the Beings array beforehand.
    destructor Destroy; override;
    private
    // Searches the Beings array for a free index, and returns it. Returns
    // zero if the array is full.
    function freeBeingSlot : Word;
    // Stores TerrainB values.
    procedure StoreTerrain;
    // Procedure setting cell at given coord.
    procedure setCell( Coord : TCoord2D;  Value : Byte );
    // Function returning cell at given coord.
    function getCell( Coord : TCoord2D ) : Byte;
    // Function returning being at given coord.
    function getTerrain( Coord : TCoord2D ) : TTerrainData;
    // Function returning the T at given coord.
    function getBeing( Coord : TCoord2D ) : TBeing;
    public
    // Returns true if a coord is in the map bounds (1..MAP_MAXX,1..MAP_MAXY),
    // returns false otherwise.
    function properCoord( Coord : TCoord2D ) : boolean;
    // Alias for TerraData[Map[x,y].Terrain].Flags
    function getFlags( Coord : TCoord2D ) : TFlags;
    // Alias for TerraData[Map[x,y].Terrain].MoveCost
    function getMoveCost( Coord : TCoord2D ) : Byte;
    // Returns wether there is an unobstructed line of sight between the
    // points (ax,ay) and (bx,by).
    function isEyeContact( Coord1, Coord2 : TCoord2D ) : boolean;
    // Property for Cell ID access.
    property Cell     [ Index : TCoord2D ] : Byte read getCell write setCell; default;
    // Property for Cell terrain information access.
    property Terrain  [ Index : TCoord2D ] : TTerrainData read getTerrain;
    // Property for Being access.
    property Being    [ Index : TCoord2D ] : TBeing read getBeing;
  end;

  // The Level singleton, for access by other units of the program. Initialized
  // and disposed of by the TBerserk singleton.
  var Level : TLevel = nil;
  

implementation

uses brmain, brui, brplayer;

{ TLevel }


constructor TLevel.Create;
begin
  Area.Create( NewCoord2D( 1, 1 ), NewCoord2D( MAP_MAXX, MAP_MAXY ) );
  Clear;
  SpawnLevel := 1;
  NoSpawn    := False;
  Mode       := modeMassacre;
  Arena      := 1;
  Cleared    := False;
  TickCount := 0;
  Vision := TIsaacVision.Create(Self,12);
  Generator := TLevelGenerator.Create;
  Seed := Random($FFFF);
end;

procedure TLevel.Generate(nArena : byte; nMode : TGameMode; nSpawnLevel : Word);
var PlayerPosition : TCoord2D;
begin
  Arena      := nArena;
  SpawnLevel := nSpawnLevel;
  Mode       := nMode;
  
  if Mode = modeEndless then Arena := Random(6)+1;


  Generator.Run;
  
  StoreTerrain;

  Beings[1] := Player;
  repeat
    PlayerPosition.Create( Random(20) + 15, Random(10) + 5 );
  until Player.TryMove( PlayerPosition ) = 0;
  Player.Move( PlayerPosition );
  if Arena = ARENA_TOWN then Spawn('townsman',10);
  Spawn('beast',4);
end;

procedure TLevel.StoreTerrain;
var c : TCoord2D;
begin
  for c in Area do
    with Map[c.X,c.Y] do
      TerrainB := Terrain;
end;

procedure TLevel.setCell(Coord: TCoord2D; Value: Byte);
begin
  Map[ Coord.X, Coord.Y ].Terrain := Value;
end;

function TLevel.getCell(Coord: TCoord2D): Byte;
begin
  Exit( Map[ Coord.X, Coord.Y ].Terrain );
end;

function TLevel.getTerrain(Coord: TCoord2D): TTerrainData;
begin
  Exit( TerraData[ Map[ Coord.X, Coord.Y ].Terrain ] );
end;

function TLevel.getBeing(Coord: TCoord2D): TBeing;
begin
  if Map[ Coord.X, Coord.Y ].Being <> 0
    then Exit( Beings[ Map[ Coord.X, Coord.Y ].Being ] )
    else Exit( nil );
end;

function TLevel.properCoord(Coord: TCoord2D): boolean;
begin
  Exit( Area.Contains( Coord ) );
end;

function TLevel.getFlags(Coord: TCoord2D): TFlags;
begin
  Exit( TerraData[ Map[ Coord.X, Coord.Y ].Terrain ].Flags );
end;

function TLevel.getMoveCost(Coord: TCoord2D): Byte;
begin
  Exit( TerraData[ Map[ Coord.x, Coord.y ].Terrain ].MoveCost);
end;

function TLevel.isEyeContact( Coord1, Coord2: TCoord2D ): boolean;
var EyeRay : TBresenhamRay;
    cnt    : byte;
begin
  if Coord1 = Coord2 then Exit( true );
  EyeRay.Init( Coord1, Coord2 );
  cnt := 0;
  repeat
    Inc(cnt);
    EyeRay.Next;
    if EyeRay.Done then Exit( True );
    if not properCoord(EyeRay.GetC) then Exit(False);
    if TF_NOSIGHT in getFlags(EyeRay.GetC) then Exit(False);
  until cnt = 15;
  Exit(False);
end;


procedure TLevel.Clear;
var Count : Word;
    Coord : TCoord2D;
begin
  for Count := 2 to MAXBEINGS do
    FreeAndNil( Beings[Count] );
  Beings[1] := nil;
  for Coord in Area do
  with Map[Coord.X,Coord.Y] do
  begin
    Being   := 0;
    Terrain := 1;
    TerrainB:= 1;
    Feature := 0;
    Damage  := 0;
  end;
  Cleared    := False;
  TickCount  := 0;
  NoSpawn    := False;
end;

procedure TLevel.Tick;
var Count   : Word;
begin
  BeingsLeft := 0;
  for Count := 1 to MAXBEINGS do
  begin
    if Beings[Count] <> nil then Beings[Count].Tick;
    if Beings[Count] <> nil then Inc(BeingsLeft);
    if Berserk.Escape then Exit;
  end;
  Inc(TickCount);
  
  Generator.Tick;
end;


procedure TLevel.Bleed( const Coord : TCoord2D; amount : byte = 1 );
var Shift  : TCoord2D;
    Count  : Byte;
begin
  with Map[Coord.x,Coord.y] do
  begin
    if TerraData[Terrain].BloodID = 0 then Exit;
    Terrain := TerraData[Terrain].BloodID;
    if TF_NOMOVE in TerraData[Terrain].Flags then Exit;
    if Amount > 1 then
    for Count := 1 to Amount do
    begin
      Shift.x := Random(3)-1;
      Shift.y := Random(3)-1;
      if properCoord( Coord + Shift ) then
        Bleed( Coord + Shift , Amount-1 );
    end;
  end;
end;

function TLevel.blocksVision( const Coord : TCoord2D ): boolean;
begin
  if not properCoord( Coord ) then Exit(true);
  if TF_NOSIGHT in getFlags( Coord ) then Exit(true);
  Exit(false);
end;

function TLevel.Spawn(const id : AnsiString;  amount : byte = 1 ) : TBeing;
var Coord : TCoord2D;
begin
  Spawn := nil;
  repeat
    Coord := Area.RandomEdgeCoord;
    if ( Being[ Coord ] <> nil ) or ( TF_NOMOVE in getFlags( Coord ) ) then Continue;
    Spawn := Summon(id,Coord,amount);
    Dec(amount);
    if id = 'demon' then Break;
  until amount = 0;
end;

function TLevel.SpawnOnMap(const id : AnsiString;  amount: byte): TBeing;
var Coord : TCoord2D;
begin
  SpawnOnMap := nil;
  repeat
    Coord := Area.Shrinked.RandomCoord;
    if ( Being[ Coord ] <> nil ) or ( TF_NOMOVE in getFlags( Coord ) ) then Continue;
    SpawnOnMap := Summon(id,Coord,amount);
    Dec(amount);
    if id = 'demon' then Break;
  until amount = 0;
end;


function TLevel.Summon(const id : AnsiString; Coord : TCoord2D; demonlevel : byte = 0) : TBeing;
var slot : byte;
    c    : TCoord2D;
begin
  if Being[ Coord ] <> nil then
  begin
    if TF_NOMOVE in getFlags( Coord ) then Exit(nil);
    for c in NewArea( Coord, 1 ) do
      if Map[c.x,c.y].Being = 0 then
        Break;
    if Map[c.x,c.y].Being > 0 then Exit(nil);
    Coord := c;
  end;
  slot := freeBeingSlot;
  if slot = 0 then Exit(nil);
  Beings[slot] := TBeing.Create(slot,id,Coord,demonlevel);
  Exit(Beings[slot]);
end;

procedure TLevel.DamageTile( Coord : TCoord2D;  damage : Word);
begin
  if TerraData[Map[Coord.x,Coord.y].Terrain].DR = 0 then Exit;
  Map[Coord.x,Coord.y].Damage += Damage;
  if Map[Coord.x,Coord.y].Damage > TerraData[Map[Coord.x,Coord.y].Terrain].DR then
  begin
    Map[Coord.x,Coord.y].Terrain  := TerraData[Map[Coord.x,Coord.y].Terrain].DestroyID;
    Map[Coord.x,Coord.y].TerrainB := TerraData[Map[Coord.x,Coord.y].Terrain].DestroyID;
  end;
end;

procedure TLevel.Explosion( Where : TCoord2D; Color : byte; Range : byte;
  Strength : byte; DrawDelay : Word = 50; DamageType : Byte = DAMAGE_FIRE);
var cn       : Byte;
    Damage   : Integer;
    Coord    : TCoord2D;
begin
  if Strength <> 0 then
  for cn := 1 to MAXBEINGS do
    if Assigned(Beings[cn]) then
      with Beings[cn] do
        Affected := False;

  for cn := 1 to Range do
  begin
    UI.Explosion( Where,Color,Range,cn,DrawDelay);
    if Strength <> 0 then
    for Coord in NewArea( Where, Range ).Clamped( Area ) do
      if Distance( Coord, Where ) < cn then
      begin
        if not Level.isEyeContact( Coord, Where ) then Continue;
        Damage := Dice(Strength,6) div Max(1,Distance( Coord, Where ) div 2);
        if Being[ Coord ] <> nil then
        with Being[ Coord ] do
        begin
          if Affected then Continue;
          if HP <= 0  then Continue;
          Knockback( NewDirection( Where, Coord ),Damage);
          Affected := True;
          ApplyDamage(Damage,DamageType);
        end;
        DamageTile(Coord,Damage);
      end;
  end;
end;

procedure TLevel.Breath( Where : TCoord2D; Direction : TDirection; Color : byte; Range : byte;
  Strength : byte; DrawDelay : Word = 50);
var cn       : Byte;
    d        : Byte;
    Damage   : Integer;
    Angle    : Real;
    Coord    : TCoord2D;
begin
  for cn := 1 to MAXBEINGS do
    if Assigned(Beings[cn]) then
      with Beings[cn] do
        Affected := False;

  for cn := 1 to Range+4 do
  begin
    UI.Breath(Where,Direction,Color,Range,cn,DrawDelay);
    for Coord in NewArea( Where, Range ).Clamped( Area ) do
      begin
        d := Distance( Coord, Where );
        if (d = 0) or (d > cn) then Continue;
        
        if Direction.x <> 0 then if Sgn(Coord.x-Where.x) = -Direction.x then Continue;
        if Direction.y <> 0 then if Sgn(Coord.y-Where.y) = -Direction.y then Continue;
        if Direction.x = 0 then begin if Abs(Coord.y-Where.y) < Abs(Coord.x-Where.x) then Continue; end;
        if Direction.y = 0 then begin if Abs(Coord.x-Where.x) < Abs(Coord.y-Where.y) then Continue; end;


        angle := ((Coord.x-Where.x)*Direction.x + (Coord.y-Where.y)*Direction.y)/(vmath.RealDistance(Where.x,Where.y,Coord.x,Coord.y)*vmath.RealDistance(Where.x,Where.y,Where.x+Direction.x,Where.y+Direction.y));
        if angle < 0.76+(d*0.02) then Continue;

        if not isEyeContact(Coord,Where) then Continue;
        if d > Range then Continue;

        Damage := Round(Dice(Strength,6) / Max(1,d div 3));

        if Being[ Coord ] <> nil then
        with Being[ Coord ] do
        begin
          if Affected then Continue;
          if HP <= 0  then Continue;
          Knockback(NewDirection(Where,Coord),Damage);
          Affected := True;
          ApplyDamage(Damage,DAMAGE_FIRE);
        end;

        DamageTile(Coord,Damage);
      end;
  end;
  UI.Draw;
end;




destructor TLevel.Destroy;
begin
  FreeAndNil(Generator);
  Clear;
  FreeAndNil(Vision);
  inherited Destroy;
end;

function TLevel.freeBeingSlot : Word;
var Count : Word;
begin
  Count := 2;
  while (Count < MAXBEINGS) and (Beings[Count] <> nil) do Inc(Count);
  if Count = MAXBEINGS then Exit(0) else Exit(Count);
end;

end.

