// @abstract(BerserkRL -- THOF class unit)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(Oct 22, 2006)
// @lastmod(Oct 22, 2006)
//
// This unit just holds the hall of Fame system for BerserkRL.
//
//  @html <div class="license">
//  This file is part of BerserkRL.
//
//  BerserkRL is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  BerserkRL is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with BerserkRL; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//  @html </div>

unit brhof;
{$mode objfpc}
interface
uses vsystem, brdata;

type
  // Hall of fame entry record.
  THOFEntry = record
    // Name of the character
    Name   : string[30];
    // Number of kills
    Kills  : DWord;
    // Number of turns survived
    Turns  : DWord;
    // Race ID of players killer
    KillBy : Byte;
    // The battlefield
    Field  : Byte;
    // Night of death (modeEndless and modeCampaign only)
    Night  : Byte;
    // Place of death (modeCampaign only)
    DiedOn : Word;
  end;
  
  // Hall of fame array.
  THOFArray = array[1..HOFENTRIES] of THOFEntry;
  // HOFArray pointer.
  PHOFArray = ^THOFArray;

  // Hall of Fame Valkyrie system class. Manages all hall of fame related actions.

  { THOF }

  THOF = class(TSystem)
    // Current added entry
    Current : Byte;
    // Initializes the HOF, loads if file present.
    constructor Create; override;
    // Adds a new entry to the HOF. If the number of kills is less then the worst
    // entry, does nothing.
    procedure Add(Mode : TGameMode; const Name : string; Kills,Turns,KillBy,Battlefield : DWord; Night : Word = 0; DiedOn : Word = 0); reintroduce;
    // Deinitializes the HOF and saves it to score.dat.
    destructor Destroy; override;
    // Gets the proper HOF Array, based on Mode.
    function GetArray(Mode : TGameMode) : PHOFArray;
    private
    // The array of fame -- Massacre
    Massacre : THOFArray;
    // The array of fame -- Endless
    Endless  : THOFArray;
    // The array of fame -- Campaign
    Campaign : THOFArray;
    // Clears the HOF array.
    procedure Clear;
    // Loads the HOF array from score.dat
    procedure Load;
  end;

// THOF system singleton
var HOF : THOF;

implementation
uses SysUtils, zstream;

{ THOF }

constructor THOF.Create;
begin
  inherited Create;
  if not FileExists('score.dat') then Clear else Load;
  Current := 0;
end;

procedure THOF.Clear;
  procedure ClearHOF(HOFArray : PHOFArray);
  var Count : Word;
  begin
    for Count := 1 to HOFENTRIES do
    with HOFArray^[Count] do
    begin
      Name   := '';
      Turns  := 0;
      Kills  := 0;
      KillBy := 0;
      Night  := 0;
    end;
  end;
begin
  ClearHOF(@Massacre);
  ClearHOF(@Endless);
  ClearHOF(@Campaign);
end;

procedure THOF.Load;
var HOFFile : TGZFileStream;
begin
  HOFFile := TGZFileStream.Create('score.dat',gzOpenRead);
  HOFFile.Read(Campaign,SizeOf(THOFArray));
  HOFFile.Read(Endless ,SizeOf(THOFArray));
  HOFFile.Read(Massacre,SizeOf(THOFArray));
  HOFFile.Destroy;
end;

procedure THOF.Add(Mode : TGameMode; const Name : string; Kills, Turns, KillBy,
  Battlefield : DWord; Night : Word = 0; DiedOn : Word = 0);
var Place, Count : Word;
    HOFArray : PHOFArray;
begin
  HOFArray := GetArray(Mode);
  if Kills <= HOFArray^[HOFENTRIES].Kills then Exit;
  Place := HOFENTRIES+1;
  repeat
    Dec(Place)
  until (Place = 2) or (HOFArray^[Place-1].Kills > Kills);
  if (Place = 2) and (HOFArray^[1].Kills < Kills) then Place := 1;
  for Count := HOFENTRIES-1 downto place do
    HOFArray^[Count+1] := HOFArray^[Count];
  HOFArray^[Place].Name  := Name;
  HOFArray^[Place].Kills := Kills;
  HOFArray^[Place].Turns := Turns;
  HOFArray^[Place].KillBy:= KillBy;
  HOFArray^[Place].Field := Battlefield;
  Current := Place;
end;

destructor THOF.Destroy;
var HOFFile : TGZFileStream;
begin
  HOFFile := TGZFileStream.Create('score.dat',gzOpenWrite);
  HOFFile.Write(Campaign, SizeOf(THOFArray));
  HOFFile.Write(Endless , SizeOf(THOFArray));
  HOFFile.Write(Massacre, SizeOf(THOFArray));
  HOFFile.Destroy;
  inherited Destroy;
end;

function THOF.GetArray(Mode : TGameMode) : PHOFArray;
begin
  case Mode of
    modeCampaign : GetArray := @Campaign;
    modeEndless  : GetArray := @Endless;
    modeMassacre : GetArray := @Massacre;
  end;
end;

end.

