// @abstract(BerserkRL -- Graphical User Interface class)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(Apr 11, 2007)
// @lastmod(Apr 11, 2007)
//
// This unit holds the graphical User Interface class of Berserk!.
//
//  @html <div class="license">
//  This file is part of BerserkRL.
//
//  BerserkRL is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  BerserkRL is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with BerserkRL; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//  @html </div>

unit brgui;
interface

uses SysUtils, vrltools, vsdloutput, vsdl, vds, brui;

type TObjArray      = specialize TManagedArray<TSurface>;
     TIntAssocArray = specialize TAssocArray<Integer>;
type

{ TBerserkWindow }

TBerserkWindow = class
  // Window dimensions
  x1,y1,x2,y2 : Word;
  // Next window pointer
  Next : TBerserkWindow;
  // Creates a new window
  constructor Create(nx1,ny1,nx2,ny2 : Word; nNext : TBerserkWindow);
  // Draws the window using GL, calls next draw
  procedure Draw;
  // Destroys this and next windows
  destructor Destroy;
end;

// An overriden TSDLOutput class
type

{ TBerserkOutput }

TBerserkOutput = class(TSDLOutput)
    constructor Create(FullScreen : Boolean = False); reintroduce;
    // Overriding Output to do custom paint jobs.
    procedure Update; override;
    //
    procedure DrawSprites;
    // Deinitialize
    destructor Destroy; override;
    // Holds TSurfaces
    private
    Images     : TObjArray;
    ImageNames : TIntAssocArray;
    Background : TSurface;
    Sprites    : TSurface;
    MenuBack   : TSurface;
    WindowSkin : TSurface;
    Shift      : Integer;
    Manual     : Boolean;
    targetx    : Byte;
    targety    : Byte;
    Windows    : TBerserkWindow;
  end;

type

{ TBreserkGUI }

{ TBerserkGUI }

TBerserkGUI = class(TBerserkUI)
    // Initialization of all data.
    constructor Create(FullScreen : Boolean = False);
    // Sends missile
    procedure SendMissile( Source, TargetPos : TCoord2D; mtype : Byte); override;
    // Draws target X
    procedure Target( Coord : TCoord2D; color : Byte); override;
    // Renders an explosion on the screen.
    procedure Explosion( Where : TCoord2D; Color : byte; Range : byte; Step : byte;  DrawDelay : Word = 50); override;
    // Renders a breath weapon attack
    procedure Breath( Where : TCoord2D; Direction : TDirection; Color : byte; Range : byte; Step : byte;
  DrawDelay : Word = 50); override;
    // Graphical effect of a screen flash, of the given color, and Duration in
    // miliseconds.
    procedure Blink(Color : Byte; Duration : Word = 100); override;
    // Delays for effects.
    procedure Delay(Time : Word); override;
    // Writes a tile description in the msg area.
    procedure MsgCoord( Coord : TCoord2D ); override;
    // Draws a firey background
    procedure DrawFire; override;
    // Draws the whole level to the screen. You need to Update the screen
    // afterwards to see it.
    procedure DrawLevel; override;
    // Window drawing procedure, call Clear afterwards
    procedure Window(x1,y1,x2,y2 : Word); override;
    // Clears all screen and windows
    procedure Clear; override;
    private
    BlinkColor : Byte;
    ExplImg : array[1..6] of TSurface;
  end;


implementation

uses GL, GLU, SDL, vmath, vsystems, voutput, vinput, vsdlinput, brbeing, brplayer, brdata, brlevel;

type TVector3 = object
  X, Y, Z : Real;
end;

function Vector3( X, Y, Z : Real ) : TVector3;
begin
  Vector3.X := X;
  Vector3.Y := Y;
  Vector3.Z := Z;
end;

constructor TBerserkOutput.Create(FullScreen : Boolean = False);
var SName  : string;
    SCount : Byte;
    SearchRec : TSearchRec;
begin
  inherited Create(Fullscreen);
  Windows := nil;
  targetx := 0;
  targety := 0;
  SCount     := 0;
  Manual     := False;
  Images     := TObjArray.Create(20);
  ImageNames := TIntAssocArray.Create;
  if FindFirst('graphics' + PathDelim + '*.png',faAnyFile,SearchRec) = 0 then
  repeat
    Inc(SCount);
    SName := SearchRec.Name;
    Delete(SName,Length(SName)-3,4);
    Images[SCount] := TSurface.Create('graphics' + PathDelim + SearchRec.Name);
    TSurface(Images[SCount]).RenderGL(True);
    ImageNames[SName] := SCount;
  until (FindNext(SearchRec) <> 0);

  MenuBack   := TSurface(Images[ImageNames['menuback']]);
  Background := TSurface(Images[ImageNames['background']]);
  Sprites    := TSurface(Images[ImageNames['spritesheet']]);
  WindowSkin := TSurface(Images[ImageNames['windowskin']]);
  Sprites.SetColorKey(237,20,91);
  Sprites.RenderGL(True);
{  MenuBack.RenderGL(True);
  WindowSkin.RenderGL(True);}

  // VISTA DOESN'T LIKE NONBLENDED TEXTURES! ryanazar@gmail.com
  
end;

procedure TBerserkOutput.Update;
begin
  glEnable( GL_TEXTURE_2D );
  glDisable( GL_DEPTH_TEST );
  glEnable( GL_BLEND );
  glClearColor(0.0,0.0,0.0,1.0);
  glClear(GL_COLOR_BUFFER_BIT or GL_DEPTH_BUFFER_BIT);

  glMatrixMode( GL_PROJECTION );
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0, ValkyrieSDL.Width-1, ValkyrieSDL.Height-1, 0);

  glMatrixMode( GL_MODELVIEW );
  glPushMatrix();
  glLoadIdentity();
  glColor4f(1.0,1.0,1.0,1.0);
  glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );

  if brui.UI.Screen = Game then
  with Background do
  begin
    HideCursor;
    DrawSprites;
    glBindTexture(GL_TEXTURE_2D, GLTexture);
    glBegin(GL_QUADS);
      glColor3f(1.0,1.0,1.0);
      glTexCoord2f(0,0);              glVertex2i(500-1, 0);
      glTexCoord2f(0,GLHeight);       glVertex2i(500-1, 600-1);
      glTexCoord2f(GLWidth,GLHeight); glVertex2i(800-1, 600-1);
      glTexCoord2f(GLWidth,0);        glVertex2i(800-1, 0);
    glEnd();
  end
  else
  with MenuBack do
  begin
    glBindTexture(GL_TEXTURE_2D, GLTexture);
    glBegin(GL_QUADS);
      glColor3f(1.0,1.0,1.0);
      glTexCoord2f(0,0);              glVertex2i(0, 0);
      glTexCoord2f(0,GLHeight);       glVertex2i(0, 600-1);
      glTexCoord2f(GLWidth,GLHeight); glVertex2i(800-1, 600-1);
      glTexCoord2f(GLWidth,0);        glVertex2i(800-1, 0);
    glEnd();
  end;
  if Windows <> nil then Windows.Draw;
  UpdateChars;

  with TBerserkGUI(brui.UI) do
  begin
    if BlinkColor <> 0 then
    begin
      glDisable( GL_TEXTURE_2D );
      glBegin(GL_QUADS);
        glColor4f(GLColors[BlinkColor][0],GLColors[BlinkColor][1],GLColors[BlinkColor][2],0.8);
        glVertex2i(0, 0);
        glVertex2i(0, 600-1);
        glVertex2i(800-1, 600-1);
        glVertex2i(800-1, 0);
      glEnd();
    end;
    if targetx <> 0 then
    begin
    end;
  end;

  if not Manual then SDL_GL_SwapBuffers();
end;

procedure TBerserkOutput.DrawSprites;
var X, Y : Word;
    tw : GLFloat;
    th : GLFloat;
    MinX, MaxX : Word;
    TileMapW : Word;
    seed : Word;
  procedure DrawTile(X,Y : Byte; Tile : Word; width,height : GLFloat; flip : boolean; xmod : ShortInt = 0);
  var TileX,TileY : DWord;
  begin
    if Tile = 0 then Exit;
    TileX := Tile mod TileMapW;
    TileY := Tile div TileMapW+1;

    if flip then
    begin
      glTexCoord2f(width*TileX,height*(TileY-1));       glVertex2i((X-1)*24-xmod-Shift*24, (Y-1)*24 - 8);
      glTexCoord2f(width*TileX,height*TileY);           glVertex2i((X-1)*24-xmod-Shift*24, Y*24);
      glTexCoord2f(width*(TileX-1),height*TileY);       glVertex2i(X*24+xmod-Shift*24, Y*24);
      glTexCoord2f(width*(TileX-1),height*(TileY-1));   glVertex2i(X*24+xmod-Shift*24, (Y-1)*24 - 8);
    end
    else
    begin
      glTexCoord2f(width*(TileX-1),height*(TileY-1)); glVertex2i((X-1)*24-xmod-Shift*24, (Y-1)*24 - 8);
      glTexCoord2f(width*(TileX-1),height*TileY);     glVertex2i((X-1)*24-xmod-Shift*24, Y*24);
      glTexCoord2f(width*TileX,height*TileY);         glVertex2i(X*24+xmod-Shift*24, Y*24);
      glTexCoord2f(width*TileX,height*(TileY-1));     glVertex2i(X*24+xmod-Shift*24, (Y-1)*24 - 8);
    end;
  end;
  procedure SetColor(Light : Word);
  begin
    Light+= 30;
    if not Player.isBerserk
      then glColor3f(0.01*Light,0.01*Light,0.01*Light)
      else if Light = 100 then glColor3f(1.0,0.3,0.3)
                          else glColor3f(0.01*Light,0.1,0.1)
  end;
  procedure SetColor(CO : TColorOverlay);
  begin
    if not Player.isBerserk
      then glColor3f(1.0*CO[1],1.0*CO[2],1.0*CO[3])
      else glColor3f(1.0,0.3,0.3)
  end;
  function  RandomSide(x,y : LongInt) : boolean;
  begin
    x := x + seed;
    y := y - seed;
    Exit(((x+5)*(x+3)*y mod 193*197) mod 2 = 0);
  end;

const lim = 21;
      Dark = 0;

begin
  Shift := Player.Position.X-11;
  Shift := Max(Shift,0);
  Shift := Min(Shift,MAP_MAXX-21);
  MinX := Shift+1;
  MaxX := Shift+21;
  
  Seed := Level.Seed mod 256;

  TileMapW := Sprites.SDLSurface^.w div 24;
  tw := Sprites.GLWidth * 24 / Sprites.SDLSurface^.w ;
  th := Sprites.GLHeight * 32 / Sprites.SDLSurface^.h;
  glBindTexture(GL_TEXTURE_2D, Sprites.GLTexture);
  glBegin(GL_QUADS);


  for Y := 1 to MAP_MAXY do
  begin
    for X := MinX to MaxX do
      with Level.Map[X,Y] do
      begin
        if Level.Vision.getLight(NewCoord2D(X,Y)) > 0 then
          SetColor(Level.Vision.getLight(NewCoord2D(X,Y))*7)
        else
          SetColor(Dark);
          
        if TerraData[Terrain].SpriteB <> 0 then
          DrawTile(X,Y,TerraData[Terrain].SpriteB,tw,th,RandomSide(X,Y))
        else
          if TerraData[TerrainB].SpriteB <> 0 then
            DrawTile(X,Y,TerraData[TerrainB].SpriteB,tw,th,RandomSide(X,Y))
          else DrawTile(X,Y,Level.SpriteBase,tw,th,RandomSide(X,Y))
      end;
    for X := MinX to MaxX do
      with Level.Map[X,Y] do
        if TerraData[Terrain].Sprite <> 0 then
        if Level.Vision.getLight(NewCoord2D(X,Y)) > 0 then
        begin
          SetColor(Level.Vision.getLight(NewCoord2D(X,Y))*7);
          DrawTile(X,Y,TerraData[Terrain].Sprite,tw,th,RandomSide(X,Y));
        end
        else
        begin
          SetColor(Dark);
          DrawTile(X,Y,TerraData[Terrain].Sprite,tw,th,RandomSide(X,Y));
        end;
    for X := MinX to MaxX do
      with Level.Map[X,Y] do
        if Level.Vision.getLight(NewCoord2D(X,Y)) > 0 then
          if Being <> 0 then
          begin
            SetColor(Level.Beings[Being].ColorOverlay);
            if Being = 1 then DrawTile(X,Y,Level.Beings[1].Sprite,Sprites.GLWidth * 32 / Sprites.SDLSurface^.w,th,not Level.Beings[1].FaceLeft,4)
                         else DrawTile(X,Y,Level.Beings[Being].Sprite,tw,th,not Level.Beings[Being].FaceLeft);
          end;
  end;
          
  if targetx <> 0 then
    DrawTile(targetx,targety,30,tw,th,false);

  glEnd();

end;

destructor TBerserkOutput.Destroy;
begin
  FreeAndNil(Images);
  FreeAndNil(ImageNames);
  inherited Destroy;
end;



{ TBreserkGUI }

constructor TBerserkGUI.Create(FullScreen : Boolean = False);
var SCount : Byte;
begin
  Systems.Add(Output,TBerserkOutput.Create(FullScreen));
  Systems.Add(Input,TSDLInput.Create);
  inherited Create;
  BlinkColor := 0;
  with TBerserkOutput(Output) do
    for SCount := 1 to 6 do
    begin
      ExplImg[SCount] := TSurface(Images[ImageNames['explosion00'+IntToStr(SCount)+'0']]);
//      ExplImg[SCount].RenderGL{}{}{}{}{}{(True)};
    end;
end;

procedure TBerserkGUI.SendMissile( Source, TargetPos : TCoord2D;  mtype : Byte);
var Base : UInt32;
    Time  : LongInt;
    Shift : Integer;
    scx,scy,tcx,tcy : Integer;
    dist : Integer;
    full : Integer;
    fin  : Integer;
    pos1,pos2 : Single;
var Velocity,Width : Single; // pixels per milisecond
    Color : TVector3;
    Shot, Light : TSurface;
    px,py : Integer;
begin
  Velocity := 1.0;
  Width := 2.0;
  Color := Vector3(0.7,0.7,0.7);
  case mtype of
    MTBOLT   : begin Velocity := 2.0; Width := 1.0; end;
    MTENERGY : with TBerserkOutput(Output) do
               begin
                 Color := Vector3(0.3,0.3,1.0);
                 Shot  := TSurface(Images[ImageNames['huntershot']]);
                 Light := TSurface(Images[ImageNames['huntershotlight']]);
               end;
    MTICE    : with TBerserkOutput(Output) do
               begin
                 Color := Vector3(0.6,0.6,1.0);
                 Shot  := TSurface(Images[ImageNames['huntershot']]);
                 Light := TSurface(Images[ImageNames['huntershotlight']]);
               end;
    MTBOMB   : begin Color := Vector3(0.6,0.3,0.0); Velocity := 0.5; Width := 4.0; end;
    MTSPORE  : begin Color := Vector3(0.3,1.0,0.3); Velocity := 0.3; Width := 4.0; end;
  end;


  Shift := TBerserkOutput(Output).Shift;
  scx := (Source.x-1)*24-Shift*24+12;
  scy := (Source.y-1)*24+12;
  tcx := (TargetPos.x-1)*24-Shift*24+12+Random(21)-10;
  tcy := (TargetPos.y-1)*24+12+Random(21)-10;

  TBerserkOutput(Output).Manual := True;
  Base := SDL_GetTicks;
  Dist := TriDistance(scx,scy,tcx,tcy);
  if Dist < 1 then Dist := 1;
  Full := Round(Dist/Velocity);
  Fin  := Round(Full*(Min(Dist,250)/Dist));
  if Full = 0 then Full := 1;
  repeat
    Output.Update;
    Sleep(1);
    Time := SDL_GetTicks - Base;
    if Time > Fin then Break;
    pos1 := Time/Full;
    px := Round(scx+(tcx-scx)*pos1);
    py := Round(scy+(tcy-scy)*pos1);

    if mtype <> MTENERGY then
    begin
      glDisable( GL_TEXTURE_2D );

      glLineWidth(Width);
      glColor3f(Color.x,Color.y,Color.z);
      pos2 := Max(Time-10,0)/Full;

      glBegin(GL_LINES);
        glVertex2i(Round(scx+(tcx-scx)*pos2),Round(scy+(tcy-scy)*pos2));
        glVertex2i(px,py);
      glEnd;
    end
    else
    begin
      glBlendFunc( GL_SRC_ALPHA, GL_ONE );
      glBindTexture(GL_TEXTURE_2D, Shot.GLTexture);
      glBegin(GL_QUADS);
        glColor3f(1.0,1.0,1.0);
        glTexCoord2f(0,0);                        glVertex2i(px-5, py-5);
        glTexCoord2f(0,Shot.GLHeight);            glVertex2i(px-5, py+5);
        glTexCoord2f(Shot.GLWidth,Shot.GLHeight); glVertex2i(px+5, py+5);
        glTexCoord2f(Shot.GLWidth,0);             glVertex2i(px+5, py-5);
      glEnd();
      glBindTexture(GL_TEXTURE_2D, Light.GLTexture);
      glBegin(GL_QUADS);
        glColor4f(1.0,1.0,1.0,0.5);
        glTexCoord2f(0,0);                          glVertex2i(px-20, py-20);
        glTexCoord2f(0,Light.GLHeight);             glVertex2i(px-20, py+20);
        glTexCoord2f(Light.GLWidth,Light.GLHeight); glVertex2i(px+20, py+20);
        glTexCoord2f(Light.GLWidth,0);              glVertex2i(px+20, py-20);
      glEnd();
    end;

    SDL_GL_SwapBuffers();
  until Time > Full;
  TBerserkOutput(Output).Manual := False;
  Output.Update;
  TBerserkOutput(Output).targetx := 0;
  TBerserkOutput(Output).targety := 0;
end;

procedure TBerserkGUI.Target( Coord : TCoord2D; color : Byte);
begin
  TBerserkOutput(Output).targetx := Coord.x;
  TBerserkOutput(Output).targety := Coord.y;
end;

procedure TBerserkGUI.Explosion( Where : TCoord2D; Color : byte; Range : byte;Step : byte;
  DrawDelay : Word);
var GLColor        : TVector3;
    sx,sy,s,Shift  : LongInt;
    Base,Time      : UInt32;
begin
  GLColor := Vector3(1.0,1.0,1.0);
  case color of
    Blue    : begin GLColor := Vector3(0.5,0.5,1.0); end;
    Magenta : begin GLColor := Vector3(1.0,0.5,1.0); end;
    Green   : begin GLColor := Vector3(0.5,1.0,0.5); end;
    LightRed: begin GLColor := Vector3(1.2,1.2,1.2); end;
  end;
  
  if Range = 1 then Step := 3;

  Shift := TBerserkOutput(Output).Shift;
  sx := (Where.x-1-Shift)*24+12;
  sy := (Where.y-1      )*24+12;
  TBerserkOutput(Output).Manual := True;
  
  s := 12*Range+24;

  Base := SDL_GetTicks;
  if Step > 6 then Step := 6;
  repeat
    Sleep(1);
    Time := SDL_GetTicks - Base;
    if Time > DrawDelay then Break;

    glBlendFunc( GL_SRC_ALPHA, GL_ONE );
    glBindTexture(GL_TEXTURE_2D, ExplImg[Step].GLTexture);
    glBegin(GL_QUADS);
      glColor3f(GLColor.X,GLColor.Y,GLColor.Z);
      glTexCoord2f(0,0);                                          glVertex2i(sx-s, sy-s);
      glTexCoord2f(0,ExplImg[Step].GLHeight);                     glVertex2i(sx-s, sy+s);
      glTexCoord2f(ExplImg[Step].GLWidth,ExplImg[Step].GLHeight); glVertex2i(sx+s, sy+s);
      glTexCoord2f(ExplImg[Step].GLWidth,0);                      glVertex2i(sx+s, sy-s);
    glEnd();

    SDL_GL_SwapBuffers();
  until Time > DrawDelay;
  TBerserkOutput(Output).Manual := False;
  Output.Update;
end;

procedure TBerserkGUI.Breath( Where : TCoord2D; Direction : TDirection; Color : byte;
  Range : byte; Step : byte;  DrawDelay : Word);
var ax,ay : Byte;
    dx,dy    : ShortInt;
    d        : Byte;
    c1,c2,c3 : Byte;
    Vis      : boolean;
    Angle    : Real;
    img      : TSurface;
var GLColor      : TVector3;
    sx,sy,Shift  : LongInt;
    Base,Time    : UInt32;
const s = 24;
begin
  GLColor := Vector3(1.0,1.0,1.0);
  case color of
    Blue    : begin GLColor := Vector3(0.5,0.5,1.0); end;
    Magenta : begin GLColor := Vector3(1.0,0.5,1.0); end;
    Green   : begin GLColor := Vector3(0.5,1.0,0.5); end;
    LightRed: begin GLColor := Vector3(1.2,1.2,1.2); end;
  end;

  Shift := TBerserkOutput(Output).Shift;
  TBerserkOutput(Output).Manual := True;

  Base := SDL_GetTicks;

  dx := DirXValue(Direction.Code);
  dy := DirYValue(Direction.Code);
  repeat
    Vis := False;
    Sleep(1);
    Output.Update;
    Time := SDL_GetTicks - Base;
    if Time > DrawDelay then Break;


    for ax := Max(1,Where.x-Range) to Min(MAP_MAXX,Where.x+Range) do
      for ay := Max(1,Where.y-Range) to Min(MAP_MAXY,Where.y+Range) do
      begin
        if dx <> 0 then if Sgn(ax-Where.x) = -dx then Continue;
        if dy <> 0 then if Sgn(ay-Where.y) = -dy then Continue;
        if dx = 0 then begin if Abs(ay-Where.y) < Abs(ax-Where.x) then Continue; end;
        if dy = 0 then begin if Abs(ax-Where.x) < Abs(ay-Where.y) then Continue; end;

        d := Distance(ax,ay,Where.x,Where.y);
        if d = 0 then Continue;

        angle := ((ax-Where.x)*dx + (ay-Where.y)*dy)/(RealDistance(Where.x,Where.y,ax,ay)*RealDistance(Where.x,Where.y,Where.x+dx,Where.y+dy));
        if angle < 0.76+(d*0.02) then Continue;

        if Level.Vision.getLight(NewCoord2D(ax,ay)) > 0 then Vis := True else Continue;
        if not Level.isEyeContact(NewCoord2D(ax,ay),Where) then Continue;

        if d > Range then Continue;
        if Random(2) = 0 then d := d + Random(2);

        sx := (ax-1-Shift)*24+12;
        sy := (ay-1      )*24+12;

        img := nil;

        if d = Step   then img := ExplImg[1];
        if d = Step-1 then img := ExplImg[2];
        if d = Step-2 then img := ExplImg[3];
        if d = Step-3 then img := ExplImg[4];
        if d = Step-4 then img := ExplImg[5];

        if img = nil then Continue;
        glBlendFunc( GL_SRC_ALPHA, GL_ONE );
        glBindTexture(GL_TEXTURE_2D, Img.GLTexture);
        glBegin(GL_QUADS);
          glColor3f(GLColor.X,GLColor.Y,GLColor.Z);
          glTexCoord2f(0,0);                      glVertex2i(sx-s, sy-s);
          glTexCoord2f(0,Img.GLHeight);           glVertex2i(sx-s, sy+s);
          glTexCoord2f(Img.GLWidth,Img.GLHeight); glVertex2i(sx+s, sy+s);
          glTexCoord2f(Img.GLWidth,0);            glVertex2i(sx+s, sy-s);
        glEnd();
      end;

    SDL_GL_SwapBuffers();
  until Time > DrawDelay;
  TBerserkOutput(Output).Manual := False;
end;

procedure TBerserkGUI.Blink(Color : Byte; Duration : Word);
begin
  BlinkColor := Color;
  Output.Update;
  Delay(Duration);
  BlinkColor := 0;
  Output.Update;
end;

procedure TBerserkGUI.Delay(Time : Word);
var Base : UInt32;
    Count : LongInt;
begin
  Base := SDL_GetTicks;
  repeat
    Sleep(1);
    Output.Update;
  until SDL_GetTicks >= Base+Time;
end;

procedure TBerserkGUI.MsgCoord( Coord : TCoord2D );
begin
  Output.DrawString(52,24,LightGray,'                           ');
  with Level.Map[Coord.x,Coord.y] do
  if Level.Vision.getLight(Coord) > 0 then
    if Being > 0 then
      Output.DrawString(52,24,White,Level.Beings[Being].LookDescribe)
    else
      Output.DrawString(52,24,White,TerraData[Terrain].Name)
  else
    Output.DrawString(52,24,White,'nothing');
  Output.Update;
end;

procedure TBerserkGUI.DrawFire;
begin
  Output.Clear;
end;

procedure TBerserkGUI.DrawLevel;
begin
  TBerserkOutput(Output).targetx := 0;
  TBerserkOutput(Output).targety := 0;
end;

procedure TBerserkGUI.Window(x1, y1, x2, y2: Word);
begin
  Output.ClearRect(x1, y1, x2, y2);
  with TBerserkOutput(Output) do
    Windows := TBerserkWindow.Create((x1-1)*10, (y1-1)*24,x2*10,y2*24,Windows);
end;

procedure TBerserkGUI.Clear;
begin
  Output.Clear;
  FreeAndNil(TBerserkOutput(Output).Windows)
end;


{ TBerserkWindow }

constructor TBerserkWindow.Create(nx1, ny1, nx2, ny2: Word;
  nNext: TBerserkWindow);
begin
  x1 := nx1;
  y1 := ny1;
  x2 := nx2;
  y2 := ny2;
  Next := nNext;
end;

procedure TBerserkWindow.Draw;
var c1,c13,c23 : Real;
  procedure DrawQuad(qx1,qy1,qx2,qy2 : Word; tcx1,tcy1,tcx2,tcy2 : Real);
  begin
    glTexCoord2f(tcx1,tcy1); glVertex2i(qx1, qy1);
    glTexCoord2f(tcx1,tcy2); glVertex2i(qx1, qy2);
    glTexCoord2f(tcx2,tcy2); glVertex2i(qx2, qy2);
    glTexCoord2f(tcx2,tcy1); glVertex2i(qx2, qy1);
  end;
begin
with TBerserkOutput(Output).WindowSkin do
begin
  c13 := GLHeight/3;
  c23 := (2*GLHeight)/3;
  c1  := GLHeight;

  glBindTexture(GL_TEXTURE_2D, GLTexture);
  glBegin(GL_QUADS);
    glColor4f(0.5,0.5,0.5,0.7);
    
    // Left-top
    DrawQuad(x1,    y1,    x1+10, y1+10, 0,  0,  c13,c13); // Left-top
    DrawQuad(x1+10, y1,    x2-10, y1+10, c13,0,  c23,c13); // Center-top
    DrawQuad(x2-10, y1,    x2,    y1+10, c23,0,  c1, c13); // Right-top

    DrawQuad(x1,    y1+10, x1+10, y2-10, 0,  c13,c13,c23); // Left-center
    DrawQuad(x1+10, y1+10, x2-10, y2-10, c13,c13,c23,c23); // Center
    DrawQuad(x2-10, y1+10, x2   , y2-10, c23,c13,c1 ,c23); // Right-center

    DrawQuad(x1,    y2-10, x1+10, y2   , 0,  c23,c13,c1);   // Left-bottom
    DrawQuad(x1+10, y2-10, x2-10, y2   , c13,c23,c23,c1);   // Center-bottom
    DrawQuad(x2-10, y2-10, x2   , y2   , c23,c23,c1 ,c1);   // Right-bottom

  glEnd();
end;
  if Next <> nil then Next.Draw;
end;

destructor TBerserkWindow.Destroy;
begin
  if Next <> nil then FreeAndNil(Next);
end;

end.

