// @abstract(BerserkRL -- general data unit)
// @author(Kornel Kisielewicz <admin@chaosforge.org>)
// @created(Oct 16, 2006)
// @lastmod(Oct 22, 2006)
//
// This unit holds the global variables and the game data for Berserk. It also
// implements some global helper functions.
//
//  @html <div class="license">
//  This file is part of BerserkRL.
//
//  BerserkRL is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  BerserkRL is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with BerserkRL; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
//  @html </div>

unit brdata;
{$mode objfpc}
interface
uses vutil, vds;

// Defines the game mode - Massacre, Nights or Campaign
type TGameMode = (modeMassacre, modeEndless, modeCampaign);

const
{$INCLUDE ../bin/lua/const.lua}

var
    // Holds the version of the game -- read from the first line of 'version.txt'
    Version   : string  = '';
    // Set to true if GodMode enabled -- activated by launching the game with
    // the -god parameter.
    GodMode   : Boolean = False;
    // Quickstart for starting immidately on the field for debug purposes.
    QuickStart: Boolean = False;


const // Number of entries in the hall of fame. If changed then score.dat needs
      // to be deleted.
      HOFENTRIES    = 20;
      // X position from which to draw the map
      MAP_POSX      = 1;
      // Y position from which to draw the map
      MAP_POSY      = 1;
      // The amount of TBeing.SpeedCount needed to be accumulated to make an
      // Action.
      SPEEDLIMIT    = 5000;
      // The amount of TBeing.WillCount needed to make a Energy regeneration.
      WILLLIMIT     = 1000;
      // Duration of one spawncycle
      NIGHTDURATION = 10*300;

      // Amount of damage that produces a 1 field knockback
      KNOCKBACKVALUE = 8;
      
      // Maximum distance of bomb throw
      BOMBDISTANCE   = 7;
      
      ARENA_FIELDS = 1;
      ARENA_FOREST = 2;
      ARENA_TOWN   = 3;
      ARENA_RIVER  = 4;
      ARENA_BRIDGE = 5;
      ARENA_SNOW   = 6;

      // TBeing.TryMove scan value -- no obstructions
      SCAN_OK       = 0;
      // TBeing.TryMove scan value -- out of bounds
      SCAN_OUT      = 1000;
      // TBeing.TryMove scan value -- map feature (eg. wall blocks)
      SCAN_BLOCK    = 2000;
      // TBeing.TryMove scan value -- closed door blocks
      SCAN_DOOR     = 3000;

// Graphics mode only -- sets a color overlay, where all 1.0 are natural color.
type TColorOverlay = array[1..3] of Real;

// Constant for TColorOverlay defining natural color (no overlay)
const NOCOLOROVERLAY : TColorOverlay = (1.0,1.0,1.0);

// Data for a single terrain tile type
type TTerrainData = record
       // Name of the tile for the look command
       Name      : String[20];
       // Picture of the lit tile
       Picture   : Word;
       // Picture of the tile when unlit
       DarkPic   : Word;
       // Amount of Damage needed to Destroy
       DR        : Word;
       // ID of the tile to change to if blood spilled over it
       BloodID   : Word;
       // ID of the tile to change to if tile destroyed (eg. for walls)
       DestroyID : Word;
       // ID of the tile to change to if tile acted upon (eg. doors)
       ActID     : Word;
       // Sprite Base
       SpriteB : Word;
       //
       Sprite  : Word;
       // Move cost modifier (percent)
       MoveCost  : Byte;
       // Flags of the tile (see constants starting with TF)
       Flags     : TFlags;
     end;

// Artificial intelligence type
type TAITypeSet = set of Byte;

const // Sets number of skills.
      MAXSKILLS     = 6;
      // Sets number of max skill requirements.
      MAXREQS       = 3;
      
const SKILLREQ_STAT     = 1;
      SKILLREQ_SKILL    = 2;
      STAT_STR = 1;
      STAT_DEX = 2;
      STAT_WIL = 3;
      STAT_END = 4;
      

// Skill Requirement array. First number is the requirement type, second is the
// requirement type index, and third is the required level.
type TSkillReq  = array[1..3] of Byte;

// Skill data for the Skills array.
type TSkillData = record
       Name        : String[20];
       Description : AnsiString;
       MaxLevel    : Byte;
       Picture     : String[80];
       Reqs        : array[1..MAXREQS] of TSkillReq;
     end;
     
// Skill array for the player
type TSkills = array[1..MAXSKILLS] of Byte;
     
const SKILL_IRONMAN   = 1;
      SKILL_RUNNING   = 2;
      SKILL_SWEEP     = 3;
      SKILL_WHIRLWIND = 4;
      SKILL_IMPALE    = 5;
      SKILL_JUMP      = 6;

const SkillData : array[1..MAXSKILLS] of TSkillData = (
      ( Name        : 'Ironman';
        Description : 'Each level of this skill makes you tougher, by giving you an additional 5 hitpoints.';
        MaxLevel    : 100;
        Picture     : '@g.....@/@g..@d@@@g..@/@g.....';
        Reqs        : ((0,0,0),(0,0,0),(0,0,0)); ),

      ( Name        : 'Running';
        Description : 'This skill greatly improves the benefits of Running mode. Each successive level increases speed, and reduces energy drain.';
        MaxLevel    : 3;
        Picture     : '@g.....@/@g..@d@@@g..@/@g.....';
        Reqs        : ((SKILLREQ_STAT,STAT_END,12),(0,0,0),(0,0,0)); ),

      ( Name        : 'Sweep attack';
        Description : 'This skill allows you to attack three enemies at once, as long as they stand side by side and beside you. The attack is less effective than a normal one though. Each level reduces the time needed and energy cost.';
        MaxLevel    : 3;
        Picture     : '@g.@B*@g...@/@g.@B*@g@d@@@g..@/@g.@B*@g...';
        Reqs        : ((SKILLREQ_STAT,STAT_STR,12),(0,0,0),(0,0,0)); ),

      ( Name        : 'Whirlwind attack';
        Description : 'This skill allows you to attack ALL surrounding enemies at once! The attack is a less effective than the normal attack though. Each level reduces the time needed and energy cost.';
        MaxLevel    : 3;
        Picture     : '@g.@B***@g.@/@g.@B*@d@@@B*@g.@/@g.@B***@g.';
        Reqs        : ((SKILLREQ_STAT,STAT_STR,16),(SKILLREQ_SKILL,SKILL_SWEEP,2),(0,0,0)); ),

      ( Name        : 'Impale attack';
        Description : 'This skill makes you dash one step and attack in the given direction immediately with greater strength. It works only if there is exactly one free space between you and the enemy. Each level decreases energy and time cost.';
        MaxLevel    : 3;
        Picture     : '@g.....@/@B*@@@d@@@g..@/@g.....';
        Reqs        : ((SKILLREQ_STAT,STAT_STR,14),(SKILLREQ_SKILL,SKILL_RUNNING,1),(0,0,0)); ),

      ( Name        : 'Jump attack';
        Description : 'This skill allows you to jump two squares away using the sword as a jump pole. If there''s a monster under you during the jump, he''ll get impaled!';
        MaxLevel    : 3;
        Picture     : '@g.....@/@B@@*@d@@@g..@/@g.....';
        Reqs        : ((SKILLREQ_STAT,STAT_DEX,12),(SKILLREQ_SKILL,SKILL_RUNNING,1),(SKILLREQ_SKILL,SKILL_IMPALE,1)); )

      );

type TWordAssocArray = specialize TAssocArray<Word>;

// Terrain data -- collection of tile definitions. Tile 0 is special -- it's
// not supposed to be used, but is there to prevent and identify errors.
var TerraData : array of TTerrainData;
    TerraID   : TWordAssocArray;

// Killcounter object for use by the player.
type

{ TKills }

{ TKillTable }

TKillTable = object
  private
  // Amount of each being type kills. 1 is used for storing global kill count.
  Count       : array[1..100] of DWord;
  // Holds the greatest amount of kills in one Turn
  Max         : DWord;
  // Holds the amount of uninterruptded kills
  Spree       : DWord;
  // Holds the longest killing spree
  MaxSpree       : DWord;
  // Holds the longest killing spree length
  MaxSpreeLength : DWord;
  // Holds the beginning Turn of the killing spree
  SpreeStart     : DWord;
  // Holds the last value of Kills
  Last        : DWord;
  // function for property access
  function GetKills(index : DWord) : DWord;
  public
  // Adds a kill to the table.
  procedure Add(id : Word);
  // Clears all values.
  procedure Clear;
  // Does an update. Checks for killing sprees.
  procedure Update(TurnCount : DWord);
  // Returns kills done this turn
  function ThisTurn : Word;
  // Property for accessing all kills.
  property All : DWord read Count[1];
  // Property for accessing kills of separate beings.
  property Get[index : DWord] : DWord read GetKills; default;
  // Property for accessing Max.
  property BestTurn : DWord read Max;
  // Property for accessing MaxSpree.
  property BestSpree : DWord read MaxSpree;
  // Property for accessing MaxSpree.
  property BestSpreeLength : DWord read MaxSpreeLength;
end;


// Rolls three 6-sided dice. If 3 or 4 is rolled, then the value is -100,
// if 17,18 is rolled then the value is 100.
function RollDice : Integer;
// Function for debug strings. Returns string if GodMode, '' otherwise.
function GodStr(Str : String) : string;
// Returns the name of the battlefield of given ID
function ArenaToString(ArenaID : byte) : string;
// Returns the name of the battlefield of given ID
function ModeToString(GameMode : TGameMode) : string;
// Returns a text representation of a given requirement.
function ReqToString(Req : TSkillReq) : string;


implementation

uses SysUtils, brui;

function ArenaToString(ArenaID : byte) : string;
begin
  case ArenaID of
    ARENA_FIELDS : Exit('Fields');
    ARENA_FOREST : Exit('Forest');
    ARENA_TOWN   : Exit('Town');
    ARENA_RIVER  : Exit('River');
    ARENA_BRIDGE : Exit('Bridge');
    ARENA_SNOW   : Exit('Snow');
  else Exit('Unknown');
  end
end;


function RollDice : Integer;
begin
  RollDice := Dice(3,6);
  if (RollDice = 3)  or (RollDice = 4)  then Exit(-100);
  if (RollDice = 17) or (RollDice = 18) then Exit(100);
end;


function GodStr(Str : String) : string;
begin
  if GodMode then Exit(Str) else Exit('');
end;

function ModeToString(GameMode : TGameMode) : string;
begin
  case GameMode of
    modeCampaign : Exit('Campaign');
    modeEndless  : Exit('Endless');
    modeMassacre : Exit('Massacre');
  end;
  Exit('');
end;

function ReqToString(Req : TSkillReq) : string;
begin
  case Req[1] of
    SKILLREQ_STAT :
      case Req[2] of
        STAT_STR : Exit('Strength '+IntToStr(Req[3]));
        STAT_DEX : Exit('Dexterity '+IntToStr(Req[3]));
        STAT_WIL : Exit('Willpower '+IntToStr(Req[3]));
        STAT_END : Exit('Endurance '+IntToStr(Req[3]));
      end;
    SKILLREQ_SKILL : Exit(SkillData[Req[2]].Name+' level '+IntToStr(Req[3]));
  else Exit('???');
  end;
end;

{ TKills }

procedure TKillTable.Add(id : Word);
begin
  Inc(Count[id]);
  Inc(Count[1]);
end;

procedure TKillTable.Clear;
var cnt : Word;
begin
  for cnt := 1 to 100 do Count[cnt] := 0;
  Last              := 0;
  Max               := 0;
  Spree             := 0;
  SpreeStart        := 0;
  MaxSpree          := 0;
  MaxSpreeLength    := 0;
end;

function TKillTable.ThisTurn : Word;
begin
  ThisTurn := Count[1] - Last;
end;


procedure TKillTable.Update(TurnCount : DWord);
var Diff : DWord;
begin
  Diff := ThisTurn;
  if Diff > 0 then
  begin
    if Diff > Max then
    begin
      Max := Diff;
      if Diff > 5 then UI.Msg(IntToStr(Max)+' kills!');
    end;

    if Spree = 0 then
    begin
      SpreeStart := TurnCount;
      Spree      := Diff;
    end
    else
      Spree += Diff;
  end
  else // Diff = 0
  begin
    if Spree > 0 then
    begin
      if Spree > MaxSpree then
      begin
        MaxSpree := Spree;
        MaxSpreeLength := TurnCount - SpreeStart;
      end;
      Spree := 0;
    end;
  end;
  Last := Count[1];
end;

// function for property access
function TKillTable.GetKills(index : DWord) : DWord;
begin
  Exit(Count[index]);
end;

initialization

TerraID := TWordAssocArray.Create;

finalization

FreeAndNil( TerraID );
end.

