--- Return random table element by index.
-- @param t Table from which the element will be retrieved
-- @return Random element from t
function table.random( t )
	return t[ math.random( #t ) ]
end

--- Returns a shallow simple copy of the table.
-- Metatables are not copied, just a simple pairs transfer.
-- @param t Table to be copied
-- @return Shallow copy of t
function table.copy( t )
	local u = { }
	for k, v in pairs(t) do u[k] = v end
	return u
end

--- Returns a shallow simple copy of the table, based on numeric indices.
-- Metatables are not copied, just a simple ipairs transfer.
-- @see table.copy
-- @param t Table to be copied
-- @return Shallow indexed copy of t
function table.icopy( t )
	local u = { }
	for k, v in ipairs(t) do u[k] = v end
	return u
end


--- Clones (shallow) the table.
-- Metatables are copied, apart from that works the same as copy.
-- @param t Table to be cloned
-- @return Shallow clone of t
function table.clone( t )
	local u = setmetatable({}, getmetatable(t))
	for k, v in pairs(t) do u[k] = v end
	return u
end

--- Merges two tables.
-- Metatables are taken from the first table, keys from the second table
-- overwrite the first.
-- @param t Primary table
-- @param u Table to be merged with
-- @return New table that is a merge of t and u
function table.merge( t, u )
	local r = clone( t )
	for k, v in pairs(u) do r[k] = v end
	return r
end

--- Rearches for a element in the table.
-- @param t Table to be searched
-- @param element Element searched
-- @return true if element found, false otherwise
function table.find( t, element )
	for _, v in pairs(t) do 
		if element == v then
			return true
		end
	end
	return false
end
  
--- Rearches for a element in the table, indexed.
-- @see table.find
-- @param t Table to be searched
-- @param element Element searched
-- @return true if element found, false otherwise
function table.ifind( t, element )
	for _, v in ipairs(t) do 
		if element == v then
			return true
		end
	end
	return false
end

--- Applies recursively a prototype to table.
-- Initializes keys in t that are not initialized with key,values from 
-- proto. If proto.proto is defined, runs recursively.
-- @param t Table to have the prototype applied
-- @param proto Prototype to apply
function table.prototype( t, proto )
  for k,v in pairs(proto) do
    if not t[k] then
       t[k] = v
	end
  end
  if proto.proto then
    table.prototype( t, proto.proto )
  end
  t.proto = proto
end

--- Performs a dice roll.
-- @param dice Number of dice
-- @param sides Number of sides of the dice
-- @return 
function math.die_roll( dice, sides )
	local result = 0
	for i=1,dice do
		result = result + math.random(sides)
	end
	return result
end

--- Calculates approximate distance between two coordinates.
-- @param x1 Coordinate 1 x
-- @param y1 Coordinate 1 y
-- @param x2 Coordinate 2 x
-- @param y2 Coordinate 2 y
-- @return Approximate distance between (x1,y1) and (x2,y2)
function math.distance( x1,y1, x2,y2 )
	local ax, ay = math.abs(x1-x2), math.abs(y1,y2)
	return math.floor( math.min(ax,ay) / 2 + math.max(ax,ay) )
end



--- Set operations support
set = {}

--- Set metatable
-- @field __mul Intersection as multiplication
-- @field __add Union as addition
set.__mt = {
	__mul = set.intersection,
	__add = set.union,
}

--- Creates a new set from table t
-- A set has all it's elements as keys set to true. Metatable is 
-- assigned, so that we can perform intersection using the * operator
-- and union using the + operator.
-- @param t Table to create the set from
-- @return New set with keys from table t
function set.new( t )
	local r = {}
	setmetatable(r, set.__mt)
	for _, l in ipairs(t) do r[l] = true end
	return r
end
    
--- Returns an union of sets a and b.
-- Result is a new set with all the keys from a and b.
-- @param a Must be a valid set
-- @param b Must be a valid set
-- @return Union of a and b
function set.union( a, b )
	local r = set.new{}
	for k in pairs(a) do r[k] = true end
	for k in pairs(b) do r[k] = true end
	return r
end
    
--- Returns an intersection of sets a and b.
-- Result is a new set with all the keys that are both in a and b.
-- @param a Must be a valid set
-- @param b Must be a valid set
-- @return Intersection of a and b
function set.intersection( a, b )
	local r = set.new{}
	for k in pairs(a) do r[k] = b[k] end
	return r
end