/* This file is part of Beifahrer.
 *
 * Copyright (C) 2010 Philipp Zabel
 *
 * Beifahrer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Beifahrer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Beifahrer. If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Hildon;

public class MyOffersWindow : StackableWindow {
	AdacMitfahrclub adac;
	ListStore store;
	Alignment alignment;
	TreeView tree;
	TreeViewColumn route_column;
	Label no_offers;
	bool preload = false;

	public MyOffersWindow (AdacMitfahrclub _adac) {
		adac = _adac;
	}

	construct {
		set_title ("My offers");

		var menu = new AppMenu ();
		var goto_website = new Gtk.Button.with_label (_("Show website"));
		goto_website.show ();
		menu.append (goto_website);
		set_main_menu (menu);

		store = new ListStore (4, typeof (string), typeof (string), typeof (string), typeof (Lift));

		alignment = new Alignment (0.0f, 0.0f, 1.0f, 1.0f);
		alignment.top_padding = MARGIN_HALF;
		alignment.left_padding = MARGIN_DOUBLE;
		alignment.right_padding = MARGIN_DOUBLE;

		tree = (TreeView) Hildon.gtk_tree_view_new_with_model (UIMode.NORMAL, store);
		Hildon.gtk_widget_set_theme_size (tree, SizeType.FINGER_HEIGHT);

		tree.set_headers_visible (false);
		tree.set_rules_hint (true);

		// Tree selection object
		var selection = tree.get_selection ();
		selection.set_mode (SelectionMode.SINGLE);

		// Source and destination column
		route_column = new TreeViewColumn.with_attributes ("Route", new CellRendererText (), "markup", 0);
		route_column.set_reorderable (false);
		route_column.set_sizing (TreeViewColumnSizing.AUTOSIZE);
		route_column.set_expand (true);
		tree.append_column (route_column);

		// Date and time column
		var datetime_column = new TreeViewColumn.with_attributes ("DateTime", new CellRendererText (), "text", 1);
		datetime_column.set_reorderable (false);
		datetime_column.set_sizing (TreeViewColumnSizing.AUTOSIZE);
		datetime_column.set_expand (true);
		tree.append_column (datetime_column);

		// Active column
		var renderer = new CellRendererText ();
		Pango.AttrList attr_list = new Pango.AttrList ();
		var style = Gtk.rc_get_style_by_paths (Gtk.Settings.get_default (), "SmallSystemFont", null, typeof (void));
		if (style != null) {
			var attr_font_desc = new Pango.AttrFontDesc (style.font_desc.copy ());
			attr_list.insert ((owned) attr_font_desc);
		}
		Gdk.Color color;
		this.ensure_style ();
		if (this.style.lookup_color ("ActiveTextColor", out color)) {
			Pango.Attribute attr_color = Pango.attr_foreground_new (color.red, color.green, color.blue);
			attr_list.insert ((owned) attr_color);
		}
		renderer.attributes = attr_list;
		var active_column = new TreeViewColumn.with_attributes ("Active", renderer, "text", 2);
		active_column.set_reorderable (false);
		active_column.set_sizing (TreeViewColumnSizing.AUTOSIZE);
		active_column.set_expand (false);
		tree.append_column (active_column);

		var pannable = new PannableArea ();
		pannable.add (tree);

		no_offers = new Label (_("No offers"));
		Hildon.helper_set_logical_font (no_offers, "LargeSystemFont");
		Hildon.helper_set_logical_color (no_offers, RcFlags.FG, StateType.NORMAL, "SecondaryTextColor");
		no_offers.set_size_request (-1, 6 * 70);
		no_offers.set_alignment ((float) 0.5, (float) 0.42);

		var vbox = new VBox (false, 0);
		vbox.pack_start (pannable, true, true, 0);
		vbox.pack_start (no_offers, false, false, 0);

		alignment.add (vbox);
		add (alignment);

		alignment.show_all ();
		no_offers.hide ();
		route_column.set_visible (!BeifahrerProgram.orientation.portrait);

		tree.row_activated.connect (this.on_row_activated);
		Gdk.Screen.get_default ().size_changed.connect (this.on_orientation_changed);
		goto_website.clicked.connect (on_goto_website_clicked);
	}

	public async void get_my_offers () {
		TreeIter iter;
		Hildon.gtk_window_set_progress_indicator (this, 1);
		bool logged_in = yield adac.login_async ();
		if (!logged_in) {
			Banner.show_information (this, null, "Login failed.");
			return;
		}

		var lift_list = yield adac.get_my_offers ();
		foreach (Lift lift in lift_list) {
			string datetime = "%02d.%02d.%04d".printf (lift.time.day, lift.time.month, lift.time.year);
			if (lift.time.hour >= 0)
				datetime += ", %d:%02d".printf (lift.time.hour, lift.time.minute);
			store.insert_with_values (out iter, -1, 0, lift.city_from + " - " + lift.city_to,
			                                        1, datetime,
								2, (LiftFlags.ACTIVE in lift.flags) ? _("Active") : _("Inactive"),
								3, lift);
		}

		if (lift_list.length () > 6)
			alignment.right_padding = MARGIN_DEFAULT;
		if (lift_list.length () == 0)
			no_offers.show ();

		Hildon.gtk_window_set_progress_indicator (this, 0);
	}

	void on_goto_website_clicked () {
		BeifahrerProgram.open_browser (this, adac.get_my_offers_url ());
	}

	private void on_row_activated (TreeView tree, TreePath path, TreeViewColumn column) {
		TreeModel model = tree.model;
		TreeIter iter;
		Lift lift;

		if (model.get_iter (out iter, path)) {
			model.get (iter, 3, out lift);

			var dialog = new OfferEditDialog (this, adac, lift);
			dialog.show ();
		}
	}

	private void on_orientation_changed (Gdk.Screen screen) {
	//	route_column.set_visible (!BeifahrerProgram.orientation.portrait);
	}
}
