/* This file is part of Beifahrer.
 *
 * Copyright (C) 2010 Philipp Zabel
 *
 * Beifahrer is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Beifahrer is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Beifahrer. If not, see <http://www.gnu.org/licenses/>.
 */

using Gtk;
using Hildon;

class CityButton : Hildon.Button {
	private enum Response {
		RADIUS = 1
	}

	Gtk.Dialog dialog;
	TouchSelector city_selector;
	PickerButton radius_button;
	int active = 0;
	int radius = 0;

	public CityButton (SizeType size, ButtonArrangement arrangement, List<City> city_list) {
		GLib.Object (arrangement: arrangement, size: size);
		set_style (ButtonStyle.PICKER);

		city_selector = new TouchSelectorEntry.text ();
		foreach (unowned City city in city_list)
			city_selector.append_text (city.name);

		clicked.connect (on_clicked);
	}

	public void set_active (int _active) {
		active = _active;
		city_selector.set_active (0, active);
		update_value ();
	}

	public int get_active () {
		return active;
	}

	public void set_radius (int _radius) {
		radius = _radius;
		update_value ();
	}

	public int get_radius () {
		return radius;
	}

	private void on_clicked () {
		dialog = new Gtk.Dialog ();
		dialog.set_transient_for (find_parent_window ());

		var content_area = (Box) dialog.get_content_area ();
		content_area.pack_start (city_selector, true, true, 0);
		city_selector.set_size_request (-1, 5*70);

		var radius_selector = new TouchSelector.text ();
		for (int km = 0; km <= 50; km += 10)
			radius_selector.append_text ("%d km".printf (km));

		radius_button = new PickerButton (SizeType.FINGER_HEIGHT,
		                                  ButtonArrangement.VERTICAL);
		radius_button.set_selector (radius_selector);
		radius_button.set_title (_("Radius"));
		radius_button.set_alignment (0.0f, 0.0f, 0.5f, 0.5f);
		radius_button.set_active (radius / 10);
		dialog.add_action_widget (radius_button, Response.RADIUS);

		var action_area = (ButtonBox) dialog.get_action_area ();
		action_area.set_child_secondary (radius_button, true);

		dialog.add_button (_("Done"), Gtk.ResponseType.OK);

		dialog.response.connect (on_response);

		dialog.show_all ();
	}

	private void on_response (int response_id) {
		if (response_id == Gtk.ResponseType.OK) {
			active = city_selector.get_active (0);
			radius = radius_button.get_active () * 10;
			update_value ();
		}
		if (response_id != Response.RADIUS) {
			// Unlink the city_selector so it doesn't get destroyed with the dialog
			var content_area = (Box) dialog.get_content_area ();
			content_area.remove (city_selector);
			dialog.destroy ();
			dialog = null;
		}
	}

	private void update_value () {
		if (get_radius () == 0)
			set_value (city_selector.get_current_text ());
		else
			set_value ("%s, radius %d km".printf (city_selector.get_current_text (), radius));
	}

	private Gtk.Window? find_parent_window () {
		for (Gtk.Widget p = parent; p != null; p = p.parent)
			if (p is Gtk.Window)
				return (Gtk.Window) p;
		return null;
	}
}
