#!/usr/bin/env python

#    This file is part of battery-eye.
#
#    battery-eye is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    battery-eye is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with battery-eye.  If not, see <http://www.gnu.org/licenses/>.

#    Copyright 2010 Jussi Holm

import gtk
import hildon

import pango

import beye

import time
import datetime

class MainWindow(hildon.StackableWindow):
    def __init__(self):
        hildon.StackableWindow.__init__(self)
        self.program = hildon.Program.get_instance()
        self.connect("delete_event", lambda a, b: gtk.main_quit())
        self.program.add_window(self)

        self.layout = gtk.Fixed()
        self.add(self.layout)

        self.graph = Graph()
        self.layout.put(self.graph, 0, 0)

        gtk.set_application_name("battery-eye")

        self.menu = hildon.AppMenu()
        self.filters = []
        self.filters.append(hildon.GtkRadioButton(gtk.HILDON_SIZE_AUTO, None))
        self.filters[0].set_label('%')
        self.filters[0].set_mode(False)
        self.filters[0].connect("clicked",
                                lambda button: self.graph.changeDataset('hal.battery.charge_level.percentage'))

        self.filters.append(hildon.GtkRadioButton(gtk.HILDON_SIZE_AUTO, self.filters[0]))
        self.filters[1].set_label('mAh')
        self.filters[1].set_mode(False)
        self.filters[1].connect("clicked", lambda button: self.graph.changeDataset('hal.battery.reporting.current'))

        self.filters.append(hildon.GtkRadioButton(gtk.HILDON_SIZE_AUTO, self.filters[1]))
        self.filters[2].set_label('mV')
        self.filters[2].set_mode(False)
        self.filters[2].connect("clicked", lambda button: self.graph.changeDataset('hal.battery.voltage.current'))

#        self.filters.append(hildon.GtkRadioButton(gtk.HILDON_SIZE_AUTO, None))
#        self.filters[3].set_label('1 day')
#        self.filters[3].set_mode(False)
#        self.filters[3].connect("clicked",
#                                lambda button: self.graph.changeVisibleHours(25))

#        self.filters.append(hildon.GtkRadioButton(gtk.HILDON_SIZE_AUTO, self.filters[3]))
#        self.filters[4].set_label('3 days')
#        self.filters[4].set_mode(False)
#        self.filters[4].connect("clicked", lambda button: self.graph.changeVisibleHours(25*3))

#        self.filters.append(hildon.GtkRadioButton(gtk.HILDON_SIZE_AUTO, self.filters[4]))
#        self.filters[5].set_label('7 days')
#        self.filters[5].set_mode(False)
#        self.filters[5].connect("clicked", lambda button: self.graph.changeVisibleHours(25*7))

        for f in self.filters:
            self.menu.add_filter(f)

        self.menu.show_all()
        self.set_app_menu(self.menu)

    def run(self):
        self.show_all()
        self.graph.jump_to(self.graph.drawAreaInfo.canvasSize[0]-1, 0)
        try:
            gtk.main()
        except KeyboardInterrupt:
            print ""
            print "Ctrl-C"

class DrawAreaInfo(object):
    def __init__(self, canvasSize, drawAreaSize, offset, windowSize):
        self.canvasSize = canvasSize
        self.drawAreaSize = drawAreaSize
        self.offset = offset
        self.windowSize = windowSize

class EraseableBuffer(object):
    def __init__(self, pixmap):
        self.pixmap = pixmap
        self.backbuffer = gtk.gdk.Pixmap(None, pixmap.get_size()[0], pixmap.get_size()[1], pixmap.get_depth())
        self.lastDrawParams = None # (x, y, w, h)

    def draw(self, gc, dst, x, y, w, h):
        if w == -1:
            w = self.pixmap.get_size()[0]
        if h == -1:
            h = self.pixmap.get_size()[1]
        self.backbuffer.draw_drawable(gc, dst,
                                      x, y,
                                      0, 0,
                                      w, h)
        dst.draw_drawable(gc, self.pixmap,
                          0, 0,
                          x, y,
                          w, h)
        self.lastDrawParams = (x, y, w, h)
        return gtk.gdk.Rectangle(x, y, w, h)

    def undraw(self, gc, dst):
        if self.lastDrawParams == None:
            return
        x, y, w, h = self.lastDrawParams
        dst.draw_drawable(gc, self.backbuffer,
                          0, 0,
                          x, y,
                          w, h)
        self.lastDrawParams = None
        return gtk.gdk.Rectangle(x, y, w, h)

    def get_size(self):
        return self.pixmap.get_size()

class Graph(hildon.PannableArea):

    def __init__(self):
        hildon.PannableArea.__init__(self)
        self.graphWidth = 800
        self.graphHeight = 424
        self.set_property('mov-mode', hildon.MOVEMENT_MODE_HORIZ)
        self.set_size_request(self.graphWidth, self.graphHeight)
        self.area = gtk.DrawingArea()
        self.add_with_viewport(self.area)
        
        self.area.connect("expose-event", self.redrawGraph)

        self.get_hadjustment().connect("value-changed", self.moveLabels)

        self.bgColor = self.area.get_colormap().alloc_color(0, 0, 0)
        self.dataColor = self.area.get_colormap().alloc_color(0, 255*256, 0)
        self.gridColor = self.area.get_colormap().alloc_color(40*256, 40*256, 40*256)
        self.labelsColor = self.area.get_colormap().alloc_color(120*256, 120*256, 120*256)

        self.dataset = 'hal.battery.charge_level.percentage'
        self.visibleHours = 25

        self.updateData()

    def changeDataset(self, dataset):
        self.dataset = dataset
        self.updateData()

    def changeVisibleHours(self, visibleHours):
        self.visibleHours = visibleHours
        self.updateData()
        self.jump_to(self.drawAreaInfo.canvasSize[0]-1, 0)

    def updateData(self):
        if self.dataset == 'hal.battery.charge_level.percentage':
            yMin = 0
            yMax = 100
            yLabelsInterval = 20
            yLabels = ['0%', '20%', '40%', '60%', '80%', '100%']
        elif self.dataset == 'hal.battery.reporting.current':
            yMin = 0
            yMax = 1320
            yLabelsInterval = 200
            yLabels = ['0mAh', '200mAh', '400mAh', '600mAh', '800mAh', '1000mAh', '1200mAh', '1320mAh']
        elif self.dataset == 'hal.battery.voltage.current':
            yMin = 3600
            yMax = 4200
            yLabelsInterval = 100
            yLabels = ['3600mV', '3700mV', '3800mV', '3900mV', '4000mV', '4100mV', '4200mV']

        rawData = dataStorage.getObservations(self.dataset)

        self.xMax = int(time.time()) + 2*3600

        if len(rawData) < 1:
            self.xMin = self.xMax - self.visibleHours*3600
        else:
            self.xMin = min(rawData[0][0], self.xMax - self.visibleHours*3600)

        areaWidth = max((self.xMax - self.xMin) / 3600 * (self.graphWidth/self.visibleHours), self.graphWidth - 10)

        self.drawAreaInfo = DrawAreaInfo((areaWidth, self.graphHeight),
                                         (areaWidth, self.graphHeight-40),
                                         (0, 14),
                                         (self.graphWidth-10, self.graphHeight-10))

        self.pixmap = gtk.gdk.Pixmap(None, areaWidth, self.graphHeight, 16)
        self.area.set_size_request(areaWidth, self.graphHeight)

        self.data = self.rescaleData(rawData, self.drawAreaInfo, self.xMin, self.xMax, yMin, yMax)
        self.yGrid = self.yGridSegments(self.drawAreaInfo, yMin, yMax, yLabelsInterval)
        self.xGrid = self.xGridSegments(self.drawAreaInfo, self.xMin, self.xMax)
        self.yLabels = self.createYLabels(yLabels)
        self.xLabels = self.createXLabels(self.xMin, self.xMax)
        self.charge = self.chargeInfo()

        gc = self.pixmap.new_gc()

        gc.foreground = self.bgColor
        self.pixmap.draw_rectangle(gc, True,
                                   0, 0,
                                   self.drawAreaInfo.canvasSize[0],
                                   self.drawAreaInfo.canvasSize[1])

        gc.foreground = self.gridColor
        self.pixmap.draw_segments(gc, self.yGrid)
        self.pixmap.draw_segments(gc, self.xGrid)

        gc.foreground = self.dataColor
        self.pixmap.draw_lines(gc, self.data)

        gc.foreground = self.labelsColor
        self.drawYLabels(gc, self.yLabels, self.yGrid)
        self.drawXLabels(gc, self.xLabels)

        self.area.queue_draw()

    def redrawGraph(self, widget, event):
        self.gc = widget.style.fg_gc[gtk.STATE_NORMAL]

        area = event.area

#        print "Redraw (%d, %d)->(%d, %d)" % (area.x, area.y, area.x + area.width, area.y + area.height)

        widget.window.draw_drawable(self.gc, self.pixmap,
                                    area.x, area.y,
                                    area.x, area.y,
                                    area.width, area.height)

    def moveLabels(self, adjustment):
        area = self.drawYLabels(self.pixmap.new_gc(), self.yLabels, self.yGrid)
        self.area.queue_draw_area(area.x, area.y, area.width, area.height)

    def createYLabels(self, labels):
        pContext = self.area.create_pango_context()
        pContext.set_font_description(pango.FontDescription('sans normal 10'))
        labelPixmaps = []
        for label in labels:
            layout = pango.Layout(pContext)
            layout.set_text(label)
            pixmap = gtk.gdk.Pixmap(None, layout.get_pixel_size()[0],
                                          layout.get_pixel_size()[1], 16)
            gc = pixmap.new_gc()
            pixmap.draw_rectangle(gc, True, 0, 0, pixmap.get_size()[0], pixmap.get_size()[1])
            gc.foreground = self.labelsColor
            pixmap.draw_layout(gc, 0, 0, layout)
            labelPixmaps.append(EraseableBuffer(pixmap))
        return labelPixmaps

    def unionRects(self, rects):
        r = None
        for rect in rects:
            if rect != None:
                if r == None:
                    r = rect
                else:
                    r = r.union(rect)
        return r

    def clearYLabels(self, gc, pixmaps):
        invalidated = []
        for pixmap in pixmaps:
            invalidated.append(pixmap.undraw(gc, self.pixmap))
        return self.unionRects(invalidated)

    # Returns a gtk.gdk.Rectangle describing the area that is invalided
    # by draws.
    def drawYLabels(self, gc, pixmaps, yGrid):
        wPos = int(self.get_hadjustment().get_value())
        invalidated = []
        for pixmap, position in zip(pixmaps, yGrid):
            invalidated.append(pixmap.undraw(gc, self.pixmap))
            invalidated.append(pixmap.draw(gc, self.pixmap,
                               self.drawAreaInfo.windowSize[0] - pixmap.get_size()[0] + wPos, position[1]-15,
                               -1, 15))
        return self.unionRects(invalidated)
            
    def yGridSegments(self, areaInfo, yMin, yMax, yInterval):
        y = []
        j = yMin
        while j < yMax:
            y.append(j)
            j += yInterval
        y.append(yMax)
        points = self.rescaleData(zip(range(len(y)), y), areaInfo, 0, len(y), yMin, yMax)
        ret = []
        for trash,y in points:
            ret.append((areaInfo.offset[0], y, areaInfo.drawAreaSize[0]-1, y))
        return ret

    def chargeInfo(self):
        rawData = dataStorage.getObservations(('hal.battery.rechargeable.is_charging',
                                               'hal.battery.rechargeable.is_discharging'))
#        print rawData
        
    def createXLabels(self, xMin, xMax):
        hours = self.getHoursInInterval(xMin, xMax, 1)
        pContext = self.area.create_pango_context()
        pContext.set_font_description(pango.FontDescription('sans normal 10'))
        labelPixmaps = []
        for hour in hours:
            if hour.hour % 4 == 0:
                layout = pango.Layout(pContext)
                if hour.hour == 0:
                    layout.set_text(hour.strftime('%d. %b'))
                else:
                    layout.set_text(hour.strftime('%H:%M'))
                pixmap = gtk.gdk.Pixmap(None, layout.get_pixel_size()[0],
                                              layout.get_pixel_size()[1], 16)
                gc = pixmap.new_gc()
                pixmap.draw_rectangle(gc, True, 0, 0, pixmap.get_size()[0], pixmap.get_size()[1])
                gc.foreground = self.labelsColor
                pixmap.draw_layout(gc, 0, 0, layout)
                labelPixmaps.append(pixmap)
        return labelPixmaps

    def drawXLabels(self, gc, pixmaps):
        hours = self.getHoursInInterval(self.xMin, self.xMax, 1)
        labeledHours = []
        for hour in hours:
            if hour.hour % 4 == 0:
                labeledHours.append(hour)
        y = self.drawAreaInfo.offset[1] + self.drawAreaInfo.drawAreaSize[1] + 1
        for pixmap, hour in zip(pixmaps, labeledHours):
            x = self.rescaleData([(time.mktime(hour.timetuple()), 1)], self.drawAreaInfo, self.xMin, self.xMax, 0, 1)[0][0]
            self.pixmap.draw_drawable(gc, pixmap,
                                      0, 0,
                                      x - int(pixmap.get_size()[0]/2),
                                      y,
                                      -1, -1)

    def xGridSegments(self, areaInfo, xMin, xMax, modulo = 1):
        hours = self.getHoursInInterval(xMin, xMax, modulo)
        points = []
        for hour in hours:
            points.append(time.mktime(hour.timetuple()))
        points = self.rescaleData(zip(points, range(len(points))), areaInfo, xMin, xMax, 0, 1)
        ret = []
        for x,thrash in points:
            ret.append((x, areaInfo.offset[1] + 1, x, areaInfo.offset[1] + areaInfo.drawAreaSize[1]-1))
        return ret

    def rescaleData(self, points, areaInfo, xMin, xMax, yMin, yMax):
        if len(points) == 0:
            return []
        dxOffset = xMin
        dxMax = xMax
        dxDistance = dxMax - dxOffset
        yDistance = yMax - yMin
        if dxDistance <= 0 or yDistance <= 0:
            # Let's not divide by zero
            return []

        def rescalePoint(point):
            x = (areaInfo.drawAreaSize[0]-1) * float(point[0] - dxOffset) / dxDistance
            y = (areaInfo.drawAreaSize[1]-1) * float(point[1] - yMin) / yDistance
            return (int(x) + areaInfo.offset[0], areaInfo.drawAreaSize[1] - int(y) + areaInfo.offset[1])

        return map(rescalePoint, points)
  
    def getHoursInInterval(self, start, end, modulo):
        hour = datetime.timedelta(0, 0, 0, 0, 0, 1)
        
        startDt = self.nextHour(datetime.datetime.fromtimestamp(start, beye.utc).astimezone(beye.local))
        endDt = self.nextHour(datetime.datetime.fromtimestamp(end, beye.utc).astimezone(beye.local))
        
        current = startDt
        ret = []
        while current < endDt:
            if current.hour % modulo == 0:
                ret.append(current)
            current += hour
        return ret

    def nextHour(self, dt):
        deltaBack = datetime.timedelta(0, dt.second, dt.microsecond, 0, dt.minute)
        deltaForward = datetime.timedelta(0, 0, 0, 0, 0, 1)
        return dt + deltaForward - deltaBack

if __name__ == "__main__":
    dataStorage = beye.DataStorage(beye.setupAndGetDbPath(), [beye.DataSourceHal()])
    mainwin = MainWindow()
    mainwin.run()
