import datetime
import re

# A Foursquare API timestamp string.
fs_date_match_str = "(\S+), (\d+) (\S+) (\d+) (\d+):(\d+):(\d+)"
fs_date_match_re = re.compile(fs_date_match_str)

# Internally in the database, we store dates as:
#      "%Y-%m-%d %H:%M:%S"
# Not entirely sure why we don't just convert to UNIX time
int_date_match_str = "(\d+)-(\d+)-(\d+) (\d+):(\d+):(\d+)"
int_date_match_re = re.compile(int_date_match_str)

def str_to_month(month):
        m = { 'Jan' : 1, 'Feb' : 2, 'Mar' : 3, 'Apr' : 4, \
              'May' : 5, 'Jun' : 6, 'Jul' : 7, 'Aug' : 8, \
              'Sep' : 9, 'Oct' : 10, 'Nov' : 11, 'Dec' : 12 }
        if month not in m.keys():
                return 0
        return m[month]

def int_date_to_dt(datestr):
        """ Convert an 'internal' datestring into a datetime object """
        m = int_date_match_re.match(datestr)
        if not m:
                return None
        g = m.groups()
        year = int(g[0])
        month = int(g[1])
        day = int(g[2])
        hours = int(g[3])
        minutes = int(g[4])
        seconds = int(g[5])
        return datetime.datetime(year, month, day, hours, minutes, seconds)

# The Foursquare API returns timestamps as a fixed string:
#
#       "%a, %d %b %y %H:%M:%S +0000"
#
# which can be parsed by strptime(), however it assumes that the user is in an
# English speaking locale. If a user's locale is set to anything different,
# then strptime() fails.  Changing locale back and forth during program
# execution is just annoying and slow, so we parse the date string into a
# DateTime object manually to ensure users in non-English locales can parse the
# timestamp.
def fsq_date_to_dt(datestr):
        m = fs_date_match_re.match(datestr)
        if not m:
                return None
        g = m.groups()
        day = int(g[1])
        month = str_to_month(g[2])
        year = int(g[3]) + 2000
        hour = int(g[4])
        minute = int(g[5])
        second = int(g[6])
        return datetime.datetime(year, month, day, hour, minute, second)

def deltaToFullString(delta):
        """ Takes a datetime.timedelta object and converts it to a pretty string """
        if delta.days == 1:
                return "1 day ago"
        if delta.days > 1:
                return str(delta.days) + " days ago"
        if delta.seconds < 60:
                if delta.seconds == 1:
                        return "1 second ago"
                else:
                        return str(delta.seconds) + " seconds ago"
        if delta.seconds < 3600:
                m = delta.seconds / 60
                if m == 1:
                        return "1 minute ago"
                else:
                        return str(m) + " minutes ago"
        if delta.seconds < 18000:
                mins = delta.seconds / 60
                hrs = mins / 60
                mins = mins % 60
                s = ""
                if hrs == 1:
                        s = "1 hour "
                else:
                        s = str(hrs) + " hours "
                if mins == 1:
                        s += "1 minute ago"
                else:
                        s += str(mins) + " minutes ago"
                return s

        hrs = delta.seconds / 60 / 60
        if hrs == 1:
                return "1 hour ago"
        else:
                return str(hrs) + " hours ago"

def deltaToShortString(delta):
        if delta.days > 0:
                return str(delta.days) + "d ago"
        if delta.seconds < 60:
                return str(delta.seconds) + "s ago"
        if delta.seconds < 3600:
                return str(delta.seconds / 60) + "m ago"
        if delta.seconds < 18000:
                mins = delta.seconds / 60
                hrs = mins / 60
                mins = mins % 60
                return str(hrs) + "h " + str(mins) + "m ago"
        return str(delta.seconds / 60 / 60) + "h ago"

def fsq_date_to_time(fsdatestr):
        """ Convert a Foursquare API timestamp string into an integer
            representing the number of seconds since the epoch.
        """
        dt = fsq_date_to_dt(fsdatestr)
        epoch_dt = datetime.datetime(1970, 1, 1, 0, 0, 0)
        delta = dt - epoch_dt
        return delta.seconds + (delta.days * 24 * 60 * 60)

def int_date_to_time(inttime):
        """ Convert an 'internal' timestamp string to an integer
            representing the number of seconds since the epoch.
        """
        dt = int_date_to_dt(inttime)
        delta = dt - datetime.datetime(1970, 1, 1, 0, 0, 0)
        return delta.seconds + (delta.days * 24 * 60 * 60)

def fsDateStrToDeltaFromNow(fsdatestr):
        return datetime.datetime.utcnow() - fsq_date_to_dt(fsdatestr)

def internalDateStrToDeltaFromNow(strDate):
        return datetime.datetime.utcnow() - int_date_to_dt(strDate)

def calcDateString(strDate, fullWord, formatType):
        s = ""
        if formatType == 0:
                delta = fsDateStrToDeltaFromNow(strDate)
        else:
                delta = internalDateStrToDeltaFromNow(strDate)

        if fullWord:
                s = deltaToFullString(delta)
        else:
                s = deltaToShortString(delta)

        print "calcDateString(\"%s\", %s, %i) = \"%s\"" % (strDate, str(fullWord), formatType, s)
        return s
