/*
babyphone - A baby monitor application on the Nokia N900.
    Copyright (C) 2010  Roman Morawek <maemo@morawek.at>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "settings.h"
#include "ui_settings.h"

#include <QDialog>
#include <QDesktopServices>
#include <QUrl>


// settings keys and default values
#define COMPANY                         "morawek.at"
#define PRODUCT                         "babyphone"
#define PHONE_NUMBER_KEY                "phoneNumber"
#define PHONE_NUMBER_DEFAULT            ""
#define AUDIO_AMPLIFY_KEY               "audio/volume"
#define AUDIO_AMPLIFY_DEFAULT           16
#define AUDIO_TIMER_KEY                 "audio/timer"
#define AUDIO_TIMER_DEFAULT             4
#define CALL_SETUP_TIMER_KEY            "call/setupTimer"
#define CALL_SETUP_TIMER_DEFAULT        30
#define ACTIVATION_DELAY_KEY            "application/activationDelay"
#define ACTIVATION_DELAY_DEFAULT        0
#define RECALL_TIMER_KEY                "call/recallTimer"
#define RECALL_TIMER_DEFAULT            180
#define SWITCH_PROFILE_KEY              "application/switchProfile"
#define SWITCH_PROFILE_DEFAULT          true
#define FIRST_RUN_KEY                   "application/firstRun"
#define SEND_SMS_KEY                    "application/sendSMS"
#define SEND_SMS_DEFAULT                false
#define REJECT_INCOMING_CALLS_KEY       "call/rejectIncoming"
#define REJECT_INCOMING_CALLS_DEFAULT   true
#define DISABLE_GRAPHS_KEY              "view/disableGraphsWhileMonitoring"
#define DISABLE_GRAPHS_DEFAULT          false

// further settings
#define URL_HELP                        "http://babyphone.garage.maemo.org/index.html#sec_settings"


/*!
  The constructor initializes the constant settings and loads the saved values
  from the persistant storage.
*/
Settings::Settings(QObject *parent) :
    QSettings(COMPANY, PRODUCT, parent),
    VERSION("0.2"),
    CALL_HOLD_TIMER(300000),    // 300s maximum call time
    THRESHOLD_VALUE(100),
    VOLUME_COUNTER_MAX(120),    // clipping occurs at this value
    VOLUME_COUNTER_DEC(1),
    MSG_BOX_TIMEOUT(10000),     // 10s until auto-close of message boxes
    REFOCUS_TIMER(2000),        // 2s after the call is finished
    GRAPH_MIN_VALUE(0),         // minimum value that is shown
    GRAPH_MAX_VALUE(130),       // maximum value that is shown
    GRAPH_WIDTH(579),           // static value determined by UI design
    GRAPH_HEIGHT(115),          // static value determined by UI design
    AUDIO_SAMPLE_INTERVAL(800),
    AUDIO_SAMPLE_SUBINTERVAL(16),
    AUDIO_ON_TIMER(5000),
    AUDIO_OFF_TIMER(100)
{
    itsAudioAmplify = value(AUDIO_AMPLIFY_KEY, AUDIO_AMPLIFY_DEFAULT).toInt();
    itsAudioTimer = value(AUDIO_TIMER_KEY, AUDIO_TIMER_DEFAULT).toInt();
    itsPhonenumber = value(PHONE_NUMBER_KEY, PHONE_NUMBER_DEFAULT).toString();
    itsCallSetupTimer = value(CALL_SETUP_TIMER_KEY, CALL_SETUP_TIMER_DEFAULT).toInt();
    itsActivationDelay = value(ACTIVATION_DELAY_KEY, ACTIVATION_DELAY_DEFAULT).toInt();
    itsRecallTimer = value(RECALL_TIMER_KEY, RECALL_TIMER_DEFAULT).toInt();
    itsSwitchProfile = value(SWITCH_PROFILE_KEY, SWITCH_PROFILE_DEFAULT).toBool();
    itsSendSMS = value(SEND_SMS_KEY, SEND_SMS_DEFAULT).toBool();
    itsHandleIncomingCalls = value(REJECT_INCOMING_CALLS_KEY, REJECT_INCOMING_CALLS_DEFAULT).toBool();
    itsDisableGraphs = value(DISABLE_GRAPHS_KEY, DISABLE_GRAPHS_DEFAULT).toBool();
    itsFirstRun = value(FIRST_RUN_KEY, true).toBool();
}


/*!
  Save saves all current setting values to the persistant storage.
*/
void Settings::Save()
{
    setValue(FIRST_RUN_KEY, false);
    setValue(AUDIO_AMPLIFY_KEY, itsAudioAmplify);
    setValue(AUDIO_TIMER_KEY, itsAudioTimer);
    setValue(PHONE_NUMBER_KEY, itsPhonenumber);
    setValue(CALL_SETUP_TIMER_KEY, itsCallSetupTimer);
    setValue(ACTIVATION_DELAY_KEY, itsActivationDelay);
    setValue(RECALL_TIMER_KEY, itsRecallTimer);
    setValue(SWITCH_PROFILE_KEY, itsSwitchProfile);
    setValue(SEND_SMS_KEY, itsSendSMS);
    setValue(REJECT_INCOMING_CALLS_KEY, itsHandleIncomingCalls);
    setValue(DISABLE_GRAPHS_KEY, itsDisableGraphs);
}


/*!
  ShowDialog opens the settings user interface dialog and stores its values if
  demanded.
*/
void Settings::ShowDialog()
{
    // setup settings dialog
    Ui::Dialog *ui = new Ui::Dialog();
    QDialog *dialog = new QDialog(static_cast<QWidget*>(this->parent()));
    ui->setupUi(dialog);
    ui->spinBox_volume->setValue(itsAudioAmplify);
    ui->spinBox_timeWeight->setValue(itsAudioTimer);
    ui->spinBox_callSetupTimer->setValue(itsCallSetupTimer);
    ui->spinBox_activationDelay->setValue(itsActivationDelay);
    ui->spinBox_recallTimer->setValue(itsRecallTimer);
    ui->checkBox_switchProfile->setChecked(itsSwitchProfile);
    ui->checkBox_sendSMS->setChecked(itsSendSMS);
    ui->checkBox_rejectIncomingCalls->setChecked(itsHandleIncomingCalls);
    ui->checkBox_disableGraphs->setChecked(itsDisableGraphs);
    connect(ui->buttonBox, SIGNAL(helpRequested()), this, SLOT(ShowHelp()));

    // show dialog
    dialog->setWindowModality(Qt::WindowModal);
    if (dialog->exec() == QDialog::Accepted)
    {
        // store new settings
        itsAudioAmplify = ui->spinBox_volume->value();
        itsAudioTimer = ui->spinBox_timeWeight->value();
        itsCallSetupTimer = ui->spinBox_callSetupTimer->value();
        itsActivationDelay = ui->spinBox_activationDelay->value();
        itsRecallTimer = ui->spinBox_recallTimer->value();
        itsSwitchProfile = ui->checkBox_switchProfile->checkState();
        itsSendSMS = ui->checkBox_sendSMS->checkState();
        itsHandleIncomingCalls = ui->checkBox_rejectIncomingCalls->checkState();
        itsDisableGraphs = ui->checkBox_disableGraphs->checkState();
    }

    // clean up
    delete ui;
    delete dialog;
}


/*!
  ShowHelp launches the online help.
*/
void Settings::ShowHelp()
{
    QDesktopServices::openUrl(QUrl(URL_HELP));
}
