/*
babyphone - A baby monitor application on the Nokia N900.
    Copyright (C) 2010  Roman Morawek <maemo@morawek.at>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QTimer>
#include <QDebug>
#include <QMessageBox>


MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    // load settings
    itsSettings = new Settings();

    // setup audio monitor
    itsAudioMonitor = new AudioMonitor(itsSettings, this);
    connect(itsAudioMonitor, SIGNAL(update(int, int)), this, SLOT(refreshAudioData(int, int)));
    itsMonitorActive = false;
    itsNotificationPending = false;

    // setup phone call handler
    itsUserNotifier = new UserNotifier(itsSettings, this);
    connect(itsUserNotifier, SIGNAL(notifyFinished()), this, SLOT(notifyFinished()));

    // setup GUI
    ui->setupUi(this);

    // setup graphs
    itsAudioLevelGraphics = new AudioLevelGraphicsScene(ui->graphicsViewLevel, itsSettings, this);
    itsAudioCounterGraphics = new AudioLevelGraphicsScene(ui->graphicsViewCounter, itsSettings, this);

    // setup default UI values
    ui->lineEdit_phone->setText(itsSettings->itsPhonenumber);
    ui->spinBox_volume->setValue(itsSettings->itsAudioAmplify);
    ui->spinBox_timeWeight->setValue(itsSettings->itsAudioTimer);
    ui->spinBox_recallTimer->setValue(itsSettings->itsRecallTimer);

    // start audio capturing
    startAudio();
}


MainWindow::~MainWindow()
{
    // save settings
    itsSettings->Save();

    delete ui;
}


void MainWindow::changeEvent(QEvent *e)
{
    QMainWindow::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}


void MainWindow::on_spinBox_volume_valueChanged()
{
    // update settings
    itsSettings->itsAudioAmplify = ui->spinBox_volume->value();
}


void MainWindow::on_spinBox_timeWeight_valueChanged()
{
    // update settings
    itsSettings->itsAudioTimer = ui->spinBox_timeWeight->value();
}


void MainWindow::on_spinBox_recallTimer_valueChanged()
{
    // update settings
    itsSettings->itsRecallTimer = ui->spinBox_recallTimer->value();
}


void MainWindow::on_lineEdit_phone_textChanged()
{
    // update settings
    itsSettings->itsPhonenumber = ui->lineEdit_phone->text();
}


void MainWindow::on_pushButton_clicked()
{
    if (itsMonitorActive) {
        itsMonitorActive = false;

        // update GUI
        ui->pushButton->setText(tr("Inactive - Click to start"));
        ui->tabSettings->setEnabled(true);
    }
    else {
        itsMonitorActive = true;

        // update GUI
        QString labelText = tr("Active on ");
        labelText += itsSettings->itsPhonenumber;
        labelText += tr(" - Click to stop");
        ui->pushButton->setText(labelText);
        ui->tabSettings->setEnabled(false);
    }
}


void MainWindow::notifyFinished()
{
    // notifcation finished
    itsNotificationPending = false;

    // update GUI
    ui->pushButton->setEnabled(true);

    // reset audio monitor warning
    itsAudioMonitor->itsCounter = 0;

    // start monitor after timeout
    QTimer::singleShot(itsSettings->itsRecallTimer*1000, this, SLOT(startAudio()));
}


void MainWindow::startAudio()
{
    itsAudioMonitor->start();

    // setup safety timer
    QTimer::singleShot(itsSettings->AUDIO_ON_TIMER, this, SLOT(stopAudio()));
}


void MainWindow::stopAudio()
{
    // suspend audio monitoring
    itsAudioMonitor->stop();

    // in case of notification, we get an event on its end
    // otherwise we must restart automatically via a timer
    if (!itsNotificationPending)
        QTimer::singleShot(itsSettings->AUDIO_OFF_TIMER, this, SLOT(startAudio()));
}


void MainWindow::refreshAudioData(int counter, int value)
{
    // update GUI
    itsAudioLevelGraphics->AddValue(value);
    itsAudioCounterGraphics->AddValue(counter);

    // check for noise
    if (itsMonitorActive && (counter > itsSettings->THRESHOLD_VALUE)) {
        qDebug() << tr("Audio threshold reached. Notifying user.");

        // store event
        itsNotificationPending = true;

        // update GUI
        ui->pushButton->setEnabled(false);

        // stop monitoring
        stopAudio();

        // notify user
        if (itsUserNotifier->Notify() == false)
        {
            // the notify command yielded an error
            QMessageBox::critical(this, tr("Babyphone"),
                  tr("Could not establish phone call. Please check the phone number settings."));

            // restart audio capturing
            notifyFinished();
        }
    }
}
