/*
babyphone - A baby monitor application on the Nokia N900.
    Copyright (C) 2010  Roman Morawek <maemo@morawek.at>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
#include "audiomonitor.h"

#include <stdlib.h>
#include <cmath>

#include <QDebug>
#include <QAudioDeviceInfo>
#include <QAudioInput>
#include <QTimer>


AudioMonitor::AudioMonitor(Settings *settings, QObject *parent)
    :QIODevice(parent), itsSettings(settings)
{
    // open IODevice
    open(QIODevice::WriteOnly);

    // determine suitable audio format

    // this is what we want
    itsAudioFormat.setFrequency(8000);
    itsAudioFormat.setChannels(1);
    itsAudioFormat.setSampleSize(16);
    itsAudioFormat.setSampleType(QAudioFormat::SignedInt);
    itsAudioFormat.setByteOrder(QAudioFormat::LittleEndian);
    itsAudioFormat.setCodec("audio/pcm");

    // this is what we get
    QAudioDeviceInfo info(QAudioDeviceInfo::defaultInputDevice());
    if (!info.isFormatSupported(itsAudioFormat)) {
        itsAudioFormat = info.nearestFormat(itsAudioFormat);
        qWarning() << tr("Could not get desired audio format. Nearest available format has")
                   << tr("frequency") << itsAudioFormat.frequency()
                   << tr("sample size") << itsAudioFormat.sampleSize();
    }
    if (itsAudioFormat.sampleSize() != 16) {
        qCritical() << tr("Audio device doesn't support needed format, exiting");
        exit(1);
        return;
    }

    // create device
    itsDevice = new QAudioInput(itsAudioFormat, this);
    itsDevice->setNotifyInterval(itsSettings->AUDIO_SAMPLE_INTERVAL);
    itsDevice->start(this);
    itsDevice->suspend();
    itsActive = false;

    // reset counter
    itsCounter = 0;

    //connect(itsDevice, SIGNAL(notify()), SLOT(status()));
}


AudioMonitor::~AudioMonitor()
{
    // close IODevice
    close();

    // close AudioInput
    itsDevice->stop();
}


void AudioMonitor::start()
{
    // start capturing
    itsActive = true;
    itsDevice->resume();
}


void AudioMonitor::stop()
{
    // stop capturing
    itsActive = false;
    itsDevice->suspend();
}


qint64 AudioMonitor::readData(char *data, qint64 maxlen)
{
    Q_UNUSED(data)
    Q_UNUSED(maxlen)

    return 0;
}


qint64 AudioMonitor::writeData(const char *data, qint64 len)
{
    quint32 curEnergy = 0;
    qint16 max = 0;

    // sample format is S16LE, only!
    const qint16 *buffer = (qint16*)data;

    // derive energy
    for (int i = 0; i < len/2; ++i) {
        // store maximum
        if (*buffer > max)
            max = *buffer;

        // subinterval expired
        if ((i % itsSettings->AUDIO_SAMPLE_SUBINTERVAL) == 0) {
            // add current maximum and reset it
            curEnergy += max;
            //qDebug() << "add" << max << "new value:" << curEnergy;
            max = 0;
        }

        // process next sample
        buffer++;
    }

    // scale volume
    int volume = itsSettings->itsAudioAmplify *
                 log(curEnergy*itsSettings->AUDIO_SAMPLE_SUBINTERVAL/len);

    // update timer counter
    if (volume > itsSettings->THRESHOLD_VALUE) {
        // increment counter
        itsCounter += itsSettings->itsAudioTimer;

        // check for overflow
        if (itsCounter > itsSettings->VOLUME_COUNTER_MAX)
            itsCounter = itsSettings->VOLUME_COUNTER_MAX;    // overflow, clip
    }
    else {
        // decrement counter
        itsCounter -= itsSettings->VOLUME_COUNTER_DEC;

        // check for underflow
        if (itsCounter < 0)
            itsCounter = 0;
    }

    emit update(itsCounter, volume);

    return len;
}


void AudioMonitor::status()
{
    qWarning() << "bytesReady = " << itsDevice->bytesReady()
               << " bytes, elapsedUSecs = " << itsDevice->elapsedUSecs()
               << ", processedUSecs = " << itsDevice->processedUSecs();
}
