#!/usr/bin/env python

import errno
import exceptions
import locale
import sys
import struct

from PyQt4.QtCore import QCoreApplication, QDir, QFile, QIODevice, QString, \
                         QStringList, QTextCodec
from ACE_encoding import ACEEncoding
from ACE_utils    import bytesToInt, intToBytes

dictionaryFile = QDir.homePath() + '/.osso/dictionaries/.personal.dictionary'
positions = {
    'header.DictionaryCount': [3,1],
    'header.FileSize': [4,1],
    'header.PaddingOffset': [6,2],
    'dictionary.Language': [0,2],
    'dictionary.StartOffset': [2,2],
    'dictionary.EntryCount': [4,2]
}
sizes = {
    'header': 8,
    'dictHeader': 8
}

class ACEFileReadException(Exception): #{{{1
    def __init__(self, msg, language, entry = None):
        self.msg = msg
        self.language = language
        self.entry = entry

    def __str__(self):
        msg  = 'Error reading dictionary\n\n'
        msg += '%s for language ID %d' % (self.msg, self.language)

        if self.entry != None:
            msg += '\nLast entry "%s" removed' % self.entry

        return msg

class ACEFile(): #{{{1
    _dicts = {}

    def __init__(self): #{{{2
        self._encoding = ACEEncoding()

    def _calculateDataSize(self): #{{{2
        size = sizes['header']
        for dictLanguage in self._dicts.keys():
            size += self._calculateDictionarySize(dictLanguage) + sizes['dictHeader']

        return size

    def _calculateDictionarySize(self, dictLanguage): #{{{2
        size = 0

        for dictEntry in self._dicts[dictLanguage]:
            size += len(self._encoding.fromQString(dictEntry)) + 1

        return size

    def _readBufferInt(self, buffer, pos): #{{{2
        posStart = positions[pos][0]
        posEnd   = posStart + positions[pos][1]

        return bytesToInt(buffer[posStart:posEnd])

    def _readDictionary(self, file, paddingOffset = None): #{{{2
        dictionaryData = file.read(sizes['dictHeader'])

        language = self._readBufferInt(dictionaryData, 'dictionary.Language')
        offset = self._readBufferInt(dictionaryData, 'dictionary.StartOffset')
        entryCount = self._readBufferInt(dictionaryData, 'dictionary.EntryCount')

        if paddingOffset == None:
            dictionaryDataNext = file.read(sizes['dictHeader'])
            offsetNext = self._readBufferInt(dictionaryDataNext, 'dictionary.StartOffset')
        else:
            offsetNext = paddingOffset

        self._dicts[language] = QStringList()
        i = 0

        if not file.seek(offset):
            raise IOError(errno.EIO, "Error seeking in file", dictionaryFile)

        while i < entryCount:
            self._dicts[language].append(self._readEntry(file))
            i += 1

        if file.pos() < offsetNext:
            raise ACEFileReadException('Dictionary read short', language)
        elif file.pos() > offsetNext:
            lastEntry = self._dicts[language].takeLast()

            if lastEntry.right(1) == self._encoding.toQString('\x00'):
                while len(lastEntry) > 0 and lastEntry.right(1) == self._encoding.toQString('\x00'):
                    lastEntry.chop(1)

                raise ACEFileReadException('Dictionary truncated', language, lastEntry)
            else:
                self._dicts[language].append(lastEntry)
                raise ACEFileReadException('Dictionary truncated', language)

    def _readEntry(self, file): #{{{2
        entryLength = bytesToInt(file.read(1))
        entryString = self._encoding.toQString(file.read(entryLength))

        return entryString

    def _readHeader(self, file): #{{{2
        if not file.seek(0):
            raise IOError(errno.EIO, "Error seeking in file", dictionaryFile)
        
        self._header = file.read(sizes['header'])

        paddingOffset = self._readBufferInt(self._header, 'header.PaddingOffset')
        dictionaryCount = self._readBufferInt(self._header, 'header.DictionaryCount')

        return (paddingOffset, dictionaryCount)

    def _writeBufferInt(self, buffer, pos, val): #{{{2
        posStart = positions[pos][0]
        posEnd = posStart + positions[pos][1]

        newBuffer = buffer[0:posStart] + intToBytes(val, positions[pos][1]) + buffer[posEnd:]

        return newBuffer

    def _writeEntry(self, file, entryString): #{{{2
        encEntryString = self._encoding.fromQString(entryString)
        file.write(intToBytes(len(encEntryString), 1))
        file.write(encEntryString)

    def _writeHeader(self, file, dictOrder): #{{{2
        dataSize = self._calculateDataSize()
        fileSizeK = (1023 + dataSize) / 1024
        fileSize = fileSizeK * 1024

        header = self._writeBufferInt(self._header, 'header.DictionaryCount', len(dictOrder))
        header = self._writeBufferInt(header, 'header.FileSize', fileSizeK * 4)
        header = self._writeBufferInt(header, 'header.PaddingOffset', dataSize)

        file.write(header)

        dataOffset = sizes['header'] + sizes['dictHeader'] * len(dictOrder)

        for dictLanguage in dictOrder:
            dictSize = self._calculateDictionarySize(dictLanguage)

            dictHeader = '\x00' * sizes['dictHeader']
            dictHeader = self._writeBufferInt(dictHeader, 'dictionary.Language', dictLanguage)
            dictHeader = self._writeBufferInt(dictHeader, 'dictionary.StartOffset', dataOffset)
            dictHeader = self._writeBufferInt(dictHeader, 'dictionary.EntryCount', len(self._dicts[dictLanguage]))

            file.write(dictHeader)

            dataOffset += dictSize

        return fileSize

    def _writePadding(self, file, fileSize): #{{{2
        paddingLength = fileSize - file.pos()
        file.write('\x00' * paddingLength)

    def read(self): #{{{2
        self._dicts = {}
        file = QFile(dictionaryFile)

        if not file.exists():
            raise IOError(errno.ENOENT, "File not found", dictionaryFile)

        if not file.open(QIODevice.ReadOnly):
            raise IOError(errno.EIO, "Cannot open file", dictionaryFile)

        (paddingOffset, dictionaryCount) = self._readHeader(file)
        i = 1

        while i <= dictionaryCount:
            if not file.seek(i * 8):
                raise IOError(errno.EIO, "Error seeking in file", dictionaryFile)

            if i == dictionaryCount:
                self._readDictionary(file, paddingOffset)
            else:
                self._readDictionary(file)
            i += 1

        file.close()

    def write(self): #{{{2
        file = QFile("%s.tmp" % dictionaryFile)

        if not file.open(QIODevice.WriteOnly | QIODevice.Truncate):
            raise IOError(errno.EIO, "Cannot open file", "%s.tmp" % dictionaryFile)

        dictOrder = self._dicts.keys()

        fileSize = self._writeHeader(file, dictOrder)

        for dictLanguage in dictOrder:
            for entryString in self._dicts[dictLanguage]:
                self._writeEntry(file, entryString)

        self._writePadding(file, fileSize)

        file.close()

        if QFile.exists("%s.bak" % dictionaryFile):
            if not QFile.remove("%s.bak" % dictionaryFile):
                raise IOError(errno.EIO, "Error removing old backup file", "%s.bak" % dictionaryFile)

        if not QFile.rename(dictionaryFile, "%s.bak" % dictionaryFile):
            raise IOError(errno.EIO, "Error writing backup file", "%s.bak" % dictionaryFile)

        if not QFile.rename("%s.tmp" % dictionaryFile, dictionaryFile):
            raise IOError(errno.EIO, "Error replacing dictionary", dictionaryFile)

    def getLanguages(self): #{{{2
        return self._dicts.keys()

    def getDict(self, dictLanguage): #{{{2
        return QStringList(self._dicts[dictLanguage])

    def setDict(self, dictLanguage, dictEntries): #{{{2
        self._dicts[dictLanguage] = QStringList(dictEntries)

    def deleteDict(self, dictLanguage): #{{{2
        del self._dicts[dictLanguage]

    def getEncoding(self): #{{{2
        return self._encoding.getEncoding()

    def setEncoding(self, enc): #{{{2
        if enc != self.getEncoding():
            self._encoding.setEncoding(enc)

            try:
                self.read()
            except:
                self._dicts = {}
                raise

if __name__ == '__main__': #{{{1
    app = QCoreApplication(sys.argv)
    dict = ACEFile()
    dict.write()
