#!/usr/bin/env python

import errno
import exceptions
import locale
import sys
import struct

from PyQt4.QtCore import QCoreApplication, QDir, QFile, QIODevice, QString, \
                         QStringList, QTextCodec

dictionaryFile = QDir.homePath() + '/.osso/dictionaries/.personal.dictionary'
positions = {
    'header.DictionaryCount': [3,1],
    'header.FileSize': [4,1],
    'header.PaddingOffset': [6,2],
    'dictionary.Language': [0,2],
    'dictionary.StartOffset': [2,2],
    'dictionary.EntryCount': [4,2]
}
sizes = {
    'header': 8,
    'dictHeader': 8
}
codec_overrides = {
    'ru_RU': 'cp1251',
}

class ACEFileException(Exception): #{{{1
    def __init__(self, msgsource, errmsg):
        self.msgsource = msgsource
        self.errmsg = errmsg

    def __str__(self):
        return repr("%s: %s" % (self.msgsource, self.errmsg))

class ACEFileReadException(ACEFileException): #{{{1
    def __init__(self, errmsg):
        ACEFileException.__init__(self, 'Error reading dictionary', errmsg)

class ACEFileWriteException(ACEFileException): #{{{1
    def __init__(self, errmsg):
        ACEFileException.__init__(self, 'Error writing dictionary', errmsg)

class ACEFileDecodeException(ACEFileException): #{{{1
    def __init__(self):
        ACEFileException.__init__(self, 'Error decoding string', 'Encoding tried was %s' % locale.getdefaultlocale()[1])

class ACEFile(): #{{{1
    _dicts = {}

    def __init__(self): #{{{2
        cur_locale = locale.getdefaultlocale()

        if cur_locale[0] in codec_overrides:
            self._encoding = codec_overrides[cur_locale[0]]
        else:
            self._encoding = cur_locale[1]

        self.read()

        return
    
    def _bytesToInt(self, bytes): #{{{2
        return sum(ord(c) << (i * 8) for i, c in enumerate(bytes[::-1]))

    def _calculateDataSize(self): #{{{2
        size = sizes['header']
        for dictLanguage in self._dicts.keys():
            size += self._calculateDictionarySize(dictLanguage) + sizes['dictHeader']

        return size

    def _calculateDictionarySize(self, dictLanguage): #{{{2
        size = 0

        for dictEntry in self._dicts[dictLanguage]:
            size += len(self._fromQString(dictEntry)) + 1

        return size

    def _fromQString(self, strng): #{{{2
        return unicode(strng.toUtf8(), 'utf-8').encode(self._encoding)

    def _toQString(self, strng): #{{{2
        try:
            qStrng = unicode(strng, self._encoding)
        except Exception, e:
            raise ACEFileDecodeException

        return qStrng

    def _intToBytes(self, val, count): #{{{2
        hex = "%%0%dx" % (2 * count) % val
        bitstream = ''

        for i in xrange(0, count * 2, 2):
            bitstream += chr(int(hex[i:i+2], 16))

        return bitstream

    def _readBufferInt(self, buffer, pos): #{{{2
        posStart = positions[pos][0]
        posEnd   = posStart + positions[pos][1]

        return self._bytesToInt(buffer[posStart:posEnd])

    def _readDictionary(self, file): #{{{2
        dictionaryData = file.read(sizes['dictHeader'])

        language = self._readBufferInt(dictionaryData, 'dictionary.Language')
        offset = self._readBufferInt(dictionaryData, 'dictionary.StartOffset')
        entryCount = self._readBufferInt(dictionaryData, 'dictionary.EntryCount')

        self._dicts[language] = QStringList()
        i = 0

        if not file.seek(offset):
            raise IOError(errno.EIO, "Error seeking in file", dictionaryFile)

        while i < entryCount:
            self._dicts[language].append(self._readEntry(file))
            i += 1

    def _readEntry(self, file): #{{{2
        entryLength = self._bytesToInt(file.read(1))
        entryString = self._toQString(file.read(entryLength))

        return entryString

    def _readHeader(self, file): #{{{2
        if not file.seek(0):
            raise IOError(errno.EIO, "Error seeking in file", dictionaryFile)
        
        self._header = file.read(sizes['header'])

        paddingOffset = self._readBufferInt(self._header, 'header.PaddingOffset')
        dictionaryCount = self._readBufferInt(self._header, 'header.DictionaryCount')

        return (paddingOffset, dictionaryCount)

    def _writeBufferInt(self, buffer, pos, val): #{{{2
        posStart = positions[pos][0]
        posEnd = posStart + positions[pos][1]

        newBuffer = buffer[0:posStart] + self._intToBytes(val, positions[pos][1]) + buffer[posEnd:]

        return newBuffer

    def _writeEntry(self, file, entryString): #{{{2
        encEntryString = self._fromQString(entryString)
        file.write(self._intToBytes(len(encEntryString), 1))
        file.write(encEntryString)

    def _writeHeader(self, file, dictOrder): #{{{2
        dataSize = self._calculateDataSize()
        fileSizeK = (1023 + dataSize) / 1024
        fileSize = fileSizeK * 1024

        header = self._writeBufferInt(self._header, 'header.DictionaryCount', len(dictOrder))
        header = self._writeBufferInt(header, 'header.FileSize', fileSizeK * 4)
        header = self._writeBufferInt(header, 'header.PaddingOffset', dataSize)

        file.write(header)

        dataOffset = sizes['header'] + sizes['dictHeader'] * len(dictOrder)

        for dictLanguage in dictOrder:
            dictSize = self._calculateDictionarySize(dictLanguage)

            dictHeader = '\x00' * sizes['dictHeader']
            dictHeader = self._writeBufferInt(dictHeader, 'dictionary.Language', dictLanguage)
            dictHeader = self._writeBufferInt(dictHeader, 'dictionary.StartOffset', dataOffset)
            dictHeader = self._writeBufferInt(dictHeader, 'dictionary.EntryCount', len(self._dicts[dictLanguage]))

            file.write(dictHeader)

            dataOffset += dictSize

        return fileSize

    def _writePadding(self, file, fileSize): #{{{2
        paddingLength = fileSize - file.pos()
        file.write('\x00' * paddingLength)

    def read(self): #{{{2
        file = QFile(dictionaryFile)

        if not file.exists():
            raise IOError(errno.ENOENT, "File not found", dictionaryFile)

        if not file.open(QIODevice.ReadOnly):
            raise IOError(errno.EIO, "Cannot open file", dictionaryFile)

        (paddingOffset, dictionaryCount) = self._readHeader(file)
        i = 1

        while i <= dictionaryCount:
            if not file.seek(i * 8):
                raise IOError(errno.EIO, "Error seeking in file", dictionaryFile)

            self._readDictionary(file)
            i += 1

        if file.pos() != paddingOffset:
            raise ACEFileReadException('Error reading dictionary entries')

        file.close()

    def write(self): #{{{2
        file = QFile("%s.tmp" % dictionaryFile)

        if not file.open(QIODevice.WriteOnly | QIODevice.Truncate):
            raise IOError(errno.EIO, "Cannot open file", "%s.tmp" % dictionaryFile)

        dictOrder = self._dicts.keys()

        fileSize = self._writeHeader(file, dictOrder)

        for dictLanguage in dictOrder:
            for entryString in self._dicts[dictLanguage]:
                self._writeEntry(file, entryString)

        self._writePadding(file, fileSize)

        file.close()

        if QFile.exists("%s.bak" % dictionaryFile):
            if not QFile.remove("%s.bak" % dictionaryFile):
                raise IOError(errno.EIO, "Error removing old backup file", "%s.bak" % dictionaryFile)

        if not QFile.rename(dictionaryFile, "%s.bak" % dictionaryFile):
            raise IOError(errno.EIO, "Error writing backup file", "%s.bak" % dictionaryFile)

        if not QFile.rename("%s.tmp" % dictionaryFile, dictionaryFile):
            raise IOError(errno.EIO, "Error replacing dictionary", dictionaryFile)

    def getLanguages(self): #{{{2
        return self._dicts.keys()

    def getDict(self, dictLanguage): #{{{2
        return QStringList(self._dicts[dictLanguage])

    def setDict(self, dictLanguage, dictEntries): #{{{2
        self._dicts[dictLanguage] = QStringList(dictEntries)

    def deleteDict(self, dictLanguage): #{{{2
        del self._dicts[dictLanguage]

if __name__ == '__main__': #{{{1
    app = QCoreApplication(sys.argv)
    dict = ACEFile()
    dict.write()
