#!/usr/bin/env python
# -*- coding: utf-8 -*-

from subprocess    import call

from PyQt4.QtCore  import Qt, QObject, QSettings, QString, QStringList, \
                          QVariant, SIGNAL, SLOT
from PyQt4.QtGui   import QAbstractItemView, QComboBox, QDialog, \
                          QDialogButtonBox, QHBoxLayout, QLabel, QLineEdit, \
                          QListWidget, QMenuBar, QMessageBox, QPushButton, \
                          QStringListModel, QVBoxLayout, QWidget
from ACE_file      import ACEFile;

version = '0.0.7'

langLookup = {
    3: 'English (United Kingdom)',
    4: 'Français (France)',
    5: 'Deutsch (Deutschland)',
    6: 'Español (España)',
    8: 'Português (Portugal)',
    9: 'Svenska (Sverige)',
    10: 'Suomi (Suomi)',
    11: 'Norsk (Norge)',
    12: 'Dansk (Danmark)',
    13: 'Nederlands (Nederland)',
    14: 'Ελληνικά (Ελλάδα)',
    22: 'Polski (Polska)',
    23: 'Čeština (Česká republika)',
    39: 'Русский (Россия)',
    46: 'English (United States of America)',
    47: 'Italiano (Italia)',
    51: 'Español (América Latina)',
    52: 'Français (Québec)',
}

delModes = QStringList(('Normal', 'Confirm', 'OnClick'))

class ACEAboutWindow(QDialog): #{{{1
    def __init__(self, parent): #{{{2
        QDialog.__init__(self, parent)

        self.setWindowTitle('About')

        aboutText  = QString('<h2>Auto-Complete Editor v%1</h2>').arg(version)

        aboutText += "<p>An editor for the user's auto-complete dictionary</p>"
        aboutText += QString('<p>Written by %1<br />').arg('Robin Hill <a href="mailto:maemo@robinhill.me.uk">&lt;maemo@robinhill.me.uk&gt;</a>')
        aboutText += QString('Please email or post to %1 for support</p>').arg('<a href="http://talk.maemo.org/">t.m.o</a>')

        txt = QLabel()
        txt.setOpenExternalLinks(True)
        txt.setTextFormat(Qt.RichText)
        txt.setText(aboutText)

        lyt = QHBoxLayout(self)
        lyt.addWidget(txt)

class ACEConfigWindow(QDialog): #{{{1
    def __init__(self, parent): #{{{2
        QDialog.__init__(self, parent)

        self._parent = parent
        self.setWindowTitle('Settings')

        lyt = QVBoxLayout(self)

        if parent.maemo:
            from PyQt4.QtMaemo5 import QMaemo5ListPickSelector, QMaemo5ValueButton

            lst = QStringListModel(delModes)

            sel = QMaemo5ListPickSelector()
            sel.setModel(lst)
            sel.setCurrentIndex(delModes.indexOf(parent.delMode))

            btn = QMaemo5ValueButton('Delete mode')
            btn.setPickSelector(sel)

            lyt.addWidget(btn)

            QObject.connect(sel, SIGNAL('selected(QString)'), self.changeDeleteMode)

            self._lst = lst
            self._sel = sel
        else:
            boxLyt = QHBoxLayout()

            lbl = QLabel('Delete mode:')

            btn = QComboBox()
            btn.addItems(delModes)
            btn.setCurrentIndex(delModes.indexOf(parent.delMode))

            boxLyt.addWidget(lbl)
            boxLyt.addWidget(btn)
            lyt.addLayout(boxLyt)

            QObject.connect(btn, SIGNAL('currentIndexChanged(int)'), self.changeDeleteModeIndex)

        btns = QDialogButtonBox(QDialogButtonBox.Ok|QDialogButtonBox.Cancel)
        QObject.connect(btns, SIGNAL('accepted()'), self, SLOT('accept()'))
        QObject.connect(btns, SIGNAL('rejected()'), self, SLOT('reject()'))
        lyt.addWidget(btns)

    def changeDeleteModeIndex(self, modeIndex): #{{{1
        mode = delModes[modeIndex]

        self.changeDeleteMode(mode)

    def changeDeleteMode(self, mode): #{{{1
        if mode != self._parent.newDelMode:
            self._parent.newDelMode = mode

class ACEMainWindow(QWidget): #{{{1
    maemo = False
    _dict  = None
    _dicts = {}

    def __init__(self): #{{{2
        QWidget.__init__(self, None)

        try:
            from PyQt4.QtMaemo5 import QMaemo5InformationBox
            self.maemo = True
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)
        except:
            pass

        # Set to the N900 screen size to aid testing
        self.setMinimumSize(800, 480)
        self.setWindowTitle('Auto-Complete Editor')

        self._settings = QSettings('Auto-Complete Editor', 'Auto-Complete Editor')
        self.delMode = self._settings.value('DeleteMode').toString()
        if self.delMode == '':
            self.delMode = 'Normal'

        self.newDelMode = self.delMode

        QHBoxLayout(self)

    def _getLangName(self, langId): #{{{2
        if langId in langLookup:
            return QString.fromUtf8(langLookup[langId])
        else:
            QMessageBox.information(self, 'Unknown dictionary', QString('An unknown dictionary ID has been found.\nPlease contact\n\tRobin Hill <maemo@robinhill.me.uk>\nwith:\n\t- the ID: %1\n\t- the dictionary language you have selected (from Settings/Text input)').arg(langId))
            return 'Unknown'

    def about(self): #{{{2
        aboutWin = ACEAboutWindow(self)
        aboutWin.exec_()

    def config(self): #{{{2
        configWin = ACEConfigWindow(self)
        if configWin.exec_() == QDialog.Rejected:
            self.newDelMode = self.delMode

        if self.newDelMode != self.delMode:
            if self.delMode == 'OnClick':
                self._delBtn.setEnabled(True)
                QObject.disconnect(self._listView, SIGNAL('itemClicked(QListWidgetItem *)'), self.deleteEntry)

            if self.newDelMode == 'OnClick':
                self._delBtn.setEnabled(False)
                QObject.connect(self._listView, SIGNAL('itemClicked(QListWidgetItem *)'), self.deleteEntry)

            self.delMode = self.newDelMode
            self._settings.setValue('DeleteMode', self.delMode)

    def addEntry(self, checked): #{{{2
        dlg = QDialog(self)
        dlg.setWindowTitle('Enter new value')

        txtBox = QLineEdit()
        txtBox.setInputMethodHints(Qt.ImhNoAutoUppercase|Qt.ImhNoPredictiveText)

        btnBox = QDialogButtonBox(QDialogButtonBox.Ok|QDialogButtonBox.Cancel)
        QObject.connect(btnBox, SIGNAL('accepted()'), dlg, SLOT('accept()'))
        QObject.connect(btnBox, SIGNAL('rejected()'), dlg, SLOT('reject()'))

        lyt = QHBoxLayout(dlg)
        lyt.addWidget(txtBox)
        lyt.addWidget(btnBox)

        ok = dlg.exec_()
        newText = txtBox.text()

        if not ok or len(newText) == 0:
            return

        lwrText = newText.toLower()

        if lwrText in self._dicts[self._curLang]:
            return

        self._listView.addItem(lwrText)
        self._dicts[self._curLang].append(lwrText)

    def clearEntries(self, checked): #{{{2
        if self.delMode == 'Confirm':
            if QMessageBox.question(self, 'Delete confirmation', 'Please confirm that you want to delete all entries', QMessageBox.Ok|QMessageBox.Cancel) != QMessageBox.Ok:
                return

        self._listView.clear()
        self._dicts[self._curLang].clear()

    def deleteEntries(self, checked): #{{{2
        if self.delMode == 'Confirm':
            if QMessageBox.question(self, 'Delete confirmation', QString('Please confirm that you want to delete the %1 selected entries').arg(len(self._listView.selectedItems())), QMessageBox.Ok|QMessageBox.Cancel) != QMessageBox.Ok:
                return

        for entry in self._listView.selectedItems():
            self.deleteEntry(entry)

    def deleteEntry(self, entry): #{{{2
        row = self._listView.row(entry)
        self._listView.takeItem(row)
        val = entry.data(Qt.DisplayRole).toString()

        self._dicts[self._curLang].removeAt(self._dicts[self._curLang].indexOf(val))
        
    def loadDictionary(self, row): #{{{2
        self._listView.clear()

        if row >= 0:
            (dictId, check) = self._dictList.itemData(row).toInt()
            self._curLang = dictId
            self._listView.addItems(self._dicts[dictId])
            self._settings.setValue('displayDictionary', dictId)

    def revertChanges(self, checked): #{{{2
        self.loadData()

    def saveChanges(self, checked): #{{{2
        if self.maemo:
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)

        langs = []

        for i in self._dicts.keys():
            if len(self._dicts[i]) > 0:
                langs.append(i)

        for i in self._dicts.keys():
            if len(langs) == 0:
                langs.append(i)
            
            if i in langs:
                self._dict.setDict(i, self._dicts[i])
            else:
                self._dict.deleteDict(i)
            
        self._dict.write()

        call(['/usr/bin/killall', 'hildon-input-method'])

        if self.maemo:
            from PyQt4.QtMaemo5 import QMaemo5InformationBox
            QMaemo5InformationBox.information(self, 'Dictionary saved', QMaemo5InformationBox.DefaultTimeout)
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, False)

        self.loadData()

    def setupMenu(self): #{{{2
        menu = QMenuBar()

        act = menu.addAction('Settings')
        QObject.connect(act, SIGNAL('triggered()'), self.config)
        act = menu.addAction('About')
        QObject.connect(act, SIGNAL('triggered()'), self.about)
        self.layout().setMenuBar(menu)

    def setupScreen(self): #{{{2
        self._dictList = QComboBox()
        self._dictList.setInsertPolicy(QComboBox.NoInsert)
        QObject.connect(self._dictList, SIGNAL('currentIndexChanged(int)'), self.loadDictionary)

        self._listView = QListWidget()
        self._listView.setUniformItemSizes(True)
        self._listView.setSelectionMode(QAbstractItemView.MultiSelection)
        self._listView.setSortingEnabled(True)

        if self.delMode == 'OnClick':
            QObject.connect(self._listView, SIGNAL('itemClicked(QListWidgetItem *)'), self.deleteEntry)

        btnBox = QDialogButtonBox()
        btnBox.setOrientation(Qt.Vertical)
        btn = btnBox.addButton('Delete all', QDialogButtonBox.ActionRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.clearEntries)
        btn = btnBox.addButton('Delete selected', QDialogButtonBox.ActionRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.deleteEntries)
        if self.delMode == 'OnClick':
            btn.setEnabled(False)
        self._delBtn = btn
        btn = btnBox.addButton('Add entry', QDialogButtonBox.ActionRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.addEntry)
        btn = btnBox.addButton('Save changes', QDialogButtonBox.AcceptRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.saveChanges)
        btn = btnBox.addButton('Revert changes', QDialogButtonBox.ResetRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.revertChanges)

        lyt = QVBoxLayout()
        lyt.addWidget(self._dictList)
        lyt.addWidget(self._listView)

        self.layout().addLayout(lyt)
        self.layout().addWidget(btnBox)

        # Clear the progress indicator
        if self.maemo:
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, False)

    def loadData(self): #{{{2
        updateSpinner = False
        if self.maemo and not self.testAttribute(Qt.WA_Maemo5ShowProgressIndicator):
            updateSpinner = True
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)

        if self._dict == None:
            self._dict = ACEFile()

        (curDict, check) = self._settings.value('displayDictionary').toInt()
        dictLangs = self._dict.getLanguages()
        self._dictList.clear()
        self._dicts = {}

        for lang in dictLangs:
            self._dicts[lang] = self._dict.getDict(lang)
            self._dictList.addItem(self._getLangName(lang), QVariant(lang))

        if curDict > 0 and curDict in dictLangs:
            self._curLang = curDict
            self._dictList.setCurrentIndex(dictLangs.index(curDict))
        else:
            self._curLang = dictLangs[0]
            self._dictList.setCurrentIndex(0)

        if len(dictLangs) == 1:
            self._dictList.setEnabled(False)
        
        if updateSpinner:
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, False)
