#!/usr/bin/env python
# -*- coding: utf-8 -*-

from subprocess    import call

from PyQt4.QtCore  import Qt, QObject, QSettings, QString, QVariant, \
                          SIGNAL, SLOT
from PyQt4.QtGui   import QAbstractItemView, QComboBox, QDialog, \
                          QDialogButtonBox, QHBoxLayout, QLabel, QLineEdit, \
                          QListWidget, QMenuBar, QMessageBox, QPushButton, \
                          QVBoxLayout, QWidget
from ACE_file      import ACEFile;

version = '0.0.5'

langLookup = {
    3: 'English (United Kingdom)',
    4: 'Français (France)',
    5: 'Deutsch (Deutschland)',
    6: 'Español (España)',
    8: 'Português (Portugal)',
    9: 'Svenska (Sverige)',
    10: 'Suomi (Suomi)',
    11: 'Norsk (Norge)',
    12: 'Dansk (Danmark)',
    13: 'Nederlands (Nederland)',
    14: 'Ελληνικά (Ελλάδα)',
    22: 'Polski (Polska)',
    23: 'Čeština (Česká republika)',
    39: 'Русский (Россия)',
    46: 'English (United States of America)',
    47: 'Italiano (Italia)',
    51: 'Español (América Latina)',
    52: 'Français (Québec)',
}

class ACEAboutWindow(QDialog): #{{{1
    def __init__(self, parent): #{{{2
        QDialog.__init__(self, parent)

        self.setWindowTitle('About')

        aboutText  = QString('<h2>Auto-Complete Editor v%1</h2>').arg(version)

        aboutText += "<p>An editor for the user's auto-complete dictionary</p>"
        aboutText += QString('<p>Written by %1<br />').arg('Robin Hill <a href="mailto:maemo@robinhill.me.uk">&lt;maemo@robinhill.me.uk&gt;</a>')
        aboutText += QString('Please email or post to %1 for support</p>').arg('<a href="http://talk.maemo.org/">t.m.o</a>')

        txt = QLabel()
        txt.setOpenExternalLinks(True)
        txt.setTextFormat(Qt.RichText)
        txt.setText(aboutText)

        lyt = QHBoxLayout(self)
        lyt.addWidget(txt)

class ACEMainWindow(QWidget): #{{{1
    _maemo = False
    _dict  = None
    _dicts = {}

    def __init__(self): #{{{2
        QWidget.__init__(self, None)

        try:
            from PyQt4.QtMaemo5 import QMaemo5InformationBox
            self._maemo = True
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)
        except:
            pass

        # Set to the N900 screen size to aid testing
        self.setMinimumSize(800, 480)
        self.setWindowTitle('AutoComplete Editor')

        self._settings = QSettings('Robin Hill', 'Auto-Complete Editor')

        QHBoxLayout(self)

    def _getLangName(self, langId): #{{{2
        if langId in langLookup:
            return QString.fromUtf8(langLookup[langId])
        else:
            QMessageBox.information(self, 'Unknown dictionary', QString('An unknown dictionary ID has been found.\nPlease contact\n\tRobin Hill <maemo@robinhill.me.uk>\nwith:\n\t- the ID: %1\n\t- the dictionary language you have selected (from Settings/Text input)').arg(langId))
            return 'Unknown'

    def about(self): #{{{2
        aboutWin = ACEAboutWindow(self)
        aboutWin.exec_()

    def addEntry(self, checked): #{{{2
        dlg = QDialog(self)
        dlg.setWindowTitle('Enter new value')

        txtBox = QLineEdit()
        txtBox.setInputMethodHints(Qt.ImhNoAutoUppercase|Qt.ImhNoPredictiveText)

        btnBox = QDialogButtonBox(QDialogButtonBox.Ok|QDialogButtonBox.Cancel)
        QObject.connect(btnBox, SIGNAL('accepted()'), dlg, SLOT('accept()'))
        QObject.connect(btnBox, SIGNAL('rejected()'), dlg, SLOT('reject()'))

        lyt = QHBoxLayout(dlg)
        lyt.addWidget(txtBox)
        lyt.addWidget(btnBox)

        ok = dlg.exec_()
        newText = txtBox.text()

        if not ok or len(newText) == 0:
            return

        lwrText = newText.toLower()

        if lwrText in self._dicts[self._curLang]:
            return

        self._listView.addItem(lwrText)
        self._dicts[self._curLang].append(lwrText)

    def clearEntries(self, checked): #{{{2
        self._listView.clear()
        self._dicts[self._curLang].clear()

    def deleteEntries(self, checked): #{{{2
        for entry in self._listView.selectedItems():
            row = self._listView.row(entry)
            self._listView.takeItem(row)
            val = entry.data(Qt.DisplayRole).toString()

            self._dicts[self._curLang].removeAt(self._dicts[self._curLang].indexOf(val))

    def loadDictionary(self, row): #{{{2
        self._listView.clear()

        if row >= 0:
            (dictId, check) = self._dictList.itemData(row).toInt()
            self._curLang = dictId
            self._listView.addItems(self._dicts[dictId])
            self._settings.setValue('displayDictionary', dictId)

    def revertChanges(self, checked): #{{{2
        self.loadData()

    def saveChanges(self, checked): #{{{2
        if self._maemo:
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)

        langs = []

        for i in self._dicts.keys():
            if len(self._dicts[i]) > 0:
                langs.append(i)

        for i in self._dicts.keys():
            if len(langs) == 0:
                langs.append(i)
            
            if i in langs:
                self._dict.setDict(i, self._dicts[i])
            else:
                self._dict.deleteDict(i)
            
        self._dict.write()

        call(['/usr/bin/killall', 'hildon-input-method'])

        if self._maemo:
            from PyQt4.QtMaemo5 import QMaemo5InformationBox
            QMaemo5InformationBox.information(self, 'Dictionary saved', QMaemo5InformationBox.DefaultTimeout)
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, False)

        self.loadData()

    def setupMenu(self): #{{{2
        menu = QMenuBar()

        act = menu.addAction('About')
        QObject.connect(act, SIGNAL('triggered()'), self.about)
        self.layout().setMenuBar(menu)

    def setupScreen(self): #{{{2
        self._dictList = QComboBox()
        self._dictList.setInsertPolicy(QComboBox.NoInsert)
        QObject.connect(self._dictList, SIGNAL('currentIndexChanged(int)'), self.loadDictionary)

        self._listView = QListWidget()
        self._listView.setUniformItemSizes(True)
        self._listView.setSelectionMode(QAbstractItemView.MultiSelection)
        self._listView.setSortingEnabled(True)

        btnBox = QDialogButtonBox()
        btnBox.setOrientation(Qt.Vertical)
        btn = btnBox.addButton('Delete all', QDialogButtonBox.ActionRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.clearEntries)
        btn = btnBox.addButton('Delete selected', QDialogButtonBox.ActionRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.deleteEntries)
        btn = btnBox.addButton('Add entry', QDialogButtonBox.ActionRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.addEntry)
        btn = btnBox.addButton('Save changes', QDialogButtonBox.AcceptRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.saveChanges)
        btn = btnBox.addButton('Revert changes', QDialogButtonBox.ResetRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.revertChanges)

        lyt = QVBoxLayout()
        lyt.addWidget(self._dictList)
        lyt.addWidget(self._listView)

        self.layout().addLayout(lyt)
        self.layout().addWidget(btnBox)

        # Clear the progress indicator
        if self._maemo:
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, False)

    def loadData(self): #{{{2
        updateSpinner = False
        if self._maemo and not self.testAttribute(Qt.WA_Maemo5ShowProgressIndicator):
            updateSpinner = True
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)

        if self._dict == None:
            self._dict = ACEFile()

        (curDict, check) = self._settings.value('displayDictionary').toInt()
        dictLangs = self._dict.getLanguages()
        self._dictList.clear()
        self._dicts = {}

        for lang in dictLangs:
            self._dicts[lang] = self._dict.getDict(lang)
            self._dictList.addItem(self._getLangName(lang), QVariant(lang))

        if curDict > 0 and curDict in dictLangs:
            self._curLang = curDict
            self._dictList.setCurrentIndex(dictLangs.index(curDict))
        else:
            self._curLang = dictLangs[0]
            self._dictList.setCurrentIndex(0)

        if len(dictLangs) == 1:
            self._dictList.setEnabled(False)
        
        if updateSpinner:
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, False)
