#!/usr/bin/env python

from subprocess    import call

from PyQt4.QtCore  import Qt, QObject, SIGNAL, SLOT
from PyQt4.QtGui   import QAbstractItemView, QCompleter, QDialog, \
                          QDialogButtonBox, QHBoxLayout, QLabel, QLineEdit, \
                          QListWidget, QMenuBar, QMessageBox, QPushButton, \
                          QVBoxLayout, QWidget
from ACE_file      import ACEFile;

version = '0.0.3'

class ACEAboutWindow(QDialog): #{{{1
    def __init__(self, parent): #{{{2
        QDialog.__init__(self, parent)

        self.setWindowTitle('About')

        aboutText  = '<h2>AutoComplete Editor v' + version + '</h2>'

        aboutText += "<p>An editor for the user's auto-complete dictionary"
        aboutText += '<p>Written by Robin Hill <a href="mailto:maemo@robinhill.me.uk">&lt;maemo@robinhill.me.uk&gt;</a><br/>'
        aboutText += 'Please email or post to <a href="http://talk.maemo.org/">t.m.o</a> for support</p>'

        txt = QLabel()
        txt.setOpenExternalLinks(True)
        txt.setTextFormat(Qt.RichText)
        txt.setText(aboutText)

        lyt = QHBoxLayout(self)
        lyt.addWidget(txt)

class ACEMainWindow(QWidget): #{{{1
    _maemo = False
    _delList = []
    _addList = []

    def __init__(self): #{{{2
        QWidget.__init__(self, None)

        try:
            from PyQt4.QtMaemo5 import QMaemo5InformationBox
            self._maemo = True
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)
        except:
            pass

        # Set to the N900 screen size to aid testing
        self.setMinimumSize(800, 480)
        self.setWindowTitle('AutoComplete Editor')

        QHBoxLayout(self)

    def about(self): #{{{2
        aboutWin = ACEAboutWindow(self)
        aboutWin.exec_()

    def addEntry(self, checked): #{{{2
        dlg = QDialog(self)
        dlg.setWindowTitle('Enter new value')

        txtBox = QLineEdit()
        txtBox.setInputMethodHints(Qt.ImhNoAutoUppercase|Qt.ImhNoPredictiveText)

        btnBox = QDialogButtonBox(QDialogButtonBox.Ok|QDialogButtonBox.Cancel)
        QObject.connect(btnBox, SIGNAL('accepted()'), dlg, SLOT('accept()'))
        QObject.connect(btnBox, SIGNAL('rejected()'), dlg, SLOT('reject()'))

        lyt = QHBoxLayout(dlg)
        lyt.addWidget(txtBox)
        lyt.addWidget(btnBox)

        ok = dlg.exec_()
        newText = txtBox.text()

        if not ok or len(newText) == 0:
            return

        lwrText = newText.toLower()

        if lwrText in self._addList:
            return

        self._listView.addItem(lwrText)

        if lwrText in self._delList:
            self._delList.removeAt(self._delList.indexOf(lwrText))
        else:
            self._addList.append(lwrText)

    def deleteEntries(self, checked): #{{{2
        for entry in self._listView.selectedItems():
            row = self._listView.row(entry)
            self._listView.takeItem(row)
            val = entry.data(Qt.DisplayRole).toString()

            if val in self._addList:
                self._addList.removeAt(self._addList.indexOf(val))
            else:
                self._delList.append(val)

    def revertChanges(self, checked): #{{{2
        self._delList = []
        self._listView.clear()
        self._listView.addItems(self._dict.dictEntries)

    def saveChanges(self, checked): #{{{2
        if self._maemo:
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)

        for entry in self._delList:
            pos = self._dict.dictEntries.indexOf(entry)
            self._dict.dictEntries.removeAt(pos)

        for entry in self._addList:
            self._dict.dictEntries.append(entry)

        self._dict.write()

        call(['/usr/bin/killall', 'hildon-input-method'])

        if self._maemo:
            from PyQt4.QtMaemo5 import QMaemo5InformationBox
            QMaemo5InformationBox.information(self, 'Dictionary saved', QMaemo5InformationBox.DefaultTimeout)
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, False)

    def setupMenu(self): #{{{2
        menu = QMenuBar()

        act = menu.addAction('About')
        QObject.connect(act, SIGNAL('triggered()'), self.about)
        self.layout().setMenuBar(menu)

    def setupScreen(self): #{{{2
        self._listView = QListWidget()
        self._listView.setUniformItemSizes(True)
        self._listView.setSelectionMode(QAbstractItemView.MultiSelection)
        self._listView.setSortingEnabled(True)

        btnBox = QDialogButtonBox()
        btnBox.setOrientation(Qt.Vertical)
        btn = btnBox.addButton('Delete selected', QDialogButtonBox.ActionRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.deleteEntries)
        btn = btnBox.addButton('Add entry', QDialogButtonBox.ActionRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.addEntry)
        btn = btnBox.addButton('Save changes', QDialogButtonBox.AcceptRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.saveChanges)
        btn = btnBox.addButton('Revert changes', QDialogButtonBox.ResetRole)
        QObject.connect(btn, SIGNAL('clicked(bool)'), self.revertChanges)

        self.layout().addWidget(self._listView)
        self.layout().addWidget(btnBox)

        # Clear the progress indicator
        if self._maemo:
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, False)

    def loadData(self): #{{{2
        updateSpinner = False
        if self._maemo and not self.testAttribute(Qt.WA_Maemo5ShowProgressIndicator):
            updateSpinner = True
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)

        self._dict = ACEFile()

        if not self._dict.validate():
            res = QMessageBox.question(self, 'Dictionary file problem', 'The custom auto-complete dictionary does not match a current known format.  Please contact Robin Hill <maemo@robinhill.me.uk> to help get this format supported.  Continuing with editing may break your auto-complete functionality.  Do you wish to continue?', QMessageBox.Yes|QMessageBox.No, QMessageBox.No)

            if res != QMessageBox.Yes:
                exit()

        self._listView.addItems(self._dict.dictEntries)

        if updateSpinner:
            self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, False)

