#!/usr/bin/env python

import errno
import exceptions
import sys
import struct

from PyQt4.QtCore import QCoreApplication, QDir, QFile, QIODevice, QStringList

dictionaryFile = QDir.homePath() + '/.osso/dictionaries/.personal.dictionary'
posPaddingOffset = 5
posEntryCount    = 12
posEntryStart    = 16

class ACEFileException(Exception): #{{{1
    def __init__(self, msgsource, errmsg):
        self.msgsource = msgsource
        self.errmsg = errmsg

    def __str__(self):
        return repr("%s: %s" % (self.msgsource, self.errmsg))

class ACEFileReadException(ACEFileException): #{{{1
    def __init__(self, errmsg):
        ACEFileException.__init__(self, 'Error reading dictionary', errmsg)

class ACEFileWriteException(ACEFileException): #{{{1
    def __init__(self, errmsg):
        ACEFileException.__init__(self, 'Error writing dictionary', errmsg)

class ACEFile(): #{{{1
    dictEntries = QStringList()
    def __init__(self): #{{{2
        self.read()
        return
    
    def _bytesToInt(self, bytes): #{{{2
        return sum(ord(c) << (i * 8) for i, c in enumerate(bytes[::-1]))

    def _calculateDataSize(self): #{{{2
        size = 0
        for dictEntry in self.dictEntries:
            size += len(dictEntry) + 1

        return size

    def _intToBytes(self, val, count): #{{{2
        hex = "%%0%dx" % (2 * count) % val
        bitstream = ''

        for i in xrange(0, count * 2, 2):
            bitstream += chr(int(hex[i:i+2], 16))

        return bitstream

    def _readEntry(self, file): #{{{2
        entryLength = self._bytesToInt(file.read(1))
        entryString = file.read(entryLength)

        return entryString

    def _readMetadata(self, file): #{{{2
        if not file.seek(posPaddingOffset):
            raise IOError(errno.EIO, "Error seeking in file", dictionaryFile)

        paddingOffset = self._bytesToInt(file.read(3))

        if not file.seek(posEntryCount):
            raise IOError(errno.EIO, "Error seeking in file", dictionaryFile)

        entryCount = self._bytesToInt(file.read(2))

        if not file.seek(16):
            raise IOError(errno.EIO, "Error seeking in file", dictionaryFile)

        return (paddingOffset, entryCount)

    def _validateMetadata(self, file): #{{{2
        if not file.seek(0):
            raise IOError(errno.EIO, "Error seeking in file", dictionaryFile)

        bfr = file.read(16)

        if bfr[0:4] != '\x80\x00\x01\x01':
            print "Fails 1"
            return False

        if bfr[8:12] != '\x00\x03\x00\x10':
            print "Fails 2"
            return False

        if bfr[14:16] != '\x00\x00':
            print "Fails 3"
            return False

        return True

    def _writeEntry(self, file, entryString): #{{{2
        file.write(self._intToBytes(len(entryString), 1))
        file.write(entryString)

    def _writeMetadata(self, file): #{{{2
        dataSize = self._calculateDataSize()
        fileSizeK = (1023 + 16 + dataSize) / 1024
        fileSize = fileSizeK * 1024

        file.write('\x80\x00\x01\x01')
        file.write(self._intToBytes(fileSizeK * 4, 1))
        file.write(self._intToBytes(16 + dataSize, 3))
        file.write('\x00\x03\x00\x10')
        file.write(self._intToBytes(len(self.dictEntries), 2))
        file.write('\x00\x00')

        return fileSize

    def _writePadding(self, file, fileSize): #{{{2
        paddingLength = fileSize - file.pos()
        file.write('\x00' * paddingLength)

    def read(self): #{{{2
        file = QFile(dictionaryFile)

        if not file.exists():
            raise IOError(errno.ENOENT, "File not found", dictionaryFile)

        if not file.open(QIODevice.ReadOnly):
            raise IOError(errno.EIO, "Cannot open file", dictionaryFile)

        (paddingOffset, entryCount) = self._readMetadata(file)

        if not file.seek(posEntryStart):
            raise IOError(errno.EIO, "Error seeking in file", dictionaryFile)

        while file.pos() < paddingOffset:
            entryString = self._readEntry(file)
            self.dictEntries.append(entryString)

        if len(self.dictEntries) != entryCount:
            raise ACEFileReadException('Entry count mismatch')

        file.close()

    def validate(self): #{{{2
        file = QFile(dictionaryFile)

        if not file.open(QIODevice.ReadOnly):
            raise IOError(errno.EIO, "Cannot open file", dictionaryFile)

        retval = self._validateMetadata(file)

        file.close()

        return retval

    def write(self): #{{{2
        file = QFile("%s.tmp" % dictionaryFile)

        if not file.open(QIODevice.WriteOnly | QIODevice.Truncate):
            raise IOError(errno.EIO, "Cannot open file", "%s.tmp" % dictionaryFile)

        fileSize = self._writeMetadata(file)

        for entryString in self.dictEntries:
            self._writeEntry(file, entryString.toLocal8Bit())

        self._writePadding(file, fileSize)

        file.close()

        if QFile.exists("%s.bak" % dictionaryFile):
            if not QFile.remove("%s.bak" % dictionaryFile):
                raise IOError(errno.EIO, "Error removing old backup file", "%s.bak" % dictionaryFile)

        if not QFile.rename(dictionaryFile, "%s.bak" % dictionaryFile):
            raise IOError(errno.EIO, "Error writing backup file", "%s.bak" % dictionaryFile)

        if not QFile.rename("%s.tmp" % dictionaryFile, dictionaryFile):
            raise IOError(errno.EIO, "Error replacing dictionary", dictionaryFile)

if __name__ == '__main__': #{{{1
    app = QCoreApplication(sys.argv)
    dict = ACEFile()
    dict.write()
